//
//  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_COLLECTION_OF_H
#define _SYSADM_COLLECTION_OF_H

#include <sysadm/sysadm.h>
#include <assert.h>

BEGIN_NAMESPACE(sysadm);

template <class Type> class BaseIterator;
template <class Type> class IteratorOver;
template <class Type> class ConstIteratorOver;
template <class derivedType, class baseType> class CastIteratorOver;
template <class derivedType, class baseType> class ConstCastIteratorOver;
class IteratorOverImpl;

class CollectionOfImpl;


// All collections are created using the form 'CollOf<Type>' and contain
// collections of _pointers_ to 'Type'.  These pointer are not owned by
// the collection, though there is a utility function at the bottom of this
// header that can be used to delete the contents of a collection.
//
// Collections of 'Type' can be iterated over by using IteratorOver<Type>.
// Though, in the case of ArrayOf, care must be taken.  ArrayOf,
// because it can be used just like a 'C' vector, may have NULL elements,
// which may cause early termination of an iterator.  Arrays can be
// iterated over using a traditional 'for()' loop and index safely.
//
// Collections should not be added to or removed from directly while being
// iterated over.  To do so would confuse the iterator.  Use the provided
// iterator insert/remove operations instead.

// Class CollectionOf is a generic, non-ordered collection or 'Bag'.
template <class Type> class CollectionOf {
  public:
    inline CollectionOf();
    inline CollectionOf( const CollectionOf& copy );
    
    // This destructor is intentionally non-virtual.  All state should
    // be in the impl class, which is destructed here.  Subclasses
    // of this template should have empty destructors!!!
    inline ~CollectionOf();  

    inline CollectionOf& operator=( const CollectionOf& copy );

    inline int getSize() const;

    inline void  add( Type* element );
    inline Type* remove( const Type& element );  // requires Type::operator==()
    inline Type* removePointer( const Type* element ); // pointer eq test
    inline void  removeAll();

  protected:
    inline CollectionOf( CollectionOfImpl* adoptedImpl );

    inline CollectionOfImpl* getImpl();
    inline const CollectionOfImpl* getImpl() const;

  private:
    friend class BaseIterator<Type>;

    CollectionOfImpl*  _impl;
};

template <class Type> class BaseIterator {
  public:
    inline BaseIterator(const CollectionOf<Type>* collectionAlias);
    inline ~BaseIterator();  // See notes of CollectionOf destructor!

  protected:
    inline IteratorOverImpl *getImpl() const;

  private:
    IteratorOverImpl*  _impl;
};

template <class Type> class ConstIteratorOver : 
    public BaseIterator<Type> {
  public:
    inline ConstIteratorOver( const CollectionOf<Type>* collectionAlias );
    inline ~ConstIteratorOver();  // See notes of CollectionOf destructor!

    // one option for an iteration syntax (simple):
    // NULL is returned when end (or beginning) is reached
    inline Type* getFirst();
    inline Type* getLast();
    inline Type* getNext();
    inline Type* getPrev();

    // alternate iteration syntax:
    inline void  reset();
    inline void  resetToEnd();
    // NULL is returned when end (or beginning) is reached
    inline Type* operator()( int skipBy );
    inline Type* operator()();
};

template <class Type>
class IteratorOver : public ConstIteratorOver<Type> {
  public:
    inline IteratorOver( CollectionOf<Type>* collectionAlias );

    // other features

    // Use this to remove the current value from the collection being iterated
    // without disturbing the iteration
    inline Type* remove();

    // These are used to add to the collection without disturbing the iteration
    // and will insert either before or after the current value
    inline void  insertAfter( Type* element );
    inline void  insertBefore( Type* element );
};




template <class derivedType, class baseType> 
class ConstCastIteratorOver : public BaseIterator<baseType> {
  public:
    ConstCastIteratorOver( const CollectionOf<baseType>* collectionAlias );
    ~ConstCastIteratorOver();  // See notes of CollectionOf destructor!

    // one option for an iteration syntax (simple):
    // NULL is returned when end (or beginning) is reached
    derivedType* getFirst();
    derivedType* getLast();
    derivedType* getNext();
    derivedType* getPrev();

    // alternate iteration syntax:
    void  reset();
    void  resetToEnd();

// This method is not provided, because it would cause excessive bloat
//    derivedType* operator()( int skipBy = 1 );
};

template <class derivedType, class baseType> 
class CastIteratorOver : 
public ConstCastIteratorOver<derivedType, baseType> {
  public:
    CastIteratorOver( CollectionOf<baseType>* collectionAlias );

    // other features
    derivedType* remove();
    void  insertAfter( baseType* element );
    void  insertBefore( baseType* element );
};


// Convenience functions to remove and delete each object in a
// collection.  They're not member functions of CollectionOf because
// then Collection of const objects would not work right.

//This version uses the delete operator on each of the elements.
template <class Type> inline
void RemoveAndDestroyContentsOf( CollectionOf<Type> *collectionAlias );

//This version is identical to the RemoveAndDestrtoyContentsOf,
//only the name is different.
template <class Type> inline
void RemoveAndDeleteContentsOf(CollectionOf<Type>
                                  * collectionAlias);

//This version uses the delete[] operator on each of the elements.
template <class Type> inline
void RemoveAndDeleteArrayContentsOf(CollectionOf<Type>
                                        *collectionAlias);

//This version uses the free() function on each of the elements.
template <class Type> inline
void RemoveAndFreeContentsOf( CollectionOf<Type>
                                  * collectionAlias);

END_NAMESPACE(sysadm);

#include <sysadm/CollectionsI.h>
#include <sysadm/CollectionOfI.h>
#endif  //  _SYSADM_COLLECTION_OF_H
