//
//  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#include <stdlib.h>

#include <sysadm/CollectionOf.h>
#include <sysadm/OrderedCollectionOf.h>

#include "CollectionsP.h"

#include <string.h>

USING_NAMESPACE(sysadm);

// DLink implimentation.
void DLink::insertBefore( DLink* newLink )
{
    assert( newLink );

    if( _prev )
	_prev->_next = newLink;
    newLink->_prev = _prev;
    newLink->_next = this;
    _prev = newLink;
}

void DLink::insertAfter( DLink* newLink )
{
    assert( newLink );

    if( _next )
	_next->_prev = newLink;
    newLink->_prev = this;
    newLink->_next = _next;
    _next = newLink;
}

void DLink::remove()
{
    if( _prev )
	_prev->_next = _next;
    if( _next )
	_next->_prev = _prev;
}

#ifdef USE_RECYCLED_DLINKS
DLink* DLink::Recycler::_list(NULL);
long DLink::Recycler::_liveCount = 0;
long DLink::Recycler::_deadCount = 0;

DLink::Recycler::Recycler()
{
    if( !_list )
	_list = ::new DLink;
}

DLink::Recycler::~Recycler()
{
    DLink* link = _list->_next;
    while( link ) {
	link->remove();
	::delete link;
	link = _list->_next;
    }

    ::delete _list;
    _list = NULL;
}

DLink* DLink::Recycler::fetch()
{
    _liveCount++;

    if( _deadCount > 0 ) {
	_deadCount--;
	DLink* ptr = _list->_next;
	ptr->remove();
	ptr->_prev = ptr->_next = NULL;
	ptr->_data = NULL;

	return ptr;
    }

    return ::new DLink;
}

void DLink::Recycler::put( DLink* item )
{
    _deadCount++;
    _liveCount--;

    _list->insertAfter( item );
}

DLink::Recycler MasterList;


void* DLink::operator new( size_t /*size*/ )
{
    return MasterList.fetch();
}

void DLink::operator delete( void* item )
{
    MasterList.put( (DLink*)item );
}

#endif



CollectionOfImpl::~CollectionOfImpl()
{
}


// RealCollectionOfImpl

RealCollectionOfImpl::RealCollectionOfImpl()
: CollectionOfImpl(), _anchor(new DLink()), _count(0)
{
    _anchor->_next = _anchor->_prev = _anchor;
}

RealCollectionOfImpl::RealCollectionOfImpl( const RealCollectionOfImpl& copy )
: CollectionOfImpl(copy), _anchor( new DLink() ), _count(0)
{
    _anchor->_next = _anchor->_prev = _anchor;

    copyFrom( copy );
}

CollectionOfImpl* RealCollectionOfImpl::createClone() const
{
    return new RealCollectionOfImpl( *this );
}

RealCollectionOfImpl::~RealCollectionOfImpl()
{
    removeAll();

    delete _anchor;
}

RealCollectionOfImpl& 
RealCollectionOfImpl::operator=( const RealCollectionOfImpl& copy )
{
    if( this != &copy ) {
	removeAll();

	copyFrom( copy );
    }

    return *this;
}

void RealCollectionOfImpl::copyFrom( const RealCollectionOfImpl& copy )
{
    if( copy._anchor == NULL )
	return;

    DLink* link = copy._anchor->_next;

    while( link && link != copy._anchor ) {
	// This is a virtual method, so a derived class may not always depend
	// on the base copy working correctly. (the derived add will _not_
	// be called from the copy constructor)
	add( link->_data );

	link = link->_next;
    }
}

IteratorOverImpl* RealCollectionOfImpl::createIterator()
{
    return new IteratorOverDLinkImpl(this);
}

unsigned int RealCollectionOfImpl::getSize() const
{
    return _count;
}

void  RealCollectionOfImpl::add( void* element )
{
    insertBefore( _anchor, element );
}

void RealCollectionOfImpl::remove( DLink* adoptedElement )
{
    adoptedElement->remove();
    delete adoptedElement;

    _count--;
}

void  RealCollectionOfImpl::removeAll()
{
    if( _anchor->_next == _anchor )
	return;  // empty list, do nothing.

    DLink* link = _anchor->_next;

    while( link && link != _anchor ) {
	DLink* tmp = link->_next;
	delete link;
	link = tmp;
    }

    _count = 0;
    _anchor->_next = _anchor->_prev = _anchor;
}

void* RealCollectionOfImpl::getData( DLink* link )
{
    return (link) ? link->_data : NULL;
}

DLink* RealCollectionOfImpl::getNext( DLink* link )
{
    return (link==NULL || link->_next == _anchor) ? NULL : link->_next;
}

DLink* RealCollectionOfImpl::getPrev( DLink* link )
{
    return (link==NULL || link->_prev == _anchor) ? NULL : link->_prev;
}

DLink* RealCollectionOfImpl::getFirst()
{
    return getNext( _anchor );
}

DLink* RealCollectionOfImpl::getLast()
{
    return getPrev( _anchor );
}

void RealCollectionOfImpl::insertBefore( DLink* link, void* element )
{
    assert( link != NULL );
    assert( element != NULL );

    DLink* newLink = new DLink( element );

    // inserts newLink in front of link
    link->insertBefore( newLink );

    _count++;
}

void RealCollectionOfImpl::insertAfter( DLink* link, void* element )
{
    assert( link != NULL );
    assert( element != NULL );

    DLink* newLink = new DLink( element );

    // inserts newLink after link
    link->insertAfter( newLink );

    _count++;
}


// Iterator code

IteratorOverImpl::~IteratorOverImpl()
{
}

// Iterator over DLink code

IteratorOverDLinkImpl::~IteratorOverDLinkImpl()
{
}

void* IteratorOverDLinkImpl::getFirst()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    _point = impl->getFirst();

    _justRemoved = false;

    return impl->getData(_point);
}

void* IteratorOverDLinkImpl::getLast()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    _point = impl->getLast();

    _justRemoved = false;

    return impl->getData(_point);
}

void* IteratorOverDLinkImpl::getNext()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    _point = impl->getNext(_point);

    _justRemoved = false;

    return impl->getData(_point);
}

void* IteratorOverDLinkImpl::getPrev()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    if( !_justRemoved )
	_point = impl->getPrev(_point);

    _justRemoved = false;

    return impl->getData(_point);
}

void  IteratorOverDLinkImpl::reset()
{
    _point = NULL;
    _justRemoved = false;
}

void  IteratorOverDLinkImpl::resetToEnd()
{
    _point = NULL;
    _justRemoved = false;
}

void* IteratorOverDLinkImpl::operator()()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    if( _point == NULL ) 
	_point = impl->getFirst();
    else
	_point = impl->getNext(_point);

    _justRemoved = false;

    return impl->getData(_point);
}

void* IteratorOverDLinkImpl::operator()( int skipBy )
{
    assert( skipBy != 0 );

    int cnt = (skipBy > 0) ? skipBy : -skipBy;
    bool upCount = (skipBy > 0);

    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    if( _point == NULL ) {
	if( upCount )
	    _point = impl->getFirst();
	else
	    _point = impl->getLast();

	cnt--;
    }

    while( cnt-- > 0 && _point ) {
	if( upCount )
	    _point = impl->getNext(_point);
	else if( !_justRemoved )
	    _point = impl->getPrev(_point);
    }

    _justRemoved = false;

    return impl->getData(_point);
}

void* IteratorOverDLinkImpl::remove()
{
    RealCollectionOfImpl* impl = (RealCollectionOfImpl*)_collection;

    void* data = impl->getData(_point);

    DLink * prev = impl->getPrev(_point);

    if( _point )
	impl->remove(_point);

    _point = prev;

    // This is done so if the user tries to iterated again, and that iteration
    // is 'backward', then we know to not iterate (because we are at the
    // previous point)
    _justRemoved = true;

    return data;
}

void  IteratorOverDLinkImpl::insertAfter( void* element )
{
    assert( element != NULL );
    assert( _point != NULL );

    _justRemoved = false;
    if( _point )
	((RealCollectionOfImpl*)_collection)->insertAfter(_point, element);
}

void  IteratorOverDLinkImpl::insertBefore( void* element )
{
    assert( element != NULL );
    assert( _point != NULL );

    _justRemoved = false;
    if( _point )
	((RealCollectionOfImpl*)_collection)->insertBefore(_point, element);
}


// Ordered Collection

OrderedCollectionOfImpl::OrderedCollectionOfImpl()
: RealCollectionOfImpl()
{
}

OrderedCollectionOfImpl::OrderedCollectionOfImpl( const OrderedCollectionOfImpl& copy )
: RealCollectionOfImpl( copy )
{
}

OrderedCollectionOfImpl::~OrderedCollectionOfImpl()
{
}

OrderedCollectionOfImpl& 
OrderedCollectionOfImpl::operator=( const OrderedCollectionOfImpl& copy )
{
    if( this != &copy ) {
	RealCollectionOfImpl::operator=(copy);
    }

    return *this;
}

CollectionOfImpl* OrderedCollectionOfImpl::createClone() const
{
    return new OrderedCollectionOfImpl( *this );
}

void  OrderedCollectionOfImpl::append( void* element )
{
    insertBefore( _anchor, element );
}

void  OrderedCollectionOfImpl::prepend( void* element )
{
    insertAfter( _anchor, element );
}

