//
// Command.c++
//
//	Class for running commands.
//
//
//  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.11 $"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <unistd.h>
#include <fcntl.h>
#include <assert.h>
#include <stdlib.h>
#include <errno.h>

#include <sysadm/Command.h>
#include <sysadm/OrderedCollectionOf.h>

BEGIN_NAMESPACE(sysadm);

//
// Opaque implementation.
//
class CommandImpl {
  public:
    CommandImpl(const String& command);
    ~CommandImpl();
    String _command;
    String _dir;
    OrderedCollectionOf<String> _argList;
    const char* const* _argv;
    int _input;
    int _output;
    int _error;
    FILE* _inputFP;
    FILE* _outputFP;
    FILE* _errorFP;
    uid_t _uid;
    pid_t _pid;
    int _status;
    bool _beenRun;
    bool _inputSpecified;
    bool _outputSpecified;
    bool _errorSpecified;
    bool _langC;
    bool _usePty;

    void cleanup(int input, int output, int error);
};

//
// CommandImpl constructor.
//
CommandImpl::CommandImpl(const String& command)
: _command(command), _dir(String::NUL), _argv(NULL),
  _input(-1), _inputSpecified(false), _output(-1),
  _outputSpecified(false), _error(-1), _errorSpecified(false),
  _inputFP(NULL), _outputFP(NULL), _errorFP(NULL), _uid(getuid()),
  _pid(-1), _status(0), _beenRun(false), _langC(false), _usePty(false)
{
}

//
// CommandImpl destructor.
//
CommandImpl::~CommandImpl()
{
    RemoveAndDestroyContentsOf(&_argList);
    if (_inputFP != NULL) {
	fclose(_inputFP);
    }
    if (_outputFP != NULL) {
	fclose(_outputFP);
    }
    if (_errorFP != NULL) {
	fclose(_errorFP);
    }
    if (!_outputSpecified) {
	close(_output);
    }
    if (!_errorSpecified) {
	close(_error);
    }
    // _imput gets taken care of in Command::wait().
}

//
//  void CommandImpl::cleanup(int input, int output, int error)
//
//  Description:
//      Called from within Command::run() to close file descriptors
//      which are no longer needed.
//
//  Parameters:
//      input   file descriptor used in run() for subprocess stdin.
//      output  file descriptor used in run() for subprocess stdout.
//      error   file descriptor used in run() for subprocess stderr.
//
void CommandImpl::cleanup(int input, int output, int error)
{
    if (input != _input) {
	close(input);
    }
    if (output != _output) {
	close(output);
    }
    if (error != _error) {
	close(error);
    }
}

//
// Command constructor.
//
Command::Command(const String& command)
: _impl(new CommandImpl(command))
{
}

//
// Command destructor.
//
Command::~Command()
{
    wait();
    delete _impl;
}

//
//  void Command::setUid(uid_t uid)
//
//  Description:
//      Set the uid that the command should be run as.  After fork but
//      before exec, setuid will be called.
//
//  Parameters:
//      uid  uid to run as.
//
void Command::setUid(uid_t uid)
{
    _impl->_uid = uid;
}

//
//  void Command::setInput(int fd)
//
//  Description:
//      Set the file descriptor which should be command's standard
//      input.
//
//  Parameters:
//      fd  file descriptor to use for standard input.
//
void Command::setInput(int fd)
{
    _impl->_input = fd;
    _impl->_inputSpecified = true;
}

//
//  void Command::setOutput(int fd)
//
//  Description:
//      Set the file descriptor which should be command's standard
//      output.
//
//  Parameters:
//      fd  file descriptor to use for standard output.
//
void Command::setOutput(int fd)
{
    _impl->_output = fd;
    _impl->_outputSpecified = true;
}

//
//  void Command::setError(int fd)
//
//  Description:
//      Set the file descriptor which should be command's standard
//      error.
//
//  Parameters:
//      fd  file descriptor to use for standard error.
//
void Command::setError(int fd)
{
    // _usePty and _errorSpecified are mutually exclusive.
    assert(!_impl->_usePty);
    _impl->_error = fd;
    _impl->_errorSpecified = true;
}

//
//  void Command::addArg(const String& arg)
//
//  Description:
//      Add an argument to be passed to exec.
//
//  Parameters:
//      arg  Argument to be added.
//
void Command::addArg(const String& arg)
{
    // Don't allow addArg() to be mixed with setArgs().
    assert(_impl->_argv == NULL);
    _impl->_argList.append(new String(arg));
}

//
//  void Command::setArgs(const char* const* args)
//
//  Description:
//      Use "args" instead of _impl->_argList when calling execv().
//
//  Parameters:
//      args  Args to pass to execv().
//
void Command::setArgs(const char* const* args)
{
    // Don't allow addArg() to be mixed with setArgs().
    assert(_impl->_argList.getSize() == 0);
    _impl->_argv = args;
}

//
//  void Command::setLangC()
//
//  Description:
//      Command will be run with LANG=C instead of current locale.
//
void Command::setLangC()
{
    _impl->_langC = true;
}

//
//  void Command::usePty()
//
//  Description:
//      Use a pseudo-terminal for command's stderr.
//
void Command::usePty()
{
    // _usePty and _errorSpecified are mutually exclusive.
    assert(!_impl->_errorSpecified);
//    _impl->_usePty = true;
    assert(/* gahh, usePty not supported on linux yet */ false);
}

//
//  void Command::setDir(const String& dir)
//
//  Description:
//      Set the directory that the command should be run in.
//
//  Parameters:
//      dir  directory the command should be run in.
//
void Command::setDir(const String& dir)
{
    _impl->_dir = dir;
}

//
// Convenience function for setting up parent and child file
// descriptors before forking.  If caller specified an fd to be used,
// it's passed in as parentSide and we just copy it to childSide.
// Otherwise, we create a pipe and set parentSide and childSide
// appropriately, depending on whether this is the child's input.
//
static int SetupFD(int& parentSide, int& childSide, bool input)
{
    int pipes[2];

    if (parentSide == -1) {
	if (pipe(pipes) == -1) {
	    return -1;
	}

	if (input) {
	    parentSide = pipes[1];
	    childSide = pipes[0];
	} else {
	    parentSide = pipes[0];
	    childSide = pipes[1];
	}
    } else {
	childSide = parentSide;
    }
    return 0;
}

//
//  const char* SetupPty(int& master)
//
//  Description:
//      Set up a pseudo-terminal.
//
//  Parameters:
//      master  Gets the file descriptor of the master.
//
//  Returns:
//	Device to use as the slave, NULL if an error occurs.
//
const char* SetupPty(int& master)
{
//    char* slaveDevice = _getpty(&master, O_RDWR, 0600, 0);
//    return slaveDevice;
    assert(/* gahh, SetupPty not supported on linux yet */ false);
    return NULL;
}

//
//  int Command::run()
//
//  Description:
//      fork and exec our command.
//
//  Returns:
//	0 if successful, -1 if error.
//
int Command::run()
{
    // Can't run same command twice
    assert(!_impl->_beenRun);
    _impl->_beenRun = true;

    int input = -1, output = -1, error = -1;

    if (SetupFD(_impl->_input, input, true) == -1
	|| SetupFD(_impl->_output, output, false) == -1) {
	_impl->cleanup(input, output, error);
	return -1;
    }

    const char* slaveDevice = NULL;
    if (_impl->_usePty && !_impl->_errorSpecified) {
	slaveDevice = SetupPty(_impl->_error);
	if (slaveDevice == NULL) {
	    _impl->cleanup(input, output, error);
	    return -1;
	}
	error = -1;
    } else {
	if (SetupFD(_impl->_error, error, false) == -1) {
	    _impl->cleanup(input, output, error);
	    return -1;
	}
    }

    _impl->_pid = fork();
    if (_impl->_pid == -1) {
	_impl->cleanup(input, output, error);
	return -1;
    } else if (_impl->_pid == 0) {
	// Note that the error handling code below which calls exit()
	// is occurring in the child process, and will result in a
	// read from Command::getError() returning something like
	// this:
	//
	//    blah: No such file or directory.
	//
	if (_impl->_langC) {
	    putenv("LANG=C");
	}
	if (_impl->_dir != String::NUL) {
	    if (chdir(_impl->_dir) == -1) {
		perror(_impl->_dir);
	    }
	}
	dup2(input, 0);
	dup2(output, 1);
	if (slaveDevice == NULL) {
	    dup2(error, 2);
	} else {
	    close(2);
	}

	if (setuid(_impl->_uid) == -1) {
	    perror("setuid");
	    exit(1);
	}

//	for (int fd = getdtablehi() - 1; fd > 2; fd--) {
	for (int fd = getdtablesize() - 1; fd > 2; fd--) {
	    close(fd);
	}

	if (slaveDevice != NULL) {
	    // Detach from parent's terminal.
	    setsid();
	    // Open new terminal.  Will get fd 2 (stderr).
	    open(slaveDevice, O_RDWR);
	}

	char** args;
	if (_impl->_argv != NULL) {
	    args = (char **)_impl->_argv;
	} else {
	    args = new char*[_impl->_argList.getSize() + 1];
	    int i = 0;
	    IteratorOver<String> iter(&_impl->_argList);
	    String* arg;
	    while ((arg = iter()) != NULL) {
		args[i++] = arg->writableCharPtr();
	    }
	    args[i] = NULL;
	}

	execv(_impl->_command, args);

	perror(_impl->_command);
	exit(1);
    }

    _impl->cleanup(input, output, error);
    return 0;
}

//
//  int Command::getInput()
//
//  Returns:
//	input file descriptor.
//
int Command::getInput()
{
    return _impl->_input;
}

//
//  FILE* Command::getInputFP()
//
//  Returns:
//	FILE* for input file descriptor.
//
FILE* Command::getInputFP()
{
    // If input is specified, caller owns the file descriptor so
    // caller should fdopen it.
    assert(_impl->_inputSpecified == false);
    if (_impl->_inputFP == NULL) {
	_impl->_inputFP = fdopen(_impl->_input, "w");
    }
    return _impl->_inputFP;
}

//
//  int Command::getOutput()
//
//  Returns:
//	Output file descriptor.
//
int Command::getOutput()
{
    return _impl->_output;
}

//
//  FILE* Command::getOutputFP()
//
//  Returns:
//	FILE* for output file descriptor.
//
FILE* Command::getOutputFP()
{
    // If output is specified, caller owns the file descriptor so
    // caller should fdopen it.
    assert(_impl->_outputSpecified == false);
    if (_impl->_outputFP == NULL) {
	_impl->_outputFP = fdopen(_impl->_output, "r");
    }
    return _impl->_outputFP;
}

//
//  int Command::getError()
//
//  Returns:
//	error file descriptor.
//
int Command::getError()
{
    return _impl->_error;
}

//
//  void Command::closeInput()
//
//  Description:
//      Close the standard input of the command.
//
void Command::closeInput()
{
    // If input was specified, specifier should deal with closing it.
    assert(!_impl->_inputSpecified);
    close(_impl->_input);
    _impl->_input = -1;
}

//
//  FILE* Command::getErrorFP()
//
//  Returns:
//	FILE* for error file descriptor.
//
FILE* Command::getErrorFP()
{
    // If error is specified, caller owns the file descriptor so
    // caller should fdopen it.
    assert(_impl->_errorSpecified == false);
    if (_impl->_errorFP == NULL) {
	_impl->_errorFP = fdopen(_impl->_error, "r");
    }
    return _impl->_errorFP;
}

//
//  void Command::kill(int sig)
//
//  Description:
//      Send a signal to our command.
//
//  Parameters:
//      sig  signal to send.
//
void Command::kill(int sig)
{
    ::kill(_impl->_pid, sig);
}

//
//  void Command::wait()
//
//  Description:
//      Wait for our child process to exit.
//
void Command::wait()
{
    if (!_impl->_inputSpecified) {
	if (_impl->_inputFP != NULL) {
	    fclose(_impl->_inputFP);
	    _impl->_inputFP = NULL;
	}
	close(_impl->_input);
    }
    while (waitpid(_impl->_pid, &_impl->_status, 0) == -1
	   && errno == EINTR) {
	continue;
    }
}

//
//  int Command::getExitStatus()
//
//  Returns:
//	exit status of our command.
//
int Command::getExitStatus()
{
    return WIFEXITED(_impl->_status) ? WEXITSTATUS(_impl->_status) : -1;
}

//
//  void Command::dumpError(int fd)
//
//  Description:
//      Dump whatever is left on the error file descriptor to "fd".
//
//  Parameters:
//      fd  file descriptor to dump error to.
//
void Command::dumpError(int fd)
{
    // If "error" has been specified, "_error" is most likely not
    // something we can read from.
    assert(!_impl->_errorSpecified);

    char c;
    while (read(_impl->_error, &c, 1) == 1) {
	write(fd, &c, 1);
    }
}

END_NAMESPACE(sysadm);
