//
// Command.h
//
//	Class for running commands.
//
//
//  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_COMMAND_H
#define _SYSADM_COMMAND_H
#ident "$Revision: 1.8 $"

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

//
// Command is a class for running a command in another process.
//
class Command {
  public:
    Command(const String& command);
    ~Command();

    // Call these before running the command to affect behavior.
    void setUid(uid_t uid);
    void setInput(int fd);
    void setOutput(int fd);
    void setError(int fd);
    void addArg(const String& arg);

    // Set the args to be passed to execv().  Command does not make a
    // copy of these arguments; they must not be freed until after
    // Command::run() has been called.
    void setArgs(const char* const* argList);
    
    // Set LANG=C for command to be run; useful for commands whose
    // output is to be parsed, rather than displayed to the user.
    void setLangC();

    // Use a pseudo-terminal instead of a pipe for output & error.
    void usePty();

    // Set the directory in which the command should be run.
    void setDir(const String& dir);

    // Run the command.  You can only run a command once.  Returns 0
    // if successful, -1 if error.
    int run();

    // These are only useful if the corresponding "set" methods have
    // *not* been called.  The can only be called after run();
    int getInput();
    int getOutput();
    int getError();
    void closeInput();
    FILE* getInputFP();
    FILE* getOutputFP();
    FILE* getErrorFP();

    void kill(int sig = SIGTERM);

    // Wait and get status.
    void wait();
    int getExitStatus();

    // Copy commands error to "fd".  Will not work if setError() was
    // called.
    void dumpError(int fd);

  private:
    // Intentionally undefined.
    Command(const Command&);
    Command& operator=(const Command&);    

    // Opaque implementation.
    class CommandImpl* _impl;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_COMMAND_H
