/*
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#include <assert.h>
#include <math.h>

#include <sysadm/DictionaryOf.h>

BEGIN_NAMESPACE(sysadm);

struct Bucket {
    Hashable *	key;		// bucket owns this key
    void *		value;		// not owned by bucket
    Bucket *		next;

    Bucket(Hashable * adoptedKey, void * val);
    Bucket(void);
    ~Bucket() { delete key; }
};

inline
Bucket::Bucket(Hashable * adoptedKey, void * val)
: key(adoptedKey), value(val), next(NULL)
{
}

inline
Bucket::Bucket(void)
: key(NULL), value(NULL), next(NULL)
{
}


class DictionaryOfImpl {
  public:
    DictionaryOfImpl(unsigned int buckets);
    DictionaryOfImpl(const DictionaryOfImpl & copy);
    ~DictionaryOfImpl(void);

    DictionaryOfImpl & operator=(const DictionaryOfImpl & other);

    void add(void * val, Hashable * adoptedKey);
    void * remove(const Hashable & key);
    void * lookup(const Hashable & key,
		  const Hashable** retKey = NULL) const;

    bool isPrime(unsigned int val);
    void copyBuckets(const Bucket * table, unsigned int size);
    void freeBuckets(void);

    Bucket		*_table;
    unsigned int	_buckets;
    unsigned int	_size;
    unsigned int	_chained;
    unsigned int	_max_chain;
};

END_NAMESPACE(sysadm);

USING_NAMESPACE(sysadm);

DictionaryOfImpl::DictionaryOfImpl(unsigned int buckets)
: _table(NULL)
{
    // Adjust buckets to the nearest prime larger than the value given.
    // Start by rounding up to the nearest odd number.
    //
    if ((buckets % 2) == 0) {
	buckets += 1;
    }

    while(isPrime(buckets) == false) {
	buckets += 2;
    }

    _table = new Bucket[buckets];
    _buckets = buckets;

    _size = 0;
    _chained = 0;
    _max_chain = 0;
}

DictionaryOfImpl::DictionaryOfImpl(const DictionaryOfImpl & copy)
: _table( new Bucket[copy._buckets] ), _buckets(copy._buckets),
  _size( copy._size )
{
    copyBuckets(copy._table, copy._buckets);
}

DictionaryOfImpl::~DictionaryOfImpl(void)
{
    freeBuckets();
    delete [] _table;
}

DictionaryOfImpl &
DictionaryOfImpl::operator=(const DictionaryOfImpl & other)
{
    if( this != &other ) {
	freeBuckets();
	delete [] _table;

	_buckets = other._buckets;
	_table = new Bucket[_buckets];
	copyBuckets(other._table, other._buckets);
	_size = other._size;
    }

    return(*this);
}

void
DictionaryOfImpl::add(void * val, Hashable * adoptedKey)
{
    HashValue hash = adoptedKey->getHashValue();

    int slot = (int) hash % _buckets;

    int chain_length = 0;
    Bucket * b;
    for (b = &_table[slot]; b->next; b = b->next) {
	chain_length += 1;
	continue;
    }

    if (b->key == NULL) {
	b->key = adoptedKey;
	b->value = val;
	_size++;
    } else {
	if (*b->key == *adoptedKey) {
	    b->value = val;
	    delete adoptedKey;
	} else {
	    b->next = new Bucket(adoptedKey, val);
	    _size++;
	    _chained++;
	}
    }

    _max_chain = chain_length > _max_chain ? chain_length : _max_chain;
}

void *
DictionaryOfImpl::remove(const Hashable & key)
{
    HashValue hash = key.getHashValue();

    int slot = (int) hash % _buckets;

    // See if we actually have this object in the dictionary.
    //
    Bucket * last = NULL;
    Bucket * b;
    for (b = &_table[slot]; b; b = b->next) {
	if (b->key && *b->key == key) {
	    break;
	}
	last = b;
    }

    void * val = NULL;

    if (b) {
	val = b->value;
	if (last && last != &_table[slot]) {
	    last->next = b->next;
	    delete b;
	    _chained--;
	} else {
	    // Head of slot
	    delete b->key;
	    b->key = NULL;
	    b->value = NULL;
	}
	_size--;
    }

    return(val);
}

void *
DictionaryOfImpl::lookup(const Hashable & key,
			    const Hashable** retKey) const
{
    HashValue hash = key.getHashValue();

    int slot = (int) hash % _buckets;

    // See if we actually have this object in the dictionary.
    //
    for (Bucket * b = &_table[slot]; b; b = b->next) {
	if (b->key && *b->key == key) {
	    if (retKey) {
		*retKey = b->key;
	    }
	    return(b->value);
	}
    }

    return(NULL);
}

bool
DictionaryOfImpl::isPrime(unsigned int val)
{
    // Go up to the square root of the value.
    //
    int limit = (int)sqrt((double)val);

    for (int i = 3; i <= limit; i += 2) {
	if ((val % i) == 0) {
	    return(false);
	}
    }

    return(true);
}

void
DictionaryOfImpl::copyBuckets(const Bucket * table, unsigned int size)
{
    for (int b = 0; b < size; b++) {
	if (table[b].key) {
	    _table[b].key = table[b].key->clone();
	} else {
	    _table[b].key = NULL;
	}
	_table[b].value = table[b].value;
	Bucket * target = &_table[b];
	for (const Bucket * source = table[b].next;
	     source;  source = source->next) {
	    if (source->key) {
		target->next = new Bucket(source->key->clone(), source->value);
	    } else {
		target->next = new Bucket((Hashable *)NULL, source->value);
	    }
	    target = target->next;
	}

	target->next = NULL;
    }
}

void
DictionaryOfImpl::freeBuckets(void)
{
    for (int b = 0; b < _buckets; b++) {
	for (Bucket * toFree = _table[b].next; toFree;) {
	    Bucket * n = toFree->next;
	    delete toFree;
	    toFree = n;
	}
	delete _table[b].key;
	_table[b].key = NULL;
	_table[b].value = NULL;
	_table[b].next = NULL;
    }
}

DictionaryOfTempl::DictionaryOfTempl(unsigned int buckets)
: _impl(new DictionaryOfImpl(buckets))
{
}

DictionaryOfTempl::DictionaryOfTempl(const DictionaryOfTempl & copy)
: _impl(new DictionaryOfImpl(*copy._impl))
{
}

DictionaryOfTempl::~DictionaryOfTempl(void)
{
    delete _impl;
}

DictionaryOfTempl &
DictionaryOfTempl::operator=(const DictionaryOfTempl & other)
{
    *_impl = *other._impl;
    return(*this);
}

void
DictionaryOfTempl::add(void * element, Hashable * adoptedKey)
{
    _impl->add(element, adoptedKey);
}

void *
DictionaryOfTempl::remove(const Hashable & key)
{
    return(_impl->remove(key));
}

void *
DictionaryOfTempl::lookupByKey(const Hashable & key,
				  const Hashable** retKey)
{
    return(_impl->lookup(key, retKey));
}

const void *
DictionaryOfTempl::lookupByKey(const Hashable & key,
				  const Hashable** retKey) const
{
    return(_impl->lookup(key, retKey));
}

void *
DictionaryOfTempl::operator[](const Hashable & key)
{
    return(_impl->lookup(key));
}

const void *
DictionaryOfTempl::operator[](const Hashable & key) const
{
    return(_impl->lookup(key));
}

unsigned int
DictionaryOfTempl::getSize() const
{
    return(_impl->_size);
}

unsigned int
DictionaryOfTempl::getBucketOverflow() const
{
    return(_impl->_chained);
}

unsigned int
DictionaryOfTempl::getMaxChain() const
{
    return(_impl->_max_chain);
}

BEGIN_NAMESPACE(sysadm);

class DictionaryOfTImpl {
  public:
    DictionaryOfTImpl(DictionaryOfTempl * dict);
    ~DictionaryOfTImpl(void);

    void reset(void);
    void advance(void);

    void * operator()(void);
    void * remove(void);

    DictionaryOfImpl *	_dict;
    Bucket *			_cur_bucket;
    Bucket *			_prev_bucket;
    unsigned int		_cur_slot;
    bool			_at_end;
};

END_NAMESPACE(sysadm);

DictionaryOfTImpl::DictionaryOfTImpl(DictionaryOfTempl * dict)
: _dict(dict->_impl)
{
    reset();
}

DictionaryOfTImpl::~DictionaryOfTImpl(void)
{
}

void
DictionaryOfTImpl::reset(void)
{
    _cur_bucket = &_dict->_table[0];
    _cur_slot = 0;
    _prev_bucket = NULL;
    _at_end = false;

    // Advance the pointer to the first non-null bucket.
    //
    while (_cur_bucket->key == NULL) {
	_cur_bucket = _cur_bucket->next;
	if (_cur_bucket == NULL) {
	    _cur_slot += 1;
	    if (_cur_slot >= _dict->_buckets) {
		break;
	    }

	    _cur_bucket = &_dict->_table[_cur_slot];
	}
    }

    if (_cur_bucket == NULL) {
	_at_end = true; // Empty table!
    }
}

void
DictionaryOfTImpl::advance(void)
{
    // Advance the pointer to the first non-null bucket.
    //

    do {
	_cur_bucket = _cur_bucket->next;
	if (_cur_bucket == NULL) {
	    _cur_slot += 1;
	    if (_cur_slot >= _dict->_buckets) {
		break;
	    }

	    _cur_bucket = &_dict->_table[_cur_slot];
	}
    }
    while (_cur_bucket && _cur_bucket->key == NULL);

    if (_cur_bucket == NULL) {
	_at_end = true; // Empty table!
    }
}

void *
DictionaryOfTImpl::operator()(void)
{
    if (_at_end) {
	return(NULL);
    }

    void * ret = _cur_bucket->value;
    _prev_bucket = _cur_bucket;

    advance();

    return(ret);
}

void *
DictionaryOfTImpl::remove(void)
{
    if (!_prev_bucket) {
	return(NULL);
    }

    return(_dict->remove(*_prev_bucket->key));
}

DictionaryOfTemplIterator::DictionaryOfTemplIterator(DictionaryOfTempl * dict)
: _impl(new DictionaryOfTImpl(dict))
{
}

DictionaryOfTemplIterator::~DictionaryOfTemplIterator(void)
{
    delete _impl;
}

void
DictionaryOfTemplIterator::reset(void)
{
    _impl->reset();
}

void *
DictionaryOfTemplIterator::operator()(void)
{
    return((*_impl)());
}

Hashable *
DictionaryOfTemplIterator::key(void)
{
    return(_impl->_prev_bucket->key);
}

void *
DictionaryOfTemplIterator::remove(void)
{
    return(_impl->remove());
}
