#ifndef _SYSADM_DICTIONARY_OF_H
#define _SYSADM_DICTIONARY_OF_H
//
//  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#include <sysadm/Hashable.h>

#include <sysadm/DictionaryOfTempl.h>

BEGIN_NAMESPACE(sysadm);

// DictionaryOf is not a CollectionOf.  (Though it is, of course a collection)
// To support a dictionary as a collection requires us to provide a host of
// methods, which will generally never be called.  If you think you need
// disctionary to derive from collection (which it would, logically), think
// some more, then come talk to me (larrylf)
//
// Otherwise, Dictionary basically follows the rules of a collection,
// though while it doesn't own its elements, it does own its keys.

template <class Type> class ConstDictionaryOfIterator;

template <class Type> class DictionaryOf {
  public:
    // The number of buckets needs to be carefully selected. Notice the
    // default value is a prime number. Making the bucket count prime
    // reduces the likely hood of collisions, or at least keeps the
    // chain length relatively uniform across the buckets.
    //
    // The constructor will compute the next larger prime number if given
    // a non-prime value for buckets. This can create a rather compute
    // intensive construction if a large non-prime number is provided.
    // It can also generate a dramatically larger bucket count than
    // expected.
    //
    // The expectations of the contructor is that most bucket counts
    // will be relatively small numbers so the next prime number is
    // relatively close to the value given.
    //
    inline DictionaryOf(unsigned int buckets = 251);
    inline DictionaryOf(const DictionaryOf& copy);
    
    inline ~DictionaryOf(void);  

    inline DictionaryOf& operator=(const DictionaryOf& copy);

    inline void  add(Type* element, Hashable * adoptKey);
    inline Type* remove(const Hashable& key);

    // If retKey is non-NULL, it will be set to point the the key
    // used by the dictionary.  This is useful for maximizing string
    // sharing.
    inline Type* lookupByKey(const Hashable & key,
			     const Hashable** retKey  = NULL);
    inline const Type* lookupByKey(const Hashable & key,
				   const Hashable** retKey  = NULL) const;

    inline Type* operator[](const Hashable& key);
    inline const Type* operator[](const Hashable& key) const;

    inline unsigned int getSize() const;
    inline unsigned int getBucketOverflow() const;
    inline unsigned int getMaxChain() const;

  protected:
    friend class ConstDictionaryOfIterator<Type>;

    DictionaryOfTempl *		_templ;
};

template <class Type> class ConstDictionaryOfIterator {
  public:
    inline ConstDictionaryOfIterator(const DictionaryOf<Type> * dict);
    inline ~ConstDictionaryOfIterator(void);

    inline void reset(void);
    inline Type * operator()(void);

    inline const Hashable * key(void);

  protected:
    inline DictionaryOfTemplIterator * getImpl();

  private:
    DictionaryOfTemplIterator *	_templ;
};

template <class Type>
class DictionaryOfIterator : public ConstDictionaryOfIterator<Type> {
  public:
    inline DictionaryOfIterator(DictionaryOf<Type> * dict);

    inline Type * remove(void);
};

template <class Type> inline
void RemoveAndDestroyContentsOf( DictionaryOf<Type>* dictAlias );

END_NAMESPACE(sysadm);

#include <sysadm/DictionaryOfI.h>
#endif  //  _SYSADM_DICTIONARY_OF_H
