//
// Log.c++
//
//	Logging of sysadm messages.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.5 $"

#include <stdio.h>

#include <sysadm/Log.h>
#include <sysadm/LogListener.h>
#include <sysadm/i18n.h>

BEGIN_NAMESPACE(sysadm);

//
// The one instance of Log per address space.
//
static Log* gLog = NULL;

//
// The static logging methods are implemented by this macro.
//
#define LOG_METHOD(method, level) \
    void Log:: method (const char* module, const char* format, ...) \
    { \
        va_list args; \
        va_start(args, format); \
        getLog().message(module, level, format, args); \
    }

//
// static logging methods.
//
LOG_METHOD(trace, TRACE);
LOG_METHOD(debug, DEBUG);
LOG_METHOD(info, INFO);
LOG_METHOD(warning, WARNING);
LOG_METHOD(error, ERROR);
LOG_METHOD(fatal, FATAL);

//
// Constructor.
//
Log::Log()
{
}

//
// Get access to Log instance.
//
Log& Log::getLog()
{
    if (gLog == NULL) {
	gLog = new Log;
    }
    return *gLog;
}

//
//  void Log::message(const char* module, Level level, const char* format,
//  		      va_list args)
//
//  Description:
//      Log a message.  This consists of notifying our listeners that
//      a message has been logged.
//
//  Parameters:
//      module    The module that is logging this message.
//      level     Log::TRACE .. Log::FATAL.
//      format    printf-style format string.
//      args      Args for printf format string.
//
void Log::message(const char* module, Level level, const char* format,
		  va_list args)
{
    // Don't accept illegal levels.
    assert((level & ~(Log::ALL)) == 0);

    IteratorOver<LogListener> iter(&_listeners);
    LogListener* listener;
    while ((listener = iter()) != NULL) {
	if (listener->testMessage(module, level)) {
	    listener->handleMessage(module, level, format, args);
	}
    }
}

//
//  void Log::adoptLogListener(LogListener* listener)
//
//  Description:
//      Add a listener to the list of listeners that gets notified
//      when a message is logged.
//
//  Parameters:
//      listener  The listener to add.
//
void Log::adoptLogListener(LogListener* listener)
{
    _listeners.add(listener);
}

//
//  LogListener* Log::orphanLogListener(LogListener* listener)
//
//  Description:
//      Remove a listener from the list of listeners that gets
//      notified when a message is logged.
//
//  Parameters:
//      listener  The listener to remove.
//
//  Returns:
//	The removed listener.
//
LogListener* Log::orphanLogListener(LogListener* listener)
{
    return _listeners.removePointer(listener);
}

//
//  void Log::removeLogListener(LogListener* listener)
//
//  Description:
//      Remove a listener from the list of listeners that gets
//      notified when a message is logged.  Delete the listener.
//
//  Parameters:
//      listener  The listener to remove and delete.
//
void Log::removeLogListener(LogListener* listener)
{
    LogListener* toDelete = orphanLogListener(listener);
    // This prevents the caller from having to worry about whether it
    // should delete "listener" if it's not actually installed.
    assert(toDelete == listener);
    delete toDelete;
}

//
//  const char* Log::levelToString(Level level)
//
//  Parameters:
//      level  level to represent as a string.
//
//  Returns:
//	Localized string represnting level.
//
const char* Log::levelToString(Level level)
{
    // Don't accept illegal levels.
    assert((level & ~(Log::ALL)) == 0);

    switch (level) {
    case FATAL:     return i18n("FATAL");
    case ERROR:     return i18n("ERROR");
    case WARNING:   return i18n("WARNING");
    case INFO:      return i18n("INFO");
    case DEBUG:     return i18n("DEBUG");
    case TRACE:     return i18n("TRACE");
    }
    return "";
}

END_NAMESPACE(sysadm);
