//
// Log.h
//
//	Interface for logging sysadm messages
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ifndef _SYSADM_LOG_H
#define _SYSADM_LOG_H
#ident "$Revision: 1.4 $"

#include <stdarg.h>

#include <sysadm/CollectionOf.h>

BEGIN_NAMESPACE(sysadm);

class LogListener;

//
// Log is an interface that sysadm services use to log various levels
// of messages.  The destination of log messages is controlled by
// sysadmd command line options (in /etc/inetd.conf), and log messages
// are also available to clients via the "log" service.
//
// Messages logged via Log do not go to syslog unless specified on the
// sysadmd command line in /etc/inetd.conf.
//
// Log notifies its list of LogListeners whenever a message is
// logged.  The LogListeners output the messages to files, syslog, or
// whatever is appropriate.
//
// Each log message consists of a "module" string that represents the
// service the message is coming from (e.g. sysadmd), a "level" (which
// is implicit in the static logging methods), and a printf-style
// format string and varargs.
//
class Log {
  public:

    // Each message that's logged has a level associated with it.
    // Messages can be filtered based on their levels.
    //
    // These values correspond to values in the java version of this
    // class, so don't change them arbitrarily.
    enum Level {
	TRACE = 0x01,		// Trace execution.
	DEBUG = 0x02,		// Message for debugging purposes.
	INFO = 0x04,		// Informational message.
	WARNING = 0x08,		// Warning message.
	ERROR = 0x10,		// Non-fatal error message.
	FATAL = 0x20,		// Fatal error.
	ALL = TRACE | DEBUG | INFO | WARNING | ERROR | FATAL,
    };
	
    // MAXLEN is the longest a message can be.
    enum { MAXLEN = 4096 };

    // These static methods are used by sysadmd code and associated
    // services for logging.
    static void trace(const char* module, const char* format, ...);
    static void debug(const char* module, const char* format, ...);
    static void info(const char* module, const char* format, ...);
    static void warning(const char* module, const char* format, ...);
    static void error(const char* module, const char* format, ...);
    static void fatal(const char* module, const char* format, ...);

    // Convenience method for turning "level" into a localized
    // string.
    static const char* levelToString(Level level);

    // Get access to an instance of Log, for calling one of the
    // non-static methods.
    static Log& getLog();

    // Log a message.  The static logging methods all end up calling
    // this eventually.
    void message(const char* module, Level level, const char* format,
		 va_list args);

    // Add a listener to be notified of messages.
    void adoptLogListener(LogListener* listener);

    // Remove a listener from the the list of listeners that get
    // notified when a message arrives.
    LogListener* orphanLogListener(LogListener* listener);

    // Remove and delete a listener from the list of listeners that
    // get notified when a message arrives.
    void removeLogListener(LogListener* listener);

  private:

    // Constructor.
    Log();

    // Intentionally undefined.
    Log(const Log&);
    Log& operator=(const Log&);

    CollectionOf<LogListener> _listeners;
};

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_LOG_H
