//
// LogFilter.c++
//
//	Filter for log messages.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#ident "$Revision: 1.3 $"

#include <string.h>

#include <sysadm/LogFilter.h>
#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

//
// Constructor.
//
LogFilter::LogFilter()
: _levelMask(0)
{
}

//
// Destructor.
//
LogFilter::~LogFilter()
{
    RemoveAndDestroyContentsOf(&_modules);
}

//
//  void LogFilter::levelOn(int level)
//
//  Description:
//      Turn on a level or set of levels for this filter.
//
//  Parameters:
//      level  mask of levels to turn on.
//
void LogFilter::levelOn(int level)
{
    // Don't accept illegal levels.
    assert((level & ~(Log::ALL)) == 0);
    _levelMask |= level;
}

//
//  void LogFilter::levelOff(int level)
//
//  Description:
//      Turn off a level or set of levels for this filter.
//
//  Parameters:
//      level  mask of levels to turn off.
//
void LogFilter::levelOff(int level)
{
    // Don't accept illegal levels.
    assert((level & ~(Log::ALL)) == 0);
    _levelMask &= ~level;
}
 
//
//  void LogFilter::addModule(const String& module)
//
//  Description:
//      Add "module" to our list of modules.
//
//  Parameters:
//      module  module to add.
//
void LogFilter::addModule(const String& module)
{
    _modules.add(new String(module));
}

//
//  void LogFilter::addModuleList(const String& moduleList)
//
//  Description:
//      Add each module in a comma-separated list of modules to our
//      list.  This code is shared between sysadmd's command line
//      parsing and the log service's LogFilter specification code.
//
//  Parameters:
//      moduleList  comma separated list of modules to add.
//
void LogFilter::addModuleList(const String& moduleList)
{
    String modules(moduleList);
    char* lasts = NULL;
    char* module;
    for (module = strtok_r(modules.writableCharPtr(), ",", &lasts);
	 module != NULL;
	 module = strtok_r(NULL, ",", &lasts)) {
	addModule(module);
    }
}

//
//  void LogFilter::removeModule(const String& module)
//
//  Description:
//      Remove "module" from our list of modules.
//
//  Parameters:
//      module  The module to remove.
//
void LogFilter::removeModule(const String& module)
{
    _modules.remove(module);
}

//
//  bool LogFilter::testMessage(const char* module, Log::Level level)
//
//  Description:
//      Test a message to see if it should be logged.
//
//  Parameters:
//      module    module of message to be tested.
//      level     level of message to be tested.
//
//  Returns:
//	true if the message should be logged, false otherwise.
//
bool LogFilter::testMessage(const char* module, Log::Level level)
{
    if (!(level & _levelMask)) {
	return false;
    }

    if (_modules.getSize() == 0) {
	return true;
    }

    IteratorOver<String> iter(&_modules);
    String* f;
    while ((f = iter()) != NULL) {
	if (strcmp(module, *f) == 0) {
	    return true;
	}
    }

    return false;
}

END_NAMESPACE(sysadm);
