//
// OrderedFileName.c++
//
//	Holds an file name where the name is of the form:
//      <number>.<Name>
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

#include <string.h>
#include <strings.h>
#include <sysadm/Log.h>
#include <sys/types.h>
#include <dirent.h>
#include <limits.h>
#include <stdlib.h>
#include <ctype.h>
#include <sysadm/OrderedFileName.h>

#define CLASS_NAME "OrderedFileName"

USING_NAMESPACE(sysadm);

OrderedFileName::OrderedFileName(char* fileName, char* fullPath) {
    _fullPath = new char[strlen(fullPath) + 1];
    strcpy(_fullPath, fullPath);
    _fileName = new char[strlen(fileName) + 1];
    strcpy(_fileName, fileName);
    char* dotIndex;
    if ((dotIndex = strchr(fileName, '.')) == NULL
	|| !isdigit(fileName[0])) {
	// There is no order information in this file
	// name. Order based on name alone.
	_order = new char[1];
	_order[0] = '\0';
	_name = new char[strlen(fileName) + 1];
	strcpy(_name, fileName);
	return;
    }
    
    // Parse the order key and name from the filename,
    // which is in the format 'order'.'name'.
    int orderLen = dotIndex - fileName;
    _order = new char[orderLen + 1];
    strncpy(_order, fileName, orderLen);
    _order[orderLen] = '\0';
    _name = new char[strlen(&fileName[orderLen+1]) + 1];
    strcpy(_name, &fileName[orderLen+1]);
}

OrderedFileName::~OrderedFileName() {
    delete [] _order;
    delete [] _name;
    delete [] _fileName;
    delete [] _fullPath;
}

// Comparison function for qsort(3C)
int OrderedFileName::compare(const void* te1, const void* te2) {
    OrderedFileName* name1 = *((OrderedFileName**)te1);
    OrderedFileName* name2 = *((OrderedFileName**)te2);
    int retval = strcmp(name1->_order, name2->_order);
    if (retval != 0) {
	// The entries differ on their order key.
	return retval;
    }
    
    // No difference based on order key, compare based on
    // class name.
    return strcmp(name1->_name, name2->_name);
}

OrderedFileName** 
OrderedFileName::createOrderedFileNamesFromDir(const char* dirName) {
    // The magic number '10' is used for the initial allocation of the
    // OrderedFiles array.  Additional space will be allocated if more
    // Task entries are found in this directory.
    int fileAlloc = 10;
    int numFiles = 0;
    OrderedFileName** orderedFiles = NULL;
    Log::debug(CLASS_NAME, "Will read from directory: %s", dirName);
    DIR* dirp = opendir(dirName);
    char fullPath[PATH_MAX];
    if (dirp != NULL) {
	fullPath[0] = 0;
	struct dirent *dp;
	orderedFiles = new OrderedFileName*[fileAlloc];
	while ((dp = readdir(dirp)) != NULL) {
	    if (strcmp(dp->d_name, ".") != 0
		&& strcmp(dp->d_name, "..") != 0) {
		Log::debug(CLASS_NAME, dp->d_name);
		strcpy(fullPath, dirName);
		strcat(fullPath, "/");
		strcat(fullPath, dp->d_name);
		addOrderedFileName(new OrderedFileName(dp->d_name, fullPath),
				   orderedFiles, 
				   fileAlloc, 
				   numFiles);
	    };
	}	
	closedir(dirp);
	qsort(orderedFiles, numFiles, sizeof(OrderedFileName*),
	      OrderedFileName::compare);
	addOrderedFileName(NULL, orderedFiles, fileAlloc, numFiles); 
    } else {
	Log::debug(CLASS_NAME, "The directory %s does not exist.", dirName);
    }
    return orderedFiles;
}

void 
OrderedFileName::addOrderedFileName(OrderedFileName* file,
				    OrderedFileName**& orderedFilesPtr,
				    int filesAlloc,
				    int& numFiles) { 
    if ((numFiles+1)%filesAlloc == 0) {
	// Need more space in the orderedFiles array
	OrderedFileName** largerOrderedFiles =
	    new OrderedFileName*[numFiles+ 1 + filesAlloc];
	bcopy(orderedFilesPtr, largerOrderedFiles,
	      numFiles*sizeof(OrderedFileName*));
	delete [] orderedFilesPtr;
	orderedFilesPtr = largerOrderedFiles;
    }
    orderedFilesPtr[numFiles] = file;
    numFiles++;
}

    
void 
OrderedFileName::destroyOrderedFileNameList(OrderedFileName** list) {
    for (int ii = 0; list[ii] != NULL; ii++) {
	delete list[ii];
	list[ii] = NULL;
    }
    delete[] list;
}
      
