/*
 *  Copyright (c) 1996, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#include <assert.h>
#include <strings.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <libgen.h>
#include <unistd.h>
#include <limits.h>
#include <locale.h>
//#include <sys/types.h>
//#include <sys/stat.h>
#include <fcntl.h>

#include <sysadm/String.h>

BEGIN_NAMESPACE(sysadm);

// If StringRep::_len has this value, it means it's not really set.
//
#define LENGTH_UNSET 0xffffffff

// Trying to use a deleted string almost always results in a segv due
// to a null _rep.  Try to provide more meaningful information in this case.
// Just prints to stderr.
//
#define CHECK_REP(rep) if (rep == NULL) {\
    cerr << "Null master string, probably an attempt to use a deleted String" << endl; \
    assert(rep);\
}



inline void String::_detachRep()
{
    if ((_rep != NULL) && (--_rep->_refCnt <= 0)) {
	delete _rep;
    }
    _rep = NULL;
}

inline void String::_attachRep(volatile String::StringRep* r)
{
    assert(_rep == NULL);
    CHECK_REP(r);
    r->_refCnt++;
    _rep = r;
    assert(_rep && _rep->_refCnt > 1);
}

void String::_writableRep()
{
    assert(_rep);
    if (_rep->_refCnt > 1) {
	volatile String::StringRep* oldRep = _rep;
	_detachRep();
	_rep = new String::StringRep();
	if (oldRep->_string) {
	    // use _size to keep pre-alloc'd buffer sizes.  Also faster...
	    // safe, because alloc always pads.  the -1 keeps things from
	    // growing beyound...
	    _allocString(oldRep->_size-1);
	    strcpy(_rep->_string, oldRep->_string);
	}
    } else
	_rep->_len = LENGTH_UNSET;
}


String::StringRep::StringRep() 
{
    _string = NULL;
    _size = 0;
    _refCnt = 1;
    _len = LENGTH_UNSET;
}


String::StringRep::~StringRep()
{
    assert(_refCnt == 0);
    if (_string) {
	free(_string);
    }
}


const String String::NUL;
const String String::EMPTY("");


String::String(const char* str)
{
    _rep = new String::StringRep();
    if (str) {
	_allocString(strlen(str) + 1);
	strcpy(_rep->_string, str);
    }
}


String::String(const char* str, unsigned int length)
{
    assert(str != NULL);
    _rep = new String::StringRep();
    _allocString(length + 1);
    strncpy(_rep->_string, str, length);
    _rep->_string[length] = '\0';
}


String::String(void)
{
    _rep = new String::StringRep();
}


String::String(const String& si)
{
    _rep = NULL;
    _attachRep(si._rep);
}


String::String(unsigned int initialBufferSize)
{
    _rep = new String::StringRep();
    _allocString(initialBufferSize);
}


String::String(unsigned int initialBufferSize, const char fill)
{
    _rep = new String::StringRep();
    _allocString(initialBufferSize);
    memset(_rep->_string, fill, initialBufferSize);
}


String::~String(void)
{
    _detachRep();
}


String&
String::operator=(const String& other)
{
    CHECK_REP(this->_rep);
    if (this != &other) {
	_detachRep();
	_attachRep(other._rep);
    }

    return(*this);
}


unsigned long
String::getHashValue(void) const
{
    CHECK_REP(this->_rep);
    if (!_rep->_string) {
	return 0;
    }
    
    // We have forced proper alignment and sizing so we know the string
    // can be safely cast to an unsigned long. See String::_allocString.
    //
    unsigned long hash = 0;
    for (unsigned long* block = (unsigned long*)_rep->_string;
	 (char*)block < (_rep->_string + _rep->_size);
	 block++) {
	hash ^= *block;
    }

    return(hash);
}


Hashable*
String::clone() const
{
    CHECK_REP(this->_rep);
    return new String(*this);
}


unsigned int
String::getLength(bool force) const 
{
    CHECK_REP(this->_rep);
    if( !force && _rep->_len != LENGTH_UNSET )
	return _rep->_len;
    return (_rep->_len = (_rep->_string ? strlen(_rep->_string) : 0));
}

    
String::operator const char*(void) const
{
    CHECK_REP(this->_rep);
    return(_rep->_string);
}
    

char*
String::writableCharPtr(void)
{
    CHECK_REP(this->_rep);
    _writableRep();
    return(_rep->_string);
}


char&
String::operator[](CharIndex index)
{
    // could be trying to modify
    CHECK_REP(this->_rep);
    _writableRep();
    assert(_rep->_string && index < _rep->_size);
    return(_rep->_string[index]);
}


char
String::operator[](CharIndex index) const
{
    CHECK_REP(this->_rep);
    char rval = '\000';
    if (_rep->_string && index < _rep->_size) {
	rval = _rep->_string[index];
    }
    return rval;
}


bool
String::operator==(const Hashable& other) const
{
    // Are we dealing with another string?
    CHECK_REP(this->_rep);
    const String* str = dynamic_cast<const String*>(&other);
    if (str == NULL) {
	// Nope. Well, it probably isn't equal then.
	return(false);
    }

    return(*this == *str);
}

bool
String::operator!=(const Hashable& other) const
{
    return(!(*this == other));
}

bool
String::isSame(const String& other) const
{
    return isSame(other._rep->_string);
}

bool
String::operator==(const String& other) const
{
    return(*this == other._rep->_string);
}


bool
String::operator!=(const String& other) const
{
    return(*this != other._rep->_string);
}


bool
String::operator<(const String& other) const
{
    return(*this < other._rep->_string);
}


bool
String::operator>(const String& other) const
{
    return(*this > other._rep->_string);
}


bool
String::operator<=(const String& other) const
{
    return(*this <= other._rep->_string);
}


bool
String::operator>=(const String& other) const
{
    return(*this >= other._rep->_string);
}


bool
String::isSame(const char* other) const
{
    CHECK_REP(this->_rep);
    // If both strings are NULL, then call them equal.
    //
    if (_rep->_string == NULL && other == NULL) {
	return(true);
    }

    // If only one is null, the strings are not equal.
    //
    if (!_rep->_string || !other) {
	return(false);
    }

    // If both strings refer to the same bits, they are equal.
    // This also catches comparing an String to itself
    //
    if (_rep->_string == other) {
	return(true);
    }

    // We have two buffers, so compare the bits.
    //
    return(strcasecmp(_rep->_string, other) == 0);
}

bool
String::operator==(const char* other) const
{
    CHECK_REP(this->_rep);
    // If both strings are NULL, then call them equal.
    //
    if (_rep->_string == NULL && other == NULL) {
	return(true);
    }

    // If only one is null, the strings are not equal.
    //
    if (!_rep->_string || !other) {
	return(false);
    }

    // If both strings refer to the same bits, they are equal.
    // This also catches comparing an String to itself
    //
    if (_rep->_string == other) {
	return(true);
    }

    // We have two buffers, so compare the bits.
    //
    return(strcmp(_rep->_string, other) == 0);
}


bool
String::operator!=(const char* other) const
{
    return(!(*this == other));
}

bool
String::operator<(const char* other) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL && other == NULL) {
	return(false);
    }

    if (_rep->_string == NULL && other) {
	return(true);
    }

    if (_rep->_string && other == NULL) {
	return(false);
    }

    return(strcmp(_rep->_string, other) < 0);
}


bool
String::operator>(const char* other) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL && other == NULL) {
	return(false);
    }

    if (_rep->_string == NULL && other) {
	return(false);
    }

    if (_rep->_string && other == NULL) {
	return(true);
    }

    return(strcmp(_rep->_string, other) > 0);
}


bool
String::operator<=(const char* other) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL && other == NULL) {
	return(true);
    }

    if (_rep->_string == NULL && other) {
	return(true);
    }

    if (_rep->_string && other == NULL) {
	return(false);
    }

    return(strcmp(_rep->_string, other) <= 0);
}


bool
String::operator>=(const char* other) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL && other == NULL) {
	return(true);
    }

    if (_rep->_string == NULL && other) {
	return(false);
    }

    if (_rep->_string && other == NULL) {
	return(true);
    }

    return(strcmp(_rep->_string, other) >= 0);
}


String
String::operator+(const String& other) const
{
    CHECK_REP(this->_rep);

    String str;

    int len = getLength() + other.getLength() + 1;
    if (len == 0) {
	return(str);
    }

    str._allocString(len);

    if (_rep->_string) {
	strcpy(str._rep->_string, _rep->_string);
    } else {
	*str._rep->_string = '\0';
    }

    if (other._rep->_string) {
	strcat(str._rep->_string, other._rep->_string);
    }

    return(str);
}

String
String::operator+(const char* other) const
{
    CHECK_REP(this->_rep);

    String str;

    int len = getLength() + strlen(other) + 1;
    if (len == 0) {
	return(str);
    }

    str._allocString(len);

    if (_rep->_string) {
	strcpy(str._rep->_string, _rep->_string);
    } else {
	*str._rep->_string = '\0';
    }

    if (other) {
	strcat(str._rep->_string, other);
    }

    return(str);
}


String&
String::operator+=(const String& other)
{
    CHECK_REP(this->_rep);

    int otherLen = other._rep->_string ? strlen(other._rep->_string) : 0;    
    int totalLen = otherLen + 1;

    if( _rep->_string )
	totalLen += strlen( _rep->_string );

    if( _rep->_refCnt > 1 || _rep->_size < totalLen ) {
	// this little dance keeps rep from being deleted when detached
	_rep->_refCnt++;
	volatile StringRep* oldRep = _rep;

	_detachRep();
	_rep = new StringRep;

	_allocString( totalLen );

	if( oldRep->_string )
	    strcpy( _rep->_string, oldRep->_string );
	else
	    *_rep->_string = '\0';

	// ok, now let it go...
	if( --oldRep->_refCnt <= 0 )
	    delete oldRep;
    } 
    // We don;t need to do a _writableRep on an 'else' clause because we
    // know refCnt == 1.

    if( other._rep->_string ) {
	strcat( _rep->_string, other._rep->_string );
    }

    return *this;
}

String&
String::operator+=(const char* other)
{
    CHECK_REP(this->_rep);

    int otherLen = other ? strlen(other) : 0;    
    int totalLen = otherLen + 1;

    if( _rep->_string )
	totalLen += strlen( _rep->_string );

    if( _rep->_refCnt > 1 || _rep->_size < totalLen ) {
	// this little dance keeps rep from being deleted when detached
	_rep->_refCnt++;
	volatile StringRep* oldRep = _rep;

	_detachRep();
	_rep = new StringRep;

	_allocString( totalLen );

	if( oldRep->_string )
	    strcpy( _rep->_string, oldRep->_string );
	else
	    *_rep->_string = '\0';

	// ok, now let it go...
	if( --oldRep->_refCnt <= 0 )
	    delete oldRep;
    } 
    // We don;t need to do a _writableRep on an 'else' clause because we
    // know refCnt == 1.

    if( other ) {
	strcat( _rep->_string, other );
    }

    return *this;
}


CharIndex	String::NotFound = ULONG_MAX;

CharIndex
String::find(const char toFind,
		CharIndex startIndex,
		bool reverse) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL) {
	return(NotFound);
    }

    int start = (int)startIndex;
    assert(start < _rep->_size);

    char* found = NULL;
    if (reverse) {
	found = strrchr(&_rep->_string[start], toFind);
    } else {
	found = strchr(&_rep->_string[start], toFind);
    }

    return(found ? ((CharIndex)(found - _rep->_string)) : NotFound);
}


CharIndex
String::find(const String& toFind,
		CharIndex startIndex,
		bool reverse) const
{
    CHECK_REP(this->_rep);
    if (_rep->_string == NULL || toFind._rep->_string == NULL) {
	return(NotFound);
    }

    int start = (int)startIndex;
    assert(start < _rep->_size);

    char* found = NULL;
    if (reverse) {
	char* last_found = NULL;

	found = _rep->_string - 1;
	do {
	    last_found = found;
	    found = strstr(last_found + 1, toFind._rep->_string);
	} while(found);
	found = last_found;
    } else {
	found = strstr(_rep->_string, toFind._rep->_string);
    }

    if (found) {
	return(found - _rep->_string);
    }

    return(NotFound);
}


String
String::getSubString(CharIndex start,
			CharIndex end) const
{
    CHECK_REP(this->_rep);
    assert(_rep->_string);
    assert((start < _rep->_size) && (end < _rep->_size));
    
    unsigned long len = end - start + 1;

    String str(len);

    strncpy(str._rep->_string, &_rep->_string[start], len - 1);
    str._rep->_string[len - 1] = '\000';

    return(str);
}


void
String::toLowerCase(void)
{
    CHECK_REP(this->_rep);
    _writableRep();
    for (char* c = _rep->_string; *c; c++) {
	if (isalpha(*c)) {
	    *c = tolower(*c);
	}
    }
}


void
String::toUpperCase(void)
{
    CHECK_REP(this->_rep);
    _writableRep();
    for (char* c = _rep->_string; *c; c++) {
	if (isalpha(*c)) {
	    *c = toupper(*c);
	}
    }
}


long
String::asLong(int base) const
{
    CHECK_REP(this->_rep);
    assert(_rep->_string);
    return(strtol(_rep->_string, NULL, base));
}


double
String::asDouble(void) const
{
    CHECK_REP(this->_rep);
    assert(_rep->_string);
    return(strtod(_rep->_string, NULL));
}


void String::fromNum(long num) 
{
    char buf[256];
    sprintf(buf, "%ld", num);

    _detachRep();
    _rep = new String::StringRep();

    _allocString(strlen(buf)+1);
    strcpy(_rep->_string, buf);
}


void String::fromNum(double num)
{
    char buf[256];
    sprintf(buf, "%lf", num);

    _detachRep();
    _rep = new String::StringRep();

    _allocString(strlen(buf)+1);
    strcpy(_rep->_string, buf);
}



ostream& operator<<(ostream& stream, const String& str)
{
    CHECK_REP(str._rep);
    if (((const char*)str)) {
	return(stream << (const char*)str);
    } else {
	return(stream << "");
    }
}


istream& operator>>(istream& stream, String& str)
{
    CHECK_REP(str._rep);
    stream >> str.writableCharPtr();
    return(stream);
}


void
String::_allocString(unsigned int size)
{
    // We have to hash this string, and we want to do it quickly. To
    // accomplish that goal, we will force the string to be aligned on
    // unsigned long boundaries, and be an even multiple of unsigned long
    // bytes long.
    //
    CHECK_REP(_rep);
    int align = sizeof(unsigned long);

    size += (align - (size % align));

    assert(_rep->_string == NULL);
    _rep->_string = (char*)memalign(align, size);
    assert(_rep->_string);

    *_rep->_string = 0;

    // Initialize the last "word" of the buffer to avoid unintialized
    // reads of the guard bytes.
    //
    memset((_rep->_string + size - align), 0, align);

    _rep->_size = size;
    _rep->_len = LENGTH_UNSET;
}





// ***** Helper Functions *****

bool StringToLong( const String& str, long& retVal )
{
    char* next;
    retVal = strtol(str, &next, 10);
    if( next != str )
	return true;
    return false;
}

bool StringToLongLong( const String& str, long long& retVal )
{
    return sscanf(str, "%lld", &retVal);
}

bool StringToDouble( const String& str, double& retVal )
{
    String unConstStr(str);
    setlocale(LC_NUMERIC, "C");
    
    char* array = unConstStr.writableCharPtr();
    
    for (int i = 0; i < str.getLength(); i++) {
	if (array[i] == ',') {
	    array[i] = '.';
	    break;
	}
    }
    
    char* next;
    retVal = strtod(unConstStr, &next);
    setlocale(LC_NUMERIC, "");
    
    if( next != str )
	return true;
    return false;
}
    
// handles t/f, true/false, on/off, 1/0
bool StringToBool( const String& str, bool& retVal )
{
    if( str.isSame("t") || str.isSame("true") || 
        str.isSame("on") || str.isSame("1") ) {
	retVal = true;
	return true;
    }

    if( str.isSame("f") || str.isSame("false") || 
        str.isSame("off") || str.isSame("0") ) {
	retVal = false;
	return true;
    }

    return false;
}

bool StringToLongPair( const String& str, long& retVal1, long& retVal2 )
{
    char* next;
    retVal1 = strtol(str, &next, 10);
    if( next != str ) {
	char* n;
	retVal2 = strtol( next+1, &n, 10 );

	if( n != next )
	    return true;
    }

    return false;
}

String StringFromLong( long val )
{
    char buf[20];
    sprintf(buf, "%ld", val);
    return String(buf);
}

String StringFromLongLong( long long val )
{
    char buf[30];
    sprintf(buf, "%lld", val);
    return String(buf);
}

String StringFromDouble( double val )
{
    char buf[30];
    sprintf(buf, "%g", val);
    return String(buf);
}

// Converts to "true" or "false"
String StringFromBool( bool val )
{
    return StringFromBool(val, "true", "false");
}

String StringFromBool( bool val, const String& trueStr, 
			               const String& falseStr )
{
    if( val )
	return trueStr;

    return falseStr;
}

String StringFromLongPair( long val1, long val2 )
{
    char buf[50];
    sprintf(buf, "%ld, %ld", val1, val2);
    return String(buf);
}

String StringFromFile(const char * fileName) {
    static const int READ_SIZE = 1024;
    char * buf = (char *)malloc(sizeof(char) * READ_SIZE);
    char * ptr = buf;
    int bytesRead = 0;
    int numBytes;
    int fd = open(fileName, O_RDONLY);
    if (fd == -1) {
	return String::NUL;
    }
    while ((numBytes = read(fd, ptr, READ_SIZE - 1)) > 0) {
	bytesRead += numBytes;
	ptr[numBytes] = 0;
	ptr += numBytes;
	if (bytesRead + READ_SIZE < (sizeof(buf) / sizeof(char))) {
	    int offset = ptr - buf;
	    buf = (char *)realloc(buf, sizeof(buf) / sizeof(char) + READ_SIZE);
	    ptr = buf + offset;
	}
    }
    if (numBytes == -1) {
	close(fd);
	return String::NUL;
    }
    close(fd);
    String returnVal = String(buf);
    free(buf);
    return returnVal;
}

END_NAMESPACE(sysadm);









