/*
 *  Copyright (c) 1995, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ifndef _SYSADM_STRING_H
#define _SYSADM_STRING_H

#include <sysadm/Hashable.h>

#include <iostream.h>
#include <float.h>
#include <malloc.h>

// *****************************  WARNING  ************************************
//
// In <ligben.h>, both dirname() and basename() incorrectly
// claim to take const char* arguments, when in fact both will
// modify the argument.  Since Strings are reference counted and
// copy on write, using basename() or dirname() on an String can
// lead to erroneous results.
//
// ****************************************************************************

BEGIN_NAMESPACE(sysadm);

//  Nuts, utmp.h goofs us up.
#ifdef EMPTY
#undef EMPTY
#endif

class String;
typedef unsigned long CharIndex;

// String is a reference counted, copy on write string class

class String : public Hashable {

    friend ostream& operator<<(ostream& stream, const String& string);
    friend istream& operator>>(istream& stream, String& string);

  public:
    // NOTE: the void constructor gives a string with a NULL value as
    //       opposed to an empty string.  To build an empty String,
    //	     use String("").
    String(void);
    String(const char* string);
    String(const String& string);
    String(unsigned int initialBufferSize);
    String(unsigned int initialBufferSize, const char fill);
    String(const char* string, unsigned int length);
    virtual ~String(void);

    String& operator=(const String& string);

    // override hashable
    virtual HashValue getHashValue(void) const;
    virtual Hashable* clone() const;

    // getLength() returns the length of the string (like strlen).
    //
    // An attempt is made to cache the length of the string,
    // so the value returned by this method can be wrong if
    // a writable char* is used to change the string multiple times
    // combined with multiple uses of getLength();  Calling with
    // force = true can be used in these cases to ensure the correct value
    // at the cost of performence.
    unsigned int getLength(bool force = false) const;
    operator const char*(void) const;

    // We can't use operator char*() since it will prevent
    // operator const char*() from being used most of the time.
    // Hence, we provide a functional interface for getting a
    // pointer to the non-const characters.
    //
    char* writableCharPtr(void);

    // These two array operators function on logical characters in
    // the string. These take into account multi-byte issues.
    // XXX: I don't think that multi-byte is handled.
    //
    char& operator[](CharIndex index);
    char operator[](CharIndex index) const;

    virtual bool operator==(const Hashable & other) const;
    virtual bool operator!=(const Hashable & other) const;

    bool isSame( const String& other) const; // case not significant
    bool operator==(const String& other) const;
    bool operator!=(const String& other) const;
    bool operator<(const String& other) const;
    bool operator>(const String& other) const;
    bool operator<=(const String& other) const;
    bool operator>=(const String& other) const;

    bool isSame( const char* other) const; // case not significant
    bool operator==(const char* other) const;
    bool operator!=(const char * other) const;
    bool operator<(const char * other) const;
    bool operator>(const char * other) const;
    bool operator<=(const char * other) const;
    bool operator>=(const char * other) const;
 
    String operator+(const String& string) const;
    String& operator+=(const String& string);
    String operator+(const char* string) const;
    String& operator+=(const char* string);

    CharIndex find(const char toFind,
		      CharIndex startIndex = 0, 
		      bool reverse = false) const;
    CharIndex find(const String& toFind,
		      CharIndex startIndex = 0, 
		      bool reverse = false ) const;
    static CharIndex	NotFound;

    // Returns a new string that is a substring of this string.  The
    // substring begins at "startIndex" and extends to the character
    // index at "endIndex - 1".
    String getSubString(CharIndex startIndex,
			   CharIndex endIndex) const;

    void   toLowerCase(void);
    void   toUpperCase(void);
    long   asLong(int base = 10) const;
    double asDouble(void) const;
    void   fromNum( long num );
    void   fromNum( double num );

    // A "null" string, as opposed to an empty string.  Useful
    // for methods which return an String& to distinguish
    // between "no string" and "emtpy string".  For example,
    //    const String& str = func();
    //    if (str == String::NUL) no_return_value();
    //
    static const String NUL;

    // The empty string, encourages string sharing
    static const String EMPTY;

  private:

    // StringRep is the master string used for reference counting
    // XXX: We need a lock here when we get cheap mutexes
    class StringRep {
     public:
	StringRep();
	~StringRep();

	// StringReps should not be copied or assigned, so these are
	// declared but left undefined.
	StringRep(const StringRep&);
	StringRep& operator=(const StringRep&);

	char*		_string;
	unsigned int	_refCnt;
	unsigned int	_size;
	unsigned int	_len;
    };
    volatile StringRep*	_rep;

    void _allocString(unsigned int size);
    inline void _attachRep(volatile StringRep* r);
    inline void _detachRep();
    void _writableRep();
};

// The following utility functions return true if the conversion occured
bool StringToLong( const String& str, long& retVal );
bool StringToLongLong( const String& str, long long& retVal );
bool StringToDouble( const String& str, double& retVal );
// handles t/f, true/false, on/off, 1/0
bool StringToBool( const String& str, bool& retVal );
bool StringToLongPair( const String& str, long& retVal1, long& retVal2 );

String StringFromLong( long val );
String StringFromLongLong( long long val );
String StringFromDouble( double val );
// Converts to "True" or "False"
String StringFromBool( bool val );
String StringFromBool( bool val, const String& trueStr,
		       const String& falseStr );
String StringFromLongPair( long val1, long val2 );

// Reads in the contents of a file and returns a String with the
// contents.  If there is a problem opening the file or reading it,
// NUL will be returned, and errno is set to indicate the error.
String StringFromFile(const char * fileName);

END_NAMESPACE(sysadm);
#endif  //  _SYSADM_STRING_H

