/*
 * io.c
 *
 *	sysadm i/o wrappers.
 *
 *
 *  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2.1 of the GNU Lesser General Public
 *  License as published by the Free Software Foundation.
 *  
 *  This program is distributed in the hope that it would be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  
 *  Further, this software is distributed without any warranty that it is
 *  free of the rightful claim of any third person regarding infringement
 *  or the like.  Any license provided herein, whether implied or
 *  otherwise, applies only to this software file.  Patent licenses, if
 *  any, provided herein do not apply to combinations of this program
 *  with other software, or any other product whatsoever.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
 *  USA.
 *  
 *  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
 *  Mountain View, CA 94043, or http://www.sgi.com/
 *  
 *  For further information regarding this notice, see:
 *  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
 */

#ident "$Revision: 1.3 $"

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#include <sysadm/io.h>

/*
 *  ssize_t SaRead(int fd, void *buf, size_t n)
 *
 *  Description:
 *      A version of read() that retries when signals are received.
 *
 *  Parameters:
 *      fd   File descriptor to read.
 *      buf  Buffer to receive data.
 *      n    Number of bytes to read.
 *
 *  Returns:
 *	Number of bytes read.
 */
ssize_t SaRead(int fd, void *buf, size_t n)
{
    int nRead;

    while ((nRead = read(fd, buf, n)) == -1
	   && errno == EINTR) {
	continue;
    }

    return nRead;
}

/*
 *  ssize_t SaReadFully(int fd, void *buf, size_t n)
 *
 *  Description:
 *      A version of read() that retries until "n" bytes are read or
 *	an error occurs.
 *
 *  Parameters:
 *      fd   File descriptor to read.
 *      buf  Buffer to receive data.
 *      n    Number of bytes to read.
 *
 *  Returns:
 *	Number of bytes read.
 */
ssize_t SaReadFully(int fd, void *buf, size_t n)
{
    int total = 0, thistime;
    char *b = buf;

    assert (buf != NULL);

    while (n > 0) {
	thistime = SaRead( fd, b, n);
	if (thistime < 0) {
	    return -1;
	} else if (thistime == 0) {
	    return total;
	}
	total += thistime;
	n -= thistime;
	b += thistime;
    }
    return total;
}

/*
 *  ssize_t SaWrite(int fd, const void *buf, size_t n)
 *
 *  Description:
 *      A version of write retries when signals are received.
 *
 *  Parameters:
 *      fd   File descriptor to write.
 *      buf  Buffer from which to write.
 *      n    Number of bytes to write.
 *
 *  Returns:
 *	Number of bytes written.
 */
ssize_t SaWrite(int fd, const void *buf, size_t n)
{
    int nWritten;

    while ((nWritten = write(fd, buf, n)) == -1
	   && errno == EINTR) {
	continue;
    }

    return nWritten;
}

/*
 *  ssize_t SaWriteFully(int fd, const void *buf, size_t n)
 *
 *  Description:
 *      A version of write() that keeps writing until "n" bytes are
 *	written or an error occurs.
 *
 *  Parameters:
 *      fd   File descriptor to write.
 *      buf  Buffer from which to write.
 *      n    Number of bytes to write.
 *
 *  Returns:
 *	Number of bytes written.
 */
ssize_t SaWriteFully(int fd, const void *buf, size_t n)
{
    int total = 0, thistime;
    const char *b = buf;

    assert (buf != NULL);

    while (n > 0)
    {
	thistime = SaWrite(fd, b, n);
	if (thistime < 0) {
	    return -1;
	}
	n -= thistime;
	total += thistime;
	b += thistime;
    }
    return total;
}
