//
//  Copyright (c) 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2 of the GNU General Public License as
//  published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
#include <sysadm/CollectionOf.h>
#include <sysadm/OrderedCollectionOf.h>
#include <sysadm/DictionaryOf.h>
#include <sysadm/String.h>

#include <iostream.h>
#include <stdio.h>

using namespace sysadm;

class Base {
  public:
    Base( int n ) : _n(n) {}
    virtual ~Base() {}

    virtual int get() const { return _n; }

  private:
    int _n;
};


class Derived : public Base {
  public:
    Derived( int n ) : Base(n) {}
    ~Derived() {}

    virtual int get() const { return 100 * Base::get(); }
};


class CPTest {
  public:
      CPTest() { cout << "CPTest created\n"; }
      ~CPTest() { cout << "CPTest deleted\n"; }
  };


void
main( int argc, char* argv[] )
{
    cout << "Testing type-safe casting mechanism\n";
    {
	Base b(1);
	Derived d(2);
	Base* bp = &b;

	Derived* dp = dynamic_cast<Derived*>(bp);
	if( dp ) {
	    cout << "Bad downcast allowed!!\n";
	    exit(1);
	}

	bp = &d;
	if( !dynamic_cast<Derived*>(bp) ) {
	    cerr << "Downcast failed!!\n";
	    exit(1);
	}

	cout << "Type-Safe cast PASSED\n\n";
    }

    cout << "Testing CollectionOf\n";

    {
	CollectionOf<long> bag;

	for( int i = 0; i < 10; i++ )
	    bag.add( new long(i) );

	IteratorOver<long> iter(&bag);
	long* item = NULL;

	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end, size = " << bag.getSize() << "\n";

	delete bag.remove( 5 );

	iter.reset();
	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end (5 should be gone)\n";

	iter.getFirst();
	delete iter.remove();

	iter.reset();
	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end (0 & 5 should be gone)\n";
	
	// Test the copy constructor
	CollectionOf<long> *bagCopy = 
	    new CollectionOf<long>(bag);
	    
	IteratorOver<long> iterCopy(bagCopy);
	long *itemCopy = NULL;

	cout << "(copy ctor) bagCopy: ";
	while( (itemCopy = iterCopy()) != NULL )
	    cout << *itemCopy << " ";

	cout << " end, size = " << bagCopy->getSize() << "\n";
	 
	bagCopy->removeAll();
	delete bagCopy;
	
	// Test the assignment copy constructor
	CollectionOf<long> bagAssign = bag;
	    
	IteratorOver<long> iterAssign(&bagAssign);
	long *itemAssign = NULL;

	cout << "(assignment copy ctor) bagAssign: ";
	while( (itemAssign = iterAssign()) != NULL )
	    cout << *itemAssign << " ";

	cout << " end, size = " << bagAssign.getSize() << "\n";
	 
	bagAssign.removeAll();
	
	// Test assignment
	bagAssign = bag;
	
	IteratorOver<long> iterAssign2(&bagAssign);
	itemAssign = NULL;

	cout << "(assignment op) bagAssign: ";
	while( (itemAssign = iterAssign2()) != NULL )
	    cout << *itemAssign << " ";

	cout << " end, size = " << bagAssign.getSize() << "\n";
	 
	bagAssign.removeAll();
	
	RemoveAndDestroyContentsOf(&bag);

	cout << "after RemoveAndDestroyContentsOf, size is: " <<
		bag.getSize() << "\n";

	RemoveAndDestroyContentsOf(&bag);

	cout << " End of CollectionOf testing\n\n";
    }

    cout << "Testing CollectionOf with const objects\n";
    {
	// Do it again, with a const long
	CollectionOf<const long> bagConst;

	static const long longs[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

	for( int i = 0; i < sizeof longs/sizeof *longs; i++ )
		bagConst.add( longs + i );

	IteratorOver<const long> iterConst(&bagConst);

	const long *itemConst = NULL;

	while ( (itemConst = iterConst()) != NULL ) {
	    cout << *itemConst << " ";
	}

	cout << " end, size = " << bagConst.getSize() << "\n";
	
	// Can't delete it 'cause it's const.  Sure, this is a leak.
	bagConst.remove(5);

	iterConst.reset();

	while ( (itemConst = iterConst()) != NULL ) {
	    cout << *itemConst << " ";
	}

	cout << " end (5 should be gone)\n";

	iterConst.getFirst();

	// Again, can't delete 'cause it's const.
	iterConst.remove();

	iterConst.reset();
	while( (itemConst = iterConst()) != NULL )
	    cout << *itemConst << " ";

	cout << " end (0 & 5 should be gone)\n";

	// Test the copy constructor
	CollectionOf<const long> *bagConstCopy = 
	    new CollectionOf<const long>(bagConst);
	    
	IteratorOver<const long> iterConstCopy(bagConstCopy);
	const long *itemConstCopy = NULL;

	cout << "(copy ctor) bagConstCopy: ";
	while( (itemConstCopy = iterConstCopy()) != NULL )
	    cout << *itemConstCopy << " ";

	cout << " end, size = " << bagConstCopy->getSize() << "\n";

	bagConstCopy->removeAll();
	delete bagConstCopy;

	// Test the assignment copy constructor
	CollectionOf<const long> bagConstAssign = bagConst;
	    
	IteratorOver<const long> iterConstAssign(&bagConstAssign);
	const long *itemConstAssign = NULL;
	    
	cout << "(assigment copy ctor) bagConstAssign: ";
	while( (itemConstAssign = iterConstAssign()) != NULL )
	    cout << *itemConstAssign << " ";

	cout << " end, size = " << bagConstAssign.getSize() << "\n";

	// Test the assignment operator
	bagConstAssign = bagConst;
	IteratorOver<const long> iterConstAssign2(&bagConstAssign);
	itemConstAssign = NULL;
	    
	cout << "(assignment op) bagConstAssign: ";
	while( (itemConstAssign = iterConstAssign2()) != NULL )
	    cout << *itemConstAssign << " ";

	cout << " end, size = " << bagConstAssign.getSize() << "\n";

	bagConstAssign.removeAll();
	
	// Can't removeAndDestroy 'cause they're const.
//	RemoveAndDestroyContentsOf(&bagConst);

	cout << " End of CollectionOf with const objects testing\n\n";
    }

    cout << "Testing ConstIteratorOver\n";
    {
	// The point of this is really just to make sure this code can
	// compile.
	CollectionOf<long> bag;

	for( int i = 0; i < 10; i++ )
	    bag.add( new long(i) );

	const CollectionOf<long>* bagAlias = &bag;
	ConstIteratorOver<long> iter(bagAlias);

	long *item = NULL;
	while( (item = iter()) != NULL ) {
	    cout << *item << " ";
	}
	
	cout << " end\n";

	RemoveAndDestroyContentsOf(&bag);
	cout << " End of ConstIteratorOver testing\n\n";
    }

    cout << "Testing OrderedCollectionOf\n";
    {
	OrderedCollectionOf<int> list;

	for( int i = 0; i < 10; i++ )
	    list.prepend( new int(i) );

	IteratorOver<int> iter(&list);
	int* item = NULL;

	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end (should be reversed)\n";

	list.insertBefore( new int(-1), 3 );
	list.insertAfter( new int(-2), 7 );

	cout << "inserted -1 before '3' and -2 after '7'\n";

	for( int* it = iter.getFirst(); it != NULL; it = iter.getNext() )
	    cout << *it << " ";

	cout << " end\n";

	RemoveAndDestroyContentsOf(&list);

	cout << "after RemoveAndDestroyContentsOf, size is: " <<
		list.getSize() << "\n";

	cout << " End of OrderedCollectionOf testing\n\n";
    }


    cout << "Testing ConstCastIteratorOver\n";
    {
	CollectionOf<Base> bag;

	for( int i = 0; i < 10; i++ )
	{
	    if( i & 1 )
		bag.add( new Derived(i) );
	    else
		bag.add( new Base(i) );
	}

	cout << "The whole bag:\n";
	ConstIteratorOver<Base> iter(&bag);

	Base* item = NULL;
	while( (item = iter()) != NULL ) {
	    cout << item->get() << " ";
	}
	
	cout << "\nDerived only:\n";
	ConstCastIteratorOver<Derived, Base> citer(&bag);

	Derived* ditem = NULL;
	for( ditem = citer.getFirst(); ditem != NULL; ditem = citer.getNext() )
	{
	    cout << ditem->get() << " ";
	}
	

	cout << " end\n";

	RemoveAndDestroyContentsOf(&bag);
	cout << " End of ConstIteratorOver testing\n\n";
    }

    cout << "Testing DictionaryOf\n";
    {
	DictionaryOf<char> dict(5);

	char buf[10];
	for( int i = 0; i < 10; i++ )
	{
	    char* item = new char('A'+i);
	    sprintf(buf, "%d", i);

	    dict.add( item, new String(buf) );
	}

	DictionaryOf<char> dict2(dict);

	DictionaryOfIterator<char> iter(&dict);
	char* item = NULL;

	cout << "values: ";
	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end \n";
	iter.reset();

	cout << "  keys: ";
	while( (item = iter()) != NULL ) {
	    const String* str = dynamic_cast<const String*>(iter.key());
	    if( str ) 
		cout << *str << " ";
	}

	cout << " end \n";
	iter.reset();

	cout << "Testing that dictionary copy constructor worked" << endl;
	cout << "Keys and values should be the same as above" << endl;
	DictionaryOfIterator<char> iter2(&dict2);
	item = NULL;

	cout << "values: ";
	while( (item = iter2()) != NULL )
	    cout << *item << " ";

	cout << " end \n";
	iter2.reset();

	cout << "  keys: ";
	while( (item = iter2()) != NULL ) {
	    const String* str = dynamic_cast<const String*>(iter2.key());
	    if( str ) 
		cout << *str << " ";
	}

	cout << " end \n";

	cout << "Size: " << dict.getSize() << endl;
	cout << "Overflow: " << dict.getBucketOverflow() << endl;
	cout << "Max Chain: " << dict.getMaxChain() << endl;

	cout << "getting item 4: " << *dict[String("4")] << endl;
	cout << "getting item 8: " << *dict[String("8")] << endl;
	cout << "getting item 1: " << *dict[String("1")] << endl;

	cout << "Removing items 3 and 9\n";
	delete dict.remove(String("3"));
	delete dict.remove(String("9"));

	cout << "values: ";
	while( (item = iter()) != NULL )
	    cout << *item << " ";

	cout << " end \n";
	iter.reset();

	cout << "  keys: ";
	while( (item = iter()) != NULL ) {
	    const String* str = dynamic_cast<const String*>(iter.key());
	    if( str ) 
		cout << *str << " ";
	}

	cout << " end \n";
	iter.reset();

	cout << "Testing that dictionary assignment operator worked" << endl;
	cout << "Keys and values should be the same as above" << endl;
	dict2 = dict;
	iter2.reset();

	cout << "values: ";
	while( (item = iter2()) != NULL )
	    cout << *item << " ";

	cout << " end \n";
	iter2.reset();

	cout << "  keys: ";
	while( (item = iter2()) != NULL ) {
	    const String* str = dynamic_cast<const String*>(iter2.key());
	    if( str ) 
		cout << *str << " ";
	}

	cout << " end \n";

	cout << "Size: " << dict.getSize() << endl;
	cout << "Overflow: " << dict.getBucketOverflow() << endl;
	cout << "Max Chain: " << dict.getMaxChain() << endl;

	RemoveAndDestroyContentsOf(&dict);

	cout << "after RemoveAndDestroyContentsOf size is: " << 
	     dict.getSize() << "\n";

	cout << " End of DictionaryOf testing\n\n";
    }
}

