//
// Association.java
//		
//      Represents relationships between monitored items.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.category;

import com.sgi.sysadm.util.Log;

/**
 * Association represents a dynamic collection of
 * child Item(s) related to a parent Item identified by a
 * selector. For example, the user account Item(s) that are related to
 * a group account Item with selector "bar" could be represented by an
 * Association.  An Association represents an 1-to-n relationship. A
 * 1-to-1 relationship is handled as a special case of 1-to-n
 * relationship.
 *
 * <P>
 * Association derives from Category and supports the same client
 * mechanisms for obtaining dynamic information about the Item(s) in an
 * Association instance. 
 *
 * <P>
 * If an Item with the specified parent selector does not exist, the
 * Association will monitor for the addition of an Item with that
 * selector.
 *
 * <P>
 * If an Item with the specified parent selector existed, but is
 * deleted while a client is monitoring the Association, all Items
 * are removed from the Association and the Association continues
 * monitoring of the system for the addition of an Item with that
 * parent selector.
 *
 * <P>
 * The actual logic for determining and monitoring Item(s) in an
 * Association resides on the server-side.  Clients can obtain a
 * Association instances using HostContext methods. 
 *
 * @see com.sgi.sysadm.ui.HostContext
 */
public abstract class Association extends Category {    

    private static final String ASSOC_PREPOSITION = "sAssocWith";
    private static final String ASSOC_SEPARATOR = "-";

    private String _parentCategoryName;
    private String _parentSelector;
    private String _childCategoryName;

    /**
     * Association constructor.
     * 
     * @param parentCategoryName The selector of the parent Category.
     * @param parentSelector The selector of the parent Item.
     * @param childCategoryName The selector of the child Category.
     */
    protected Association(String parentCategoryName,
			  String parentSelector,
			  String childCategoryName) {

	super(computeSelector(parentCategoryName, parentSelector, 
			      childCategoryName));
	_parentCategoryName = parentCategoryName;
	_parentSelector = parentSelector;
	_childCategoryName = childCategoryName;
    }

    /**
     * Compute the selector for this Association 
     * 
     * @param parentCategoryName The selector of the parent Category.
     * @param parentSelector The selector of the parent Item.   
     * @param childCategoryName The selector of the child Category.
     *
     * @return selector for this association
     */
    protected static String computeSelector(String parentCategoryName, 
					    String parentSelector,
					    String childCategoryName) {
	Log.assert((parentCategoryName != null && childCategoryName != null &&
		    parentSelector != null),
		   "Cannot pass null to computeSelector");
	return (truncate(childCategoryName) + ASSOC_PREPOSITION + 
		truncate(parentCategoryName) + 
		ASSOC_SEPARATOR + parentSelector);
    }

    /**
     * Get the selector of the parent item.
     * 
     * @return selector of the parent item.
     */
    public String getParentSelector() {
	return _parentSelector;
    }

   /**
     * Get selector of parent category.
     * 
     * @return selector of parent category.
     */
    public String getParentCategoryName() {
	return _parentCategoryName;
    }

   /**
     * Get selector of child category.
     * 
     * @return selector of child category.
     */
    public String getChildCategoryName() {
	return _childCategoryName;
    }

    /**
     * Removes the "Category" suffix from the selector of a Category.
     * 
     * @param catName Selector of Category.
     * 
     * @return Selector of Category with suffix "Category" removed.
     */
    private static String truncate(String catName) {
	int len = catName.length();
	int catLength = Category.CATEGORY.length();
	Log.assert(catName.endsWith(Category.CATEGORY), 
		   "Incorrect format of category name");
	return catName.substring(0, len - catLength);
    }


}
