//
// Category.java
//		
//	Represents a collection of monitored Item(s).
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.category;

import com.sgi.sysadm.util.*;
import java.util.*;

/**
 * Represents a collection of monitored Item(s) of a specific type.
 * For example, the collection of user account Item(s) can be
 * represented by a Category instance.  A Category is also  a subclass
 * of AttrBundle and can have a set of Attributes that apply to all
 * Item(s) of that type.
 *
 * <P>
 * Clients interested in information about a Category instance can
 * create an instance CategoryListener and register for
 * notifications by passing a CategoryListener instance  to
 * addCategoryListener().  The specific Items of interest are
 * indicated by the NotificationFilter paramater.  The
 * NotificationFilter also specifies whether  the CategoryListener
 * instance is interested in notifications about the Category
 * attributes. Call removeCategoryListener() to unregister interest in
 * notification.
 * 
 * <P>
 * Category notifies registered CategoryListener instances about an
 * Item discovered at startup that are later added via
 * CategoryListener.itemAdded() calls, Item changes via
 * CategoryListener.itemChanged() calls, and Item
 * removals via CategoryListener.itemRemoved().  When
 * addCategoryListener() is called, Category sends the listener
 * its current list of Item(s) via itemAdded() calls.  If the Category
 * has completed discovery of the existing Item(s) 
 * of the specific type in the system before the time the
 * CategoryListener registers for notification, it will notify
 * CategoryListener of all the Item(s) in its list and send the
 * endExists() notification.  If not, it will send the endExists()
 * notifications when it completes discovery of the existing Item(s).
 *
 * <P>
 * Category also calls CategoryListener methods beginBlockChanges() and
 * endBlockChanges() to indicate the start and end of a block of
 * notifications.
 * 
 * <p>
 * In order to get an Item corresponding to a specific selector
 * in a Category, you must pass a ResultListener to
 * <tt>getItem</tt>. <tt>getItem</tt> calls the <tt>succeeded</tt> method of 
 * the ResultListener, if an Item with the specified selector exists
 * in the system.  Use the <tt>getResult</tt> method of
 * ResultEvent to get the Item.  The Object returned by
 * <tt>getResult</tt> should be cast to an Item. <tt>getItem</tt>
 * calls the <tt>failed</tt> method of the ResultListener, if an
 * Item with the specified selector does not exists in the system.
 * For example:
 * <pre>
 * Category cat;
 * ...
 * cat.getItem("Test", new ResultListener() {
 *     public void succeeded(ResultEvent event) {
 *         Item item = (Item) event.getResult();
 *     }
 *     public void failed(ResultEvent event) {
 *         System.out.println("getItem failed.");
 *     }
 * });
 * </pre>
 *
 * The actual logic for determining and monitoring Item(s) in a
 * Category resides on the server-side.  Clients can obtain a
 * Category instances using HostContext methods. 
 *
 * @see com.sgi.sysadm.ui.HostContext
 * @see com.sgi.sysadm.category.NotificationFilter
 * @see com.sgi.sysadm.category.CategoryListener
 */
public abstract class Category extends AttrBundle {

    /**
     *  This is the suffix that all Category selectors must end with.
     */
    protected static final String CATEGORY = "Category";
    private Hashtable _items = new Hashtable();
    private Hashtable _listeners = new Hashtable();
    private Hashtable _addedListeners = new Hashtable();
    private Hashtable _itemExistsListeners = new Hashtable();
    private Vector _itemCountListeners = new Vector();
    private Vector _itemListListeners = new Vector();

    private boolean _monitoring = false;
    private boolean _inNotification = false;
    private boolean _inBlockChangesNotification = false;
    private boolean _existsEnded = false;

    // Elements of the collection matching listeners to filters.
    private class ListenerMap {
	
	public ListenerMap(CategoryListener listener,
			   NotificationFilter filter) {
	    _listener = listener;
	    _filter = filter;
	}
	
	private CategoryListener _listener;
	private NotificationFilter _filter;
	private boolean _removed = false;
	private boolean _isAttrListener = false;
    }

    private interface Notifier {
	public void notify(ListenerMap lMap);
    }

    /**
     * Called by clients to add a CategoryListener to the list of
     * objects which will receive types of notifications specified.
     *
     * @param listener The object to call when something happens.
     *		       
     * @param filter Specified the notifications of interest to
     *               listener object.  Category makes a copy of "filter"
     *               and any changes to "filter" after this call is made
     *		     has no effect on the filter that Category
     *		     associates with "listener".
     */
    public void addCategoryListener(CategoryListener listener,
				    NotificationFilter filter) {
	if (Log.isLevelOn(Log.DEBUG)) {
	    // String concats are expensive - skip it
	    // if it's not going to be used.
	    Log.debug(CATEGORY, getSelector() + ": addCategoryListener");
	}
	
	Log.assert((listener != null && filter != null),
		   "Invalid arguments to addCategoryListener");

	Log.assert(!_addedListeners.containsKey(listener), 
		   "Listener was previously associated with a filter");

	// Check if "listener" was previously associated with a filter
	//
	if (_listeners.containsKey(listener)) {
	    ListenerMap lMap = (ListenerMap)
		_listeners.get(listener);
	    Log.assert(lMap != null && lMap._removed,
		       "Listener was previously associated with a filter");
	}

	NotificationFilter filterClone = (NotificationFilter) filter.clone();
	ListenerMap lMap = new ListenerMap(listener, filterClone);

	if (_inNotification) {

	    // If we're in the middle of notification, we cannot add to our
	    // master list as doing so might not give the accurate
	    // picture of the category to the new listener.  
	    // So we'll hold on the the new listener in
	    // _addedListeners and add it when we're done notifying.	    
	    _addedListeners.put(listener, lMap);

	} else {
	    _listeners.put(listener, lMap);
	    notifyCurrentState(lMap);
	}
    }

    /**
     * Convenience method to specify that the CategoryListener object
     * is interested in receiving all notifications.
     *
     * @param listener The object to call when something happens.
     */
    public void addCategoryListener(CategoryListener listener) {
	addCategoryListener(listener, NotificationFilter.ALL_ITEMS);
    }

    /**
     * Remove a listener from the list of objects that will receive
     * notifications.
     * 
     * @param listener listener to remove.
     *
     */
    public void removeCategoryListener(CategoryListener listener) {
	if (Log.isLevelOn(Log.DEBUG)) {
	    // String concats are expensive - skip it
	    // if it's not going to be used.
	    Log.debug(CATEGORY,  getSelector() + ": removeCategoryListener");
	}
	if (_inNotification) {
	    Log.debug(CATEGORY, "inNotification");

	    // Could be in _addedListeners list or _listeners list
	    // If the listener is in _addedListeners list then it was
	    // never placed in the master list (_listeners), so we can
	    // remove it even if we in in the middle of notification

	    // Check if it exists in _addedListeners list
	    if (_addedListeners.containsKey(listener)) {
		 Log.debug(CATEGORY, "addedListeners");
		_addedListeners.remove(listener);
	    } else {		
		ListenerMap lMap = (ListenerMap)
		    _listeners.get(listener);

		// Set bool to indicate that this listener is not
		// longer interested in notifications and remove it
		// when we're done notifying.
		if (lMap != null) {
		    Log.debug(CATEGORY, "Setting removed");
		    lMap._removed = true;
		    // Check for lMap._filter.monitorsCategoryAttrs()
		    // is insufficient as this could be true,
		    // but it was never added as an AttrListener
		    // because of a removeCategoryListener call
		    // during notifyCurrentState()
		    if (lMap._isAttrListener) {
			super.removeAttrListener(listener);
		    }
		} else {
		    Log.debug(CATEGORY, "lMap == null");
		}
	    }
		
	} else {
	    ListenerMap lMap =  (ListenerMap) _listeners.remove(listener);

	    // Checking for 1. lMap._filter.monitorsCategoryAttrs()
	    // would have been sufficient (Vs. 2. lMap._isAttrListener)
	    // as 1 implies 2 if we are done with
	    // notifyCurrentState()
	    if (lMap != null && lMap._isAttrListener) {
		super.removeAttrListener(lMap._listener);
	    }
	}
    }

    /**
     * Category constructor.
     *
     * @param selector The string representing the type of the Item that
     *                 this Category monitors. For example, the
     *                 Category instance representing user accounts
     *                 could have selector: "UserAccountCategory".
     *                 This should be the same as the name of the
     *                 library representing the Category on the server
     *                 side which resides in /usr/sysadm/category
     */
    protected Category(String selector) {
	super(CATEGORY, selector);
    }

    /**
     * Get the Item corresponding to a selector.
     * 
     * @param selector Selector of an Item.
     * @param listener Gets notified of the result of this
     *                 operation.  <tt>listener.succeeded</tt> is
     *                 called on succcess.  <tt>listener.failed</tt>
     *                 is called on failure.  The 
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get the Item corresponding to
     *                 "selector". The Object returned should be
     *                 cast to an Item.  
     */
    public synchronized void getItem(String selector, 
				     ResultListener listener) {
	if (!_monitoring) {
	    _monitoring = true;
	    startMonitor();
	}

	Item item = (Item) _items.get(selector);
	if (item != null) {
	    Log.debug(CATEGORY, "Item exists");
	    ResultEvent event = new ResultEvent(this, item);
	    listener.succeeded(event);
	} else {
	    // Check if endExists has been called
	    if (_existsEnded) {
		Log.debug(CATEGORY, "ExistsEnded - Item does not exist");
		ResultEvent event = new ResultEvent(this);
		listener.failed(event);
	    } else {
		Log.debug(CATEGORY, "ExistsEnded not true - Waiting for endExists");
		Vector itemListeners = 
		    (Vector) _itemExistsListeners.get(selector);
		if (itemListeners == null) {
		    itemListeners = new Vector();
		    _itemExistsListeners.put(selector, itemListeners);
		}
		itemListeners.addElement(listener);
	    }
	}
    }

    /**
     * Gets the number of Items in this Category.
     *
     * @param listener <tt>listener.succeeded</tt> is
     *                 called upon completion of the operation.
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get the number of Items in this
     *                 Category. The Object returned should be 
     *                 cast to an Integer.  
     */
    public synchronized void getItemCount(ResultListener listener) {
	if (!_monitoring) {
	    _monitoring = true;
	    startMonitor();
	}
	
	// Check if endExists has been called
	if (_existsEnded) {
	    Log.debug(CATEGORY, 
		      "ExistsEnded - Returning count: " + _items.size());
	    ResultEvent event = 
		new ResultEvent(this, new Integer(_items.size()));
	    listener.succeeded(event);
	} else {
	    Log.debug(CATEGORY, 
		      "ExistsEnded not true - Waiting for endExists");
	    _itemCountListeners.addElement(listener);
	}
    }

    /**
     * Gets the list of Items in this Category.
     *
     * @param listener  <tt>listener.succeeded</tt> is
     *                 called upon completion of the operation.
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get the list of Items in this
     *                 Category. The Object returned should be cast to
     *                 a Hashtable.  The elements of the Hashtable
     *                 are Items, with selectors used as the Hashtable key.
     */
    public synchronized void getItemList(ResultListener listener) {
	if (!_monitoring) {
	    _monitoring = true;
	    startMonitor();
	}

        // Check if endExists has been called
	if (_existsEnded) {
	     ResultEvent event = new ResultEvent(this, _items);
	     listener.succeeded(event);
	} else {
	    _itemListListeners.addElement(listener);
	}
    }


    /**
     * Get an Enumeration of the Items in this Category.
     * 
     * @return Enumeration of Items in this Category.
     */
    protected synchronized Enumeration getItemEnum() {
	return _items.elements();
    } 
    
    /**
     * Called by Category when the first CategoryListener is added to
     * start monitoring the system.  Subclasses should do whatever is
     * necessary to start monitoring the system within this method.
     * None of addItem(), changeItem(), orphanItem() or removeItem()
     * should be called prior to the call to startMonitor().
     */
    protected abstract void startMonitor();

    /**
     * Called by subclasses when a new Item is discovered at startup
     * or when an Item is added.
     * Add the Item to our list and inform listeners.
     *
     * @param item The Item that was added.
     */
    protected synchronized void addItem(Item item) {

	Log.assert((item != null),
		   "Invalid argument to addItem");

	Item itemClone = (Item) item.clone();
	String selector = itemClone.getSelector();
	_items.put(selector, itemClone);
	notifyAdded(itemClone);

	if (!_existsEnded) {
	    Vector itemListeners = (Vector)
	        _itemExistsListeners.remove(selector);
	    if (itemListeners != null) {
		Log.debug(CATEGORY, "addItem - item exists");
		Enumeration enum = itemListeners.elements();
		ResultEvent event = new ResultEvent(this, itemClone);

		while (enum.hasMoreElements()) {	
		    ((ResultListener) enum.nextElement()).succeeded(event);
		}
	    }
	} else {
	    Log.assert(_itemExistsListeners.size() == 0, 
		       "There are listeners in itemExistsListeners");
	}
	
    }

    
    /**
     * Called by subclasses to check if the Category 
     * has started monitoring the system.
     *
     * @return true if the Category is monitoring the system, 
     *         false otherwise.
     */
    protected boolean isMonitoring() {
	return _monitoring;
    }

    /**
     * Called by subclasses to check if the Category 
     * is in the middle of a block changes notification.
     *
     * @return true if the Category is in the middle of a block
     *         changes notification, false otherwise.
     */
    protected boolean isInBlockChangesNotification() {
	return _inBlockChangesNotification;
    }    

    /**
     * Called by subclasses to check if the Category 
     * has received an endExists notification.
     *
     * @return true if the Category has received an endExists
     * notification, false otherwise.
     */
    protected boolean hasExistsEnded() {
	return _existsEnded;
    }

    /**
     * Called by subclasses when an item in the system changed.
     * Change the corresponding item in our list and inform listeners.
     *
     * @param item The item that changed.
     */
    protected synchronized void changeItem(Item item) {

	Log.assert((item != null),
		   "Invalid argument to replaceChangedItem");

	String selector = item.getSelector();
	Log.assert(_items.containsKey(selector), 
		   "Item to change not found");

	Item itemClone = (Item) item.clone();
	Item oldItem = (Item) _items.put(selector, itemClone);
	notifyChanged(oldItem, itemClone);
	    
    }

    /**
     * Called by subclasses when an item is removed from the
     * system.  Remove the item from our list and inform listeners
     *
     * @param  selector selector of the item that was removed.
     * 
     * @return The object that was removed from the list of items
     */
    protected synchronized Item removeItem(String selector) {
	Item removedItem =  (Item) _items.remove(selector);
	Log.assert(removedItem != null, "Item to remove not found");
	
	notifyRemoved(removedItem);
	return removedItem;
    }

    /**
     * Replace the current list of Item(s) by a new list.  The
     * Category base class computes any changes between its previous
     * list and the new "list" and notifies interested listeners of
     * any changes. "list" becomes the new list of Category items.
     * 
     * @param list New list of Item(s).
     */
    protected synchronized void replaceItemList(Hashtable list) {

	beginBlockChanges();

	Enumeration enum = list.elements();
	Item item;
	
	if (_items.size() == 0) {
	    
	    while (enum.hasMoreElements()) {
		item = (Item) enum.nextElement();
		addItem(item);
	    }

	} else {

	    Hashtable oldList = _items;
	    _items = new Hashtable();

	    while (enum.hasMoreElements()) {

		item = (Item) enum.nextElement();
		String selector = item.getSelector();
		Item oldItem = (Item) oldList.remove(selector);
		if (oldItem != null) {
		    // Determine whether the item has actually changed.
		    if (!oldItem.equals(item)) {
			Item itemClone = (Item) item.clone();
			_items.put(selector, itemClone);
			notifyChanged(oldItem, itemClone);
		    } else {
			// Put the old item back in the list since it
			// hasn't changed.
			_items.put(selector, oldItem);
		    }
		} else {
		    addItem(item);
		}
	    }

	    // Traverse thru oldList for all not accounted for items
	    enum = oldList.elements();
	    while (enum.hasMoreElements()) {
		// notifyRemoved
		item = (Item) enum.nextElement();
		notifyRemoved(item);
	    }
	}
	
	endBlockChanges();
    }

    /**
     * Begin a change block.  CategoryListeners can defer doing
     * expensive stuff until the change block ends.
     */
    protected synchronized void beginBlockChanges() { 

	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		lMap._listener.beginBlockChanges();
	    }
	});

    }

    /**
     * End a change block.
     */
    protected synchronized void endBlockChanges() {
	
	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		lMap._listener.endBlockChanges();
	    }	
	});
    
    }    

    /**
     * Called by subclasses after it has called addItem() for every
     * Item that existed in the system when startMonitor() is
     * called. Category informs listeners.
     */
    protected synchronized void endExists() {
	
	Log.assert(!_existsEnded, "endExists already called");
	_existsEnded = true;

	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		lMap._listener.endExists();
	    }	
	});

	if (_itemExistsListeners.size() > 0) {
	    Log.debug(CATEGORY, "endExists - item(s) does not exist");
	    ResultEvent event = new ResultEvent(this);
	    Enumeration listenerVectorEnum = 
	        _itemExistsListeners.elements();
	    Enumeration listenerEnum;

	    while (listenerVectorEnum.hasMoreElements()) {
		listenerEnum = ((Vector) 	
				listenerVectorEnum.nextElement()).elements();
		
		while (listenerEnum.hasMoreElements()) {
		    ((ResultListener)
		     listenerEnum.nextElement()).failed(event);
		}
	    }
	    _itemExistsListeners.clear();
	}

	if (_itemCountListeners.size() > 0) {
	     ResultEvent event = 
		 new ResultEvent(this, new Integer(_items.size()));
	     Enumeration icListenerEnum = 
		 _itemCountListeners.elements();
	     while (icListenerEnum.hasMoreElements()) {
		 ((ResultListener)
		  icListenerEnum.nextElement()).succeeded(event);
	     }
	     _itemCountListeners.removeAllElements();
	}

	if (_itemListListeners.size() > 0) {
	     ResultEvent event = new ResultEvent(this, _items);
	     Enumeration ilListenerEnum = 
		 _itemListListeners.elements();
	     while (ilListenerEnum.hasMoreElements()) {
		 ((ResultListener)
		  ilListenerEnum.nextElement()).succeeded(event);
	     }
	     _itemListListeners.removeAllElements();
	}
    }    

    /**
     * Make "listener" up to date on everything that's happened so far
     * by telling it about each existing item and category attribute.
     * Also register listener as a AttrListener if indicated by "filter".
     *
     * @param listener listener to get up to date.
     * @param filter filter specifying data that the listener is
     *               interested in
     */
    private void notifyCurrentState(ListenerMap lMap) {

	CategoryListener listener = lMap._listener;
	NotificationFilter filter = lMap._filter;
	startNotification();

	// If this is the first listener and getItem has not been called
	// _items and attrsEnum should be empty
	if (!_monitoring) {
	    Enumeration attrsEnum = getAttrEnum();
	    Log.assert(_items.size() == 0 && !attrsEnum.hasMoreElements(),
		       "List populated before call to startMonitor");
	}

	// Handle case where listener is added in the middle
	// of block changes notification
	if (_inBlockChangesNotification && !lMap._removed) {
	    listener.beginBlockChanges();
	}

	Enumeration enum = _items.elements();
	while (enum.hasMoreElements()) {

	    Item item = (Item) enum.nextElement();
	    if (filter.monitorsItem(item.getSelector()) && !lMap._removed) {
		listener.itemAdded(item);	
	    }
	}	

	if (filter.monitorsCategoryAttrs()) {

	    // Send category attributes
	    Enumeration attrEnum = super.getAttrEnum();

	    while (attrEnum.hasMoreElements() && !lMap._removed) {
		AttrEvent event = 
		    new AttrEvent(this, (Attribute) attrEnum.nextElement());
		listener.attrAdded(event);
	    }

	    if (!lMap._removed) {
		lMap._isAttrListener = true;
		// Inform future changes to category attributes
		super.addAttrListener(listener);
	    }

	}

	if (_existsEnded && !lMap._removed) {
	    listener.endExists();
	}
	
	endNotification();

	// If this is the first listener we've gotten, and getItem
	// has not been called, tell our subclass to
	// start monitoring.  Note if this is the first listener,
	// and getItem has not been called _items and attrsEnum should be empty
	if (!_monitoring) {
	    _monitoring = true;
	    startMonitor();
	}
    }

    /**
     * Notify listeners that item was added.
     * 
     * @param item The item that was added.
     */
    private void notifyAdded(final Item item) {

	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		if (lMap._filter.monitorsItem(item.getSelector())) {
		    lMap._listener.itemAdded(item);
		}
	    }
	});

    }

    /**
     * Notify listeners that item was removed.
     * 
     * @param item The item that was removed.
     */
    private void notifyRemoved(final Item item) {

	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		if (lMap._filter.monitorsItem(item.getSelector())) {
		    lMap._listener.itemRemoved(item);
		}
	    }
	});
    }	

    /**
     * Notify listeners that "oldItem" changed.
     * 
     * @param oldItem  The old item.
     *
     * @param newItem  The new item.
     */
    private void notifyChanged(final Item oldItem, 
			       final Item newItem) {

	notify(new Notifier() {
	    public void notify(ListenerMap lMap) {
		if (lMap._filter.monitorsItem(newItem.getSelector())) {
		    lMap._listener.itemChanged(oldItem, newItem);
		}
	    }
	});
	
    }
    
    /**
     * Start notifying.
     */
    private void startNotification() {
	Log.assert(!_inNotification, 
		   "startNotification:inNotification == true!");
	_inNotification = true;
    }

    /**
     * End notifying.
     */
    private void endNotification() {

	_inNotification = false;

	// Actually remove any listeners for which removeCategoryListener was
	// called during notification.
	Hashtable listeners = (Hashtable) _listeners.clone();
	Enumeration enum = listeners.elements();
	while (enum.hasMoreElements()) {
	    ListenerMap lMap = (ListenerMap) enum.nextElement();
	    if (lMap._removed) {
		_listeners.remove(lMap._listener);
	    }
	}

	// Really add any listeners for which addCategoryListener was called
	// during notification.
	if (_addedListeners.size() > 0) {
	    // Make a copy of _addedListeners, because we're going to have
	    // to do more notification to let the new listeners know what's
	    // going on, and one of the new listeners might decide to add
	    // another listener.
	    Hashtable addedListeners = (Hashtable) _addedListeners.clone();
	    _addedListeners.clear();
	    
	    enum = addedListeners.elements();
	    while (enum.hasMoreElements()) {
		 ListenerMap lMap = (ListenerMap) enum.nextElement();
		_listeners.put(lMap._listener, lMap);
	    }

	    enum = addedListeners.elements();
	    while (enum.hasMoreElements()) {
		ListenerMap lMap = (ListenerMap) enum.nextElement();
		if (lMap._removed) {
		    continue;
		}
		notifyCurrentState(lMap);
	    }
	}
    }
    
    private void notify(Notifier notifier) {
	startNotification();

	Enumeration enum = _listeners.elements();
	while (enum.hasMoreElements()) {

	    ListenerMap lMap = (ListenerMap) enum.nextElement();	    
	    if (lMap._removed) {
		continue;
	    }
	    
	    notifier.notify(lMap);

	}
	endNotification();
    }

}
