//
// CategoryListener.java
//		
//	Class for monitoring changes in a Category.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.category;

import com.sgi.sysadm.util.*;

/**
 * An instance of an implementation of CategoryListener receives
 * notifications of either the entire list or a subset of Item(s) in
 * a Category instance.  Furthermore, it can receive 
 * notifications of changes to the attributes of the Category
 * instance.
 * 
 * <P>
 * A CategoryListener can expect to receive zero or more
 * <TT>itemAdded()</TT> and <TT>attrAdded()</TT> 
 * calls, followed by an <TT>endExists()</TT> call followed by zero or
 * more <TT>addItem()</TT>, <TT>changeItem()</TT>,
 * <TT>removeItem()</TT>, <TT>attrAdded()</TT>, <TT>attrChanged</TT> and
 * <TT>attrRemoved</TT> calls.   <TT>The endExists()</TT> call signals
 * that the Category has communicated the entire set of
 * Item(s) discovered in the system to the CategoryListener. 
 * 
 * @see com.sgi.sysadm.category.Category
 */

public interface CategoryListener extends AttrListener {
    
    /**
     * Called by Category to notify current list of Item(s) at the time
     * CategoryListener registers for notifications and when Item(s) are
     * later discovered by Category or added to the system.
     * 
     * @param item The Item that existed in the list of Item(s) in
     *             Category when CategoryListener registers
     * 	           for notifications or that is later added.
     */
    public void itemAdded(Item item);

    /**
     * Called by Category when an Item changes.
     * 
     * @param oldItem The previous state of the Item.
     * @param newItem The current state of the Item.
     */
    public void itemChanged(Item oldItem, Item newItem);

    /**
     * Called by Category when an Item is removed.
     * 
     * @param item The Item that was removed from the list of Item(s).
     */
    public void itemRemoved(Item item);

    /**
     * Called by Category prior to a block of
     * of changes.  
     */
    public void beginBlockChanges();
    
    /**
     * Called by Category after a block of changes.
     */
    public void endBlockChanges();

    /**
     * Called by Category when it has finished notifying
     * CategoryListener of all Item(s) of a specific type in a system.
     * If the Category has completed discovery of the existing Item(s)
     * of the specific type in the system before the time the
     * CategoryListener registers for notification, it will notify
     * CategoryListener of all the Item(s) in its list and send the
     * endExists() notification.  If not, it will send the endExists()
     * notifications when it completes discovery of the existing Item(s).
     */
    public void endExists();

}
