//
// Item.java
//		
//	A monitored object within a Category.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.category;

import com.sgi.sysadm.util.*;
import java.util.*;

/**
 * Represents a physical or logical entity that is manipulated by
 * system administration operations. An Item is a subclass of
 * AttrBundle and has an aggregation of typed key value pairs.  Each
 * Item has an associated type and a unique name within that type. For
 * example, a user account can have the unique name "foo" within the
 * type "user account" and the following Attributes:
 * <pre>
 *	string, userName, foo
 *	long, userId, 3944
 * </pre>
 * <P>Item(s) of a specific type are aggregated in an instance of a
 * subclass of Category. The type of the Item corresponds to the
 * selector of the Category.  For example, the collection of user
 * account Item(s) can be represented by a Category instance. 
 * Category uses Item to inform CategoryListeners of
 * the current state of the system and of any changes.
 */

public class Item extends AttrBundle {

    /**
     * Construct an Item with a unique name in the context of a
     * specific Category.
     * 
     * @param type The String representing the selector of the
     *             Category instance to which this Item belongs.
     * @param selector Unique identifier within category for Item.
     */
    public Item(String type, String selector) {
	super(type, selector);
    }

    /**
     * Construct an Item from a serialized format.
     * 
     * @param stream Serialized format of Item, obtained via
     *               serialize();
     */
    public Item(String stream) {
	super(stream);
    }

    /**
     * Construct an Item that is a copy of another Item. 
     * 
     * @param other The Item to copy.
     */
    public Item(Item other) {
	super(other);
    }

    /**
     * Construct an Item that based on an AttrBundle.
     * 
     * @param attrs The Attributes for Item.
     */
    public Item(AttrBundle attrs) {
	super(attrs);
    }
    
    /**
     * Get an object that is a copy of this one.
     * 
     * @return Item that has a copy of this.
     */
    public Object clone() {
	return new Item(this);
    }

    /**
     * Find differences between this and another Item.
     * 
     * @param other Item to compare
     * 
     * @return Attributes in "other" that are different from
     *	       attributes in this.
     */
    public Vector getChanges(Item other) {

	Vector changes = new Vector();

	Log.assert(super.getType().equals(other.getType()),
		   "Cannot getChanges between items of different types");
	Log.assert(super.getSelector().equals(other.getSelector()),
		   "Cannot getChanges between items with different selector");

	Log.assert(other.size() == super.size(),
		   "Cannot getChanges betwn items with diff number of attrs");

	Enumeration newAttrs = other.getAttrEnum();
	while (newAttrs.hasMoreElements()) {
	    Attribute newAttr = (Attribute) newAttrs.nextElement();
	    Attribute oldAttr = super.getAttr(newAttr.getKey());
	    Log.assert(oldAttr != null, "Comparing items with different keys");
	    if (!oldAttr.equals(newAttr)) {
		changes.addElement(newAttr);
	    }
	}
	
	return changes;
    }
}
