//
// NotificationFilter.java
//		
//	Class representing the changes that a CategoryListener is monitoring
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.category;

import java.util.*;
import com.sgi.sysadm.util.*;

/**
 * Class representing the changes that a CategoryListener is
 * monitoring.  A NotificationFilter object specifies either an entire
 * Category or a subset of Items within a Category to be monitored. It
 * can also specify monitoring of changes of the attributes of the
 * Category.
 *
 * @see com.sgi.sysadm.category.CategoryListener
 */
public class NotificationFilter {

    /**
     * NotificationFilter specifying interest in all Item(s)
     * of a Category.  This does not specify interest in Category
     * Attributes.
     */
    public static NotificationFilter ALL_ITEMS
        = new NotificationFilter();

    private boolean _allItems = false;
    private boolean _allCategoryAttrs = false;

    private Hashtable _selectorList = new Hashtable();

    static {
	ALL_ITEMS.monitorAllItems();
    }

    /**
     * NotificationFilter constructor.
     */
    public NotificationFilter() {
    }

    
    /**
     * Copy the value of this object
     * 
     * @return NotificationFilter that has a copy of this NotificationFilter
     */
    public Object clone() {

	NotificationFilter filter = new NotificationFilter();
	filter._allItems = _allItems;
	filter._allCategoryAttrs = _allCategoryAttrs;
	filter._selectorList = (Hashtable) _selectorList.clone();
	return filter;

    }

    /**
     * Register interest in Item 
     *
     * @param selector Name of Item
     */
    public void monitorItem(String selector) {
	Log.assert(selector != null, "Invalid argument to monitorItem");
	_allItems = false;

	String selectorClone = new String(selector);
	_selectorList.put(selectorClone, selectorClone);
    }

    /**
     * Unregister interest in Item 
     * 
     * @param selector Name of Item
     */
    public void unmonitorItem(String selector) {
	Log.assert(!_allItems, "All Items are currently being monitored");
	_selectorList.remove(selector);
    }

    /**
     * Register interest in all Items in Category.
     */
    public void monitorAllItems() {
	_allItems = true;

	// Clear _selectorList
	_selectorList.clear();
	
    }

    /**
     * Unregister interest in all Items in Category.
     */
    public void unmonitorAllItems() {
	_allItems = false;

	// Clear _selectorList
	_selectorList.clear();

    }

    /**
     * Register interest in all Category attributes.
     */
    public void monitorCategoryAttrs() {
	_allCategoryAttrs = true;
    }

    /**
     * Unregister interest in all Category attributes.
     */
    public void unmonitorCategoryAttrs() {
	_allCategoryAttrs = false;
    }

    /**
     * Determine if the filter monitors specified item
     * 
     * @param selector Name of item
     * 
     * @return true if filter monitors selector, false otherwise
     */
    public boolean monitorsItem(String selector) {
	if (_allItems)
	    return true;
	return _selectorList.containsKey(selector);
    }
    
    /**
     * Determine if the filter monitors category attributes
     * 
     * @return true if filter monitors category, false otherwise
     */
    public boolean monitorsCategoryAttrs() {
	return _allCategoryAttrs;
    }
}
