//
// Authenticator.java
//
//	Base class for authentication schemes.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.comm;

import java.util.*;
import com.sgi.sysadm.util.*;

/**
 * Authenticator is an interface for authenticating a user over a
 * Connection.  Authenticator subclasses are created by subclasses to
 * use a specific authentication scheme.  The primary users of
 * Authenticator methods and the AuthListener interface are user
 * interface classes that prompt the user for authentication
 * parameters and ServerConnectionFactory.
 */
public abstract class Authenticator {

    private Vector _listeners = new Vector();
    private Connection _conn;
    int _maxRetries = -1;
    int _retries = 0;

    /**
     * Set the number of times that we will retry authentication
     * before failing.  If maxRetries is -1 (the default), we will
     * retry forever.
     * 
     * @param maxRetries 
     */
    public void setMaxRetries(int maxRetries) {
	_maxRetries = maxRetries;
    }

    /**
     * Add a listener to get notified when AuthEvents occur.
     * 
     * @param listener 
     */
    public void addAuthListener(AuthListener listener) {
 	_listeners.addElement(listener);
    }

    /**
     * Remove a listener.
     * 
     * @param listener 
     */
    public void removeListener(AuthListener listener) {
 	_listeners.removeElement(listener);
    }
     
    /**
     * Gets the connection which will be used for communicating
     * authentication parameters.
     * 
     * @returns The Connection to be used for communicating
     *             authentication parameters.
     */
    public Connection getConnection() {
	return _conn;
    }

    /**
     * Set the connection which will be used for communicating
     * authentication parameters.
     * 
     * @param conn Connection to be used for communicating
     *             authentication parameters.
     */
    public void setConnection(Connection conn) {
	_conn = conn;
    }

    /**
     * Subclasses override this method to authenticate us.
     * Authentication parameters should be exchanged with our peer.
     * The various notify methods below should be used to provide the
     * result of the attempt to authenticate.
     */
    public abstract void authenticate();

    /**
     * Notify clients that authentication has succeeded.
     * 
     * @param event 
     */
    protected void notifySucceeded(AuthEvent event) {
	_conn.setAuthenticated(true);
	// Reset retries each time we succeed.
	_retries = 0;
 	Enumeration enum = _listeners.elements();
 	while (enum.hasMoreElements()) {
 	    ((AuthListener)enum.nextElement()).authSucceeded(event);
 	}
    }

    /**
     * Notify clients that authentication has failed, and that no
     * recovery is possible.
     * 
     * @param event 
     */
    protected void notifyFailed() {
	_conn.notifyFatalError("Authentication failed");
    }

    /**
     * Notify clients that authentication has failed, but that the
     * user should try again.  On the client side, this means that the
     * user should be prompted to re-enter authentication parameters.
     * 
     * @param event 
     */
    protected void notifyRetry(AuthEvent event) {
	if (_maxRetries != -1 && ++_retries > _maxRetries) {
	    notifyFailed();
	    return;
	}
 	Enumeration enum = _listeners.elements();
 	while (enum.hasMoreElements()) {
 	    ((AuthListener)enum.nextElement()).authRetry(event);
 	}
    }	
}
