//
// ByteStreamConnection.java
//
//	Connection that sends and receives packets by turning them
//	into byte streams.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.comm;

import java.io.*;
import java.util.*;
import com.sgi.sysadm.util.*;

/**
 * ByteStreamConnection provides an implementation of the Connection
 * class that sends packets by turning them into byte streams that are
 * written to an OutputStream and receives packets by reading bytes
 * from an InputStream.  Any non-attribute data associated with a
 * packet is written to/read from the stream directly after the
 * packet attributes (see Packet).
 */
public abstract class ByteStreamConnection extends Connection {

    private InputStream _in;
    private OutputStream _out;
    private Thread _receiveThread = null;

    private static final String PACKET = "packet";
    private static final String DATA_LENGTH = "dataLength";
    private static final String BYTE_STREAM_CONN = "ByteStreamConnection";

    /**
     * ByteStreamConnection contructor.
     */
    public ByteStreamConnection() {
    }

    /**
     * Set the InputStream from which packets will be read.  This
     * method starts a new thread that reads the input stream.
     * 
     * @param in InputStream from which packets are to be read.
     */
    public void setInputStream(InputStream in) {
	// Use a BufferedInputStream so we can read one byte at a time
	// without making a system call for each one.
	_in = new BufferedInputStream(in);
	startReceiveThread();
    }

    /**
     * Set the OutputStream on which packets will be written.
     * 
     * @param out OutputStream on which packets are to be written.
     */
    public void setOutputStream(OutputStream out) {
	_out = out;
    }

    /**
     * Send <TT>packet</TT> to our peer.
     * 
     * @param packet The packet to send.
     */
    protected synchronized void sendPacketImpl(Packet packet) {
	// We send a serialized AttrBundle that has two attributes:
	// "packet" is the packet attributes and "dataLength" is the
	// amount of additional data.
	AttrBundle attrs = new AttrBundle("", "");
	attrs.setAttr(new Attribute(PACKET, packet));
	int dataLength = packet.getDataLength();
	attrs.setAttr(new Attribute(DATA_LENGTH, dataLength));
	if (Log.isLevelOn(Log.DEBUG)) {
	    // Calling packet.toString is really expensive - skip it
	    // if it's not going to be used.
	    Log.trace(BYTE_STREAM_CONN,
		      "Sending packet: " +
		      packet.toString(PACKET_LOG_PAD));
	}
	
	try {
	    _out.write((attrs.serialize() + "\n").getBytes());
	    if (dataLength > 0) {
		_out.write(packet.getData(), 0, dataLength);
	    }
	    _out.flush();
	} catch (IOException e) {
	    Log.debug(BYTE_STREAM_CONN, "IOException in sendPacket: "
		      + e.getMessage());
	    Log.debug(BYTE_STREAM_CONN, SysUtil.stackTraceToString(e));
	    // Don't do error notification if the connection is
	    // closed.  This can happen from the Applet as various
	    // things clean up while being garbage collected after
	    // we've already closed our connection.
	    if (_receiveThread != null) {
		Log.debug(BYTE_STREAM_CONN,
			  "Disrupt detected in sendPacketImpl.");
		_receiveThread.stop();
		_receiveThread = null;
		invokeLater(new Runnable() {
		    public void run() {
			notifyDisrupted();
		    }
		});
	    } else {
		Log.warning(BYTE_STREAM_CONN,
			    "Attempt to send packet on closed connection");
	    }
	}
    }
    
    /**
     * Close this connection.
     */
    public synchronized void close() {
	if (_receiveThread != null) {
	    _receiveThread.stop();
	    _receiveThread = null;
	}
    }

    // Start the thread that receives input.  This thread reads
    // packets from _in and distributes them via distributePacket().
    private void startReceiveThread() {
	// Stop thread if it's already running.
	if (_receiveThread != null) {
	    _receiveThread.stop();
	}
	_receiveThread = new Thread(new Runnable() {
	    public void run() {
		try {
		    handleInput();
		} catch (IOException e) {
		    // This has to be synchronized so that close() and
		    // sendPacketImp() can refer to _receiveThread
		    // after checking that it's not null.
		    synchronized (ByteStreamConnection.this) {
			Log.debug(BYTE_STREAM_CONN,
				  "Disrupt detected in I/O thread.");
			// This assert is important; it prevents us
			// from calling notifyDisrupted() twice on the
			// same disrupted connection.
			Log.assert(_receiveThread != null,
				   "_receiveThread is null, we have " +
				   "the lock, yet somehow we're " +
				   "still running");
			_receiveThread = null;
			invokeLater(new Runnable() {
			    public void run() {
				notifyDisrupted();
			    }
			});
		    }
		} catch (Exception e) {
		    // We're catching other exceptions to force us to
		    // exit, rather than just sit there crippled doing
		    // nothing.  Note that this doesn't actually work
		    // right until we're logged in.
		    e.printStackTrace();
		    Log.fatal(e.getMessage());
		}
	    }
	}, "Connection I/O");
	_receiveThread.setDaemon(true);
	_receiveThread.start();
    }

    // This is the main loop for the receive thread.  read and
    // distribute packets forever.
    private void handleInput() throws IOException {
	while (true) {
	    String stream = SysUtil.readLine(_in);
	    AttrBundle attrs = new AttrBundle(stream);
	    final Packet packet;
	    try {
		packet = new Packet(attrs.getAttr(PACKET).bundleValue());
	    } catch (NullPointerException ex) {
		throw new IOException();
	    }
	    long dataLength = attrs.getAttr(DATA_LENGTH).longValue();
	    if (dataLength > 0) {
		byte[] data = new byte[(int)dataLength];
		readFully(data, 0, (int)dataLength);
		packet.setData(data, (int)dataLength);
	    }
	    distributePacket(packet);
	}
    }

    /**
     * Read exactly <TT>len</TT> bytes from our input stream.  Do this
     * in multiple calls to read() if necessary.
     * 
     * @param buf Buffer to read into
     * @param off offset into buf at which to start copying bytes
     * @param len Number of bytes to copy
     * 
     * @exception IOException if the input stream ends.
     */
    private void readFully(byte[] buf, int off, int len) throws IOException {
	while (len > 0) {
	    int n = _in.read(buf, off, len);
	    if (n == -1) {
		throw new IOException("End of input stream");
	    }
	    off += n;
	    len -= n;
	}
    }
}
