//
// Connection.java
//
//	Java implementation of Rhino communication API.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.comm;
import javax.swing.SwingUtilities;

import java.util.*;
import java.io.*;
import com.sgi.sysadm.util.*;

/**
 * Connection is a channel on which Packet objects can be sent and
 * received.  In order to receive a packet, a client must register a
 * PacketListener, specifying the type of packet to receive.
 * <p>
 * Connection is a peer-to-peer channel.  Any client/server semantics
 * come from the way the Connection class is used.
 * <p>
 * Connection is the API which is used to communicate across the
 * network.  Connection relies on subclasses to actually implement the
 * sending and receiving of packets.
 */
public abstract class Connection {
    private Hashtable _packetListeners = new Hashtable ();
    private Vector _startedListeners = new Vector();
    private Vector _disruptedListeners = new Vector();
    private Vector _errorListeners = new Vector();
    private byte[] _buf = null;
    private int _bufLength = 0;
    private boolean _dispatching = false;
    private String _remoteHostName = null;
    /*package*/ final int PACKET_LOG_PAD = 2;
    private boolean _connected = false;
    private boolean _authenticated = false;
    private boolean _startNotified = false;
    static private final String CONNECTION = "CONNECTION";

    /**
     * Connection constructor.
     */
    public Connection() {
    }

    /**
     * Send a Packet to our peer.  If this Connection is currently
     * connected and authenticated, <tt>packet</tt> is sent.
     * Otherwise, <tt>packet</tt> is dropped.  The caller needs to
     * handle notification from a ConnectionStartedListener in order
     * to handle Connections that go down and subsequently come back
     * up.
     * 
     * @param packet Packet to send.
     */
    public void sendPacket(Packet packet) {
	if (!_connected || !_authenticated || !_startNotified) {
	    Log.debug(CONNECTION, "sendPacket on down connection");
	    return;
	}
	sendPacketImpl(packet);
    }

    /**
     * Send a Packet to our peer.  This method is for sending
     * Authentication packets on an unauthenticated Connection.
     * 
     * @param packet Packet to send.
     */
    public void sendAuthPacket(Packet packet) {
	if (!_connected) {
	    Log.debug(CONNECTION, "sendAuthPacket on down connection");
	    return;
	}
	sendPacketImpl(packet);
    }

    /**
     * Called by Authenticator when authentication has succeeded.
     *
     * @param authenticated true if this Connection is authenticated.
     */
    /*package*/ void setAuthenticated(boolean authenticated) {
	_authenticated = authenticated;
    }

    /**
     * Called by ServerConnectionFactory when a connection has been
     * established.
     * 
     * @param connected true if this Connection is connected.
     */
    /*package*/ void setConnected(boolean connected) {
	_connected = connected;
    }

    /**
     * Send <TT>packet</TT> to our peer.  Subclasses must override
     * this to implement transfer of a Packet.
     * 
     * @param packet Packet to send.
     */
    protected abstract void sendPacketImpl(Packet packet);

    /**
     * Close this Connection.  After calling <TT>close</TT>, packets
     * can no longer be sent or received on this connection.
     */
    public abstract void close();

    /**
     * Get the name of the remote host.
     * 
     * @return The name of the remote host.
     */

    public String getRemoteHostName() {
	return _remoteHostName;
    }

    /**
     * Set the name of the remote host.
     * 
     * @set The name of the remote host.
     */
    public void setRemoteHostName(String host) {
	_remoteHostName = host;
    }

    /**
     * Add a listener to receive packets of type <TT>type</TT>.
     * 
     * @param type The type of packet this listener should receive.
     * @param listener The listener to install.
     */
    public synchronized void addPacketListener(String type,
					       PacketListener listener) {
	Log.assert(!_packetListeners.containsKey(type),
		   "Attempt to add second listener for packet type: "
		   + type);
	_packetListeners.put(type, listener);
    }

    /**
     * Remove the listener that receives packets of type <TT>type</TT>.
     * 
     * @param type Type of packet to remove listener for.
     */
    public synchronized void removePacketListener(String type) {
	_packetListeners.remove(type);
    }
   
    /**
     * Add a listener that gets notified when the connection is
     * started.  The same connection can get started multiple times,
     * because the connection can get lost and re-established.  If the
     * connection is currently up, <tt>addStartedListener</tt>
     * notifies <tt>listener</tt> immediately.
     * 
     * @param listener listener to get notified of restarts.
     */
    public void addStartedListener(ConnectionStartedListener listener) {
	synchronized (_startedListeners) {
	    _startedListeners.addElement(listener);
	    if (_connected && _authenticated && _startNotified) {
		listener.connectionStarted(
		    new ConnectionEvent(this, ConnectionEvent.STARTED));
	    } else {
		Log.debug(CONNECTION, "addStartedListener on down connection");
	    }
	}
    }

    /**
     * Remove a listener from the started listeners.
     * 
     * @param listener 
     */
    public void removeStartedListener(ConnectionStartedListener listener) {
	_startedListeners.removeElement(listener);
    }
     
    /**
     * Add a listener that gets notified when an error occurs.
     * 
     * @param listener listener to get notified of errors.
     */
    public void addErrorListener(ConnectionErrorListener listener) {
	_errorListeners.addElement(listener);
    }

    /**
     * Remove a listener from the error listeners.
     * 
     * @param listener 
     */
    public void removeErrorListener(ConnectionErrorListener listener) {
	_errorListeners.removeElement(listener);
    }

    /**
     * Notify listeners that a non-fatal error has occurred.
     * 
     * @param errorString Localized string describing the error.
     */
    public void notifyError(String errorString) {
	notifyError(errorString, false);
    }

    /**
     * Notify listeners that a fatal error has occurred.
     * 
     * @param errorString Localized string describing the error.
     */
    public void notifyFatalError(String errorString) {
	notifyError(errorString, true);
    }

    /**
     * Called by subclasses to distribute a packet to the proper
     * listerner.
     * 
     * @param packet The packet to distribute.
     */
    protected void distributePacket(final Packet packet) {
	invokeLater(new Runnable() {
	    public void run() {
		if (Log.isLevelOn(Log.TRACE)) {
		    // Calling packet.toString is really expensive - skip it
		    // if it's not going to be used.
		    Log.trace(CONNECTION, "packet distributed: "
			      + packet.toString(PACKET_LOG_PAD));
		}
		final PacketListener listener
		    = (PacketListener)_packetListeners.get(packet.getType());
		if (listener != null) {
		    if (Log.isLevelOn(Log.TRACE)) {
			// Calling packet.toString is really expensive - skip it
			// if it's not going to be used.
			Log.trace(CONNECTION, "packet received: " +
				  packet.toString(PACKET_LOG_PAD));
		    }
		    listener.receivePacket(packet);
		}
	    }
	});
    }

    /**
     * Called by subclasses when we lose our connection.
     */
    protected void notifyDisrupted() {
	_connected = false;
	_authenticated = false;
	_startNotified = false;
	Enumeration listeners = _disruptedListeners.elements();
	ConnectionEvent event
	    = new ConnectionEvent(Connection.this,
				  ConnectionEvent.DISRUPTED);
	while (listeners.hasMoreElements()) {
	    ConnectionDisruptedListener listener
		= (ConnectionDisruptedListener)listeners.nextElement();
	    listener.connectionDisrupted(event);
	}
    }

    /**
     * Called when we have established a connection.
     */
    public void notifyStarted() {
	Log.assert(_connected && _authenticated && !_startNotified,
		   "Wrong state for start notification ");
	synchronized (_startedListeners) {
	    _startNotified = true;
	    Enumeration listeners = _startedListeners.elements();
	    ConnectionEvent event
		= new ConnectionEvent(Connection.this,
				      ConnectionEvent.STARTED);
	    while (listeners.hasMoreElements()) {
		ConnectionStartedListener listener
		    = (ConnectionStartedListener)listeners.nextElement();
		listener.connectionStarted(event);
	    }
	}
    }

    /**
     * Add a listener that gets notified when we lose our connection.
     * The only client of this should be ServerConnectionFactory.
     * 
     * @param listener listener to get notified of lost connections.
     */
    public void addDisruptedListener(ConnectionDisruptedListener listener) {
	_disruptedListeners.addElement(listener);
    }

    /**
     * Remove a listener from the disrupted listeners.
     * 
     * @param listener 
     */
    public void removeDisruptedListener(ConnectionDisruptedListener listener) {
	_disruptedListeners.removeElement(listener);
    }

    /**
     * Code shared between notifyError() and notifyFatalError().
     * 
     * @param errorString Localized string describing the error.
     * @param fatal true if this is a fatal error.
     */
    private void notifyError(final String errorString, final boolean fatal) {
	// We're never supposed to get here; if we're here, there's
	// been some sort of protocol botch.  Print some extra
	// information so someone can file a bug.
	Log.debug(CONNECTION, "Connection Error: " + errorString);
	Log.debug(CONNECTION, SysUtil.stackTraceToString(new Throwable()));

	Enumeration listeners = _errorListeners.elements();
	ConnectionEvent event = new ConnectionEvent(
	    Connection.this, errorString);
	event.setFatal(fatal);
	while (listeners.hasMoreElements()) {
	    ConnectionErrorListener listener
		= (ConnectionErrorListener)listeners.nextElement();
	    listener.connectionError(event);
	}
    }

    /**
     * This is used to send notification to listeners on the UI thread
     * rather than on the Connection I/O thread.  The reason for doing
     * this is to avoid having to make client code thread-aware since
     * Swing is not thread-safe.
     * 
     * @param functor Functor to run at a later time.
     */
    void invokeLater(final Runnable functor) {
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		try {
		    Log.assert(!_dispatching,
			       "re-entrant call to invokeLater.\n" +
			       "This probably means that swing or awt\n" +
			       "created a second thread for processing\n" +
			       "events while a modal dialog is displayed.\n\n" +
			       "Modal dialogs must not be used in the same\n" +
			       "application as Connection.  Use the\n" +
			       "methods of com.sgi.sysadm.ui.UIContext instead.");
		    _dispatching = true;
		    functor.run();
		} catch (Exception ex) {
		    // Print the exception here because
		    // SwingUtilities.InvokeComponent.processEvent() 
		    // catches all exceptions and does not print
		    // their messages.
		    System.out.println(ex.getMessage());
		    ex.printStackTrace();
		} finally {
		    _dispatching = false;
		}
	    }
	});
    }
}
