//
// ConnectionEvent.java
//
//	Events sent to Connection*Listener interfaces.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.comm;

import java.util.*;
import com.sgi.sysadm.util.Log;

/**
 * ConnectionEvent is passed to the methods of the
 * ConnectionInitializedListener, ConnectionReinitializedListener,
 * ConnectionDisruptedListener, and ConnectionErrorInterfaces.
 */
public class ConnectionEvent extends EventObject {

    /**
     * Event types.
     */
    public static final int STARTED = 1;
    public static final int DISRUPTED = 2;
    public static final int ERROR = 3;

    private String _errorString = null;
    private boolean _fatal = false;
    private int _type = 0;

    /**
     * Construct a ConnectionEvent.
     * 
     * @param conn Connection associated with this event.
     * @param type Type of this event.
     */
    public ConnectionEvent(Connection source, int type) {
	super(source);
	_type = type;
	Log.assert(STARTED <= type && type <= ERROR,
		   "Invalid ConnectionEvent type");
    }

    /**
     * Construct a ConnectionEvent representing an error.
     * 
     * @param conn Connection associated with this event.
     * @param errorString Localized string describing the error.
     */
    public ConnectionEvent(Connection source, String errorString) {
	super(source);
	_type = ERROR;
	_errorString = errorString;
    }

    /**
     * Access the Connection object associated with this event.
     * 
     * @return Connection object associated with this event.
     */
    public Connection getConnection() {
	return (Connection)getSource();
    }

    /**
     * Access the type of this event.
     * 
     * @return The type of this event.
     */
    public int getType() {
	return _type;
    }

    /**
     * If this is an ERROR event, access a localized string describing
     * the error.
     * 
     * @return localized string describing the error.
     */
    public String getErrorString() {
	Log.assert(_type == ERROR,
		   "Attempt to setFata on non-ERROR ConnectionEvent");
	return _errorString;
    }

    /**
     * If this is an ERROR event, set a flag to indicate whether or
     * not it is a fatal error.
     * 
     * @param fatal 
     */
    public void setFatal(boolean fatal) {
	Log.assert(_type == ERROR,
		   "Attempt to setFata on non-ERROR ConnectionEvent");
	_fatal = true;
    }

    /**
     * If this is an ERROR event, return true if this is a fatal
     * error, false otherwise.
     * 
     * @return true if this is a fatal error, false otherwise.
     */
    public boolean isFatal() {
	Log.assert(_type == ERROR,
		   "isFatal query on non-ERROR ConnectionEvent");
	return _fatal;
    }
}
