//
// TcpmuxServerConnectionFactory.java
//
//	Connect to sysadmd tcpmux server.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.comm;

import java.io.*;
import java.net.*;
import java.text.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.proxy.comm.*;
import javax.swing.*;

/**
 * TcpmuxServerConnectionFactory is used to create a Connection object
 * that is connected to a tcpmux sysadmd server.
 */
public class TcpmuxServerConnectionFactory implements ServerConnectionFactory {

    private static final int TCPMUX_PORT = 1;
    private static final String CLASS_NAME = "TcpmuxServerConnectionFactory";

    private int _port = TCPMUX_PORT;
    private String _connectHost = null;
    private boolean _usingProxy = false;
    private Thread _tcpmuxThread = null;
    private ResultListener _listener = null;
    private boolean _threadCancelable = false;
    private boolean _cancel = false;

    public Connection createConnection() {
	return new SocketConnection();
    }

    /* 
     * This method sets an alternate host/port to connect.
     * Used when portforwarding Connection.method is used.
     */
    public void setProxy(String host, String port) {
	int portInt;
	try {
	    portInt = Integer.parseInt(port);
	} catch (Exception ex) {
	    portInt = -1;
	}
	setProxy(host, portInt);
    }

    /* 
     * This method sets an alternate host/port to connect.
     * Used when portforwarding Connection.method is used.
     */
    public void setProxy(String host, int port) {
	_connectHost = host;
	_port = port;
	_usingProxy = true;
    }

    public void clearProxy() {
	_usingProxy = false;
	_port = TCPMUX_PORT;
    }

    public boolean usingProxy() {
	return _usingProxy;
    }

    /*
     * Start connection to remote host with actual connection
     * logic residing in a newly created thread.
     */
    public void connect(Connection conn, Object arg, ResultListener listener) {

	Log.assert(conn instanceof SocketConnection,
		   "Attempt to connect using socket not " +
		   "created by createConnection()");

	if (_usingProxy && _port < 0) {
	    notifyListener(false, new ResultEvent(this), listener);
	    return;
	}

	_cancel = false;
	_listener = listener;
	_threadCancelable = true;

	Log.info(CLASS_NAME, "Attempting " + 
	      (_usingProxy ? "port forwarding" : "tcpmux") + " connection");

	if (!_usingProxy) {
	    _connectHost = (String)arg;
	}

	createTcpmuxThread(conn).start();
    }

    /* 
     * Cancel connection thread.
     */
    public synchronized void cancelConnection() {
	if (_threadCancelable) {
	    _cancel = true;
	    if (_tcpmuxThread != null) {
		_tcpmuxThread.stop();
	    }
	    notifyListener(false, new ResultEvent(this), _listener);
	    resetThreadData();
	}
    }

    /*
     * Send name of this host to server since peer of remote
     * socket conection will not resolve to the correct host.
     */
    private void sendHostName(Connection conn) {
	final String host;
	try {
	    host = InetAddress.getLocalHost().getHostAddress();
	} catch (UnknownHostException ex) {
	    host = "unknownhost";
	}

	final SysadmProxy proxy = SysadmProxy.get(conn);

	proxy.setHostName(host, new ResultListener() {

 	    public void succeeded(ResultEvent event) {
		proxy.release();
	    }

 	    public void failed(ResultEvent event) {
		proxy.release();
	    }
	});
    }


    /*
     * Get the name of the server machine when using a proxy.
     */
    private void getServerHostName(Connection conn) {

	final SysadmProxy proxy = SysadmProxy.get(conn);

	proxy.getHostName(new ResultListener() {

 	    public void succeeded(ResultEvent event) {
		proxy.release();
	    }

 	    public void failed(ResultEvent event) {
		proxy.release();
	    }
	});
    }

    /* 
     * Create class member _tcpmuxThread, a thread which contains
     * the actual connection logic.  This method is synchronized
     * because cancelConnection uses _tcpmuxThread to determine
     * if thread has been created
     */
    private synchronized Thread createTcpmuxThread(final Connection conn) {

	_tcpmuxThread = new Thread("Tcpmux Connection") {
	
	    public void run() {

		if (_cancel) {
		    return;
		}

		try {
		    Socket sock = new Socket(_connectHost, _port);
		    OutputStream out = sock.getOutputStream();
		    InputStream in = sock.getInputStream();
		    
		    // Initially, we are talking to inetd on the
		    // server.  We write "sgi_sysadm" to request that
		    // inetd start sysadmd for us.  inetd forks and
		    // execs sysadmd, which then writes back the ACK
		    // string "+\r\n".  From then on, all
		    // communication over the socket is in the form of
		    // serialized Packets.
		    out.write("sgi_sysadm\r\n".getBytes());
		    out.flush();
		
		    String ack = SysUtil.readLine(in);
		    try {
			if (ack.charAt(0) != '+') {
			    String msg = null;
			    if (ack.length() > 2) {
				msg = ack.substring(1, ack.length());
			    }
			    throw new IOException(msg);
			}
		    } catch (IndexOutOfBoundsException ex) {
			throw new IOException("tcpmux protocol botch");
		    }

		    preventThreadCancel();

		    // close connection if it is open
		    conn.close();
		    ((SocketConnection)conn).setSocket(sock);
		    
		    if (_usingProxy) {
			sendHostName(conn);
			getServerHostName(conn);
		    } 

		    Log.info(CLASS_NAME, "Using " + 
			     (_usingProxy ? "port forwarding" : "tcpmux" )+ 
			     " connection");

		    conn.setConnected(true);
		    notifyListener(true, new ResultEvent(this), _listener);

		} catch (Exception ex) {
		    preventThreadCancel();
		    ResultEvent event = new ResultEvent(this);
		    String msg;
		    if (ex instanceof UnknownHostException) {
			ResourceStack rs = new ResourceStack();
			rs.pushBundle("com.sgi.sysadm.comm.PackageP");
			msg = MessageFormat.format(
			    rs.getString("TcpmuxConnection.unknownHost"),
			    new Object[] { _connectHost });
		    } else {
			msg = ex.getMessage();
		    }
		    if (msg != null) {
			event.setReason(msg);
		    }
		    notifyListener(false, event, _listener);
		}
		resetThreadData();
	    }
	};
	return _tcpmuxThread;
    }

    /* 
     * Disable the ability to cancel connection thread.
     */
    private synchronized void preventThreadCancel() {
	_threadCancelable = false;
    }

    /* 
     * Clear all data used by thread so there are no dangling
     * references to unused objects.
     */
    private void resetThreadData() {
	_threadCancelable = false;
	_listener = null;
	_tcpmuxThread = null;
    }

    /*
     * Notify listener of success or failure in event dispatcher 
     * thread since this call could ultimately cause a new window to be 
     * displayed.
     */
    private void notifyListener(final boolean success, final ResultEvent res,
				final ResultListener listener) {
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		if (success) {
		    listener.succeeded(res);
		}
		else {
		    listener.failed(res);
		}
	    }
	});
    }
}

