//
// RApp.java
//
//	An abstract base class for sysadm-based applications.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager;

import javax.swing.*;
import javax.swing.text.*;
import javax.swing.plaf.metal.*;
import javax.swing.plaf.basic.*;
import javax.swing.plaf.*;
import javax.swing.border.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;
import java.awt.*;
import java.net.URL;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.manager.login.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.proxy.util.*;

/**
 * RApp is an abstract base class for sysadm-based applications.
 * It provides login services at startup.
 * <P>
 * Subclasses of RApp need to provide a main() method as
 * well as the abstract createFrame(), setArgs(), and launchApp()
 * methods.
 * <P>
 * For example:
 * <PRE>
 * public class RunTask extends RApp {
 *     UIContext _uic;
 *     ResourceStack_rs;
 *     RFrame _frame;
 * 
 *     static public void main(String[] args) {
 *         new RunTask(args).loginAndRun(args);
 *     }
 * 
 *     public RunTask(String[] args) {
 *         super(RunTask.class.getName(), null, args);
 *         _uic = getUIContext();
 *         _rs = _uic.getResourceStack();
 *     }
 * 
 *     public RFrame createFrame() {
 *         _frame = new TaskFrame();
 *         return _frame;
 *     }
 * 
 *     // setArgs() and launchApp() are left as an exercise to the reader
 * }
 * </PRE>
 * <P>
 * Subclasses of RApp should override run() if they wish
 * to implement <A HREF="glossary.html#RunOnce">run-once behavior</A>.
 * <P>
 * Subclasses should also define the
 * following properties in the {app name}P.properties file:
 * <UL>
 * <LI>RApp.dialogTitle - The string that is to be displayed on any
 * application message dialogs.
 * <LI>RApp.Error.appUsage - A string containing the options and
 * operands that are legal on the application command line.  These
 * options and operands will be appended to the list of options that RApp
 * can accept on the command line. For example, RunTask accepts an
 * optional -d flag, requires the task class name, and will accept
 * additional operands on its command line. RunTask specifies the resource
 * as follows:
 * <PRE>
 *     #
 *     # RunTaskP.properties
 *     #
 *     RApp.Error.appUsage = [-d key=value] &lt;task name&gt; [operands]
 * </PRE>
 * </UL>
 * To launch a sysadm application from the command line:
 * <p>
 * <pre>
 * Usage: java {appClass} [-L ll[_CC[_variant]]] [-p product name] \ 
 *                        [-s server name] [-l login name] [-n] \
 *                        [application options and operands]
 *        -L ll[_CC[_variant]]
 *	     Set the locale.
 *        -p product name
 *           The CLASSPATH relative name of the package containing
 *           PackageP.properties for this application.
 *        -s server name
 *           The name of the server on which to run the Task.
 *        -l login name
 *           Login name to use when logging into the server.
 *        -n Don't display the RemoteHostPanel; use values from -s
 *           and -l.
 *        all other arguments are passed to the application.
 * </pre>
 *
 * If the application should display a splash screen, create a file
 * <CODE>SplashP.properties</CODE> under the package directory passed to
 * the <i>productName</i> argument of the constructor.  In that file,
 * put three resources: SPLASH_IMAGE, SPLASH_MESSAGE, and SPLASH_FONT.
 *
 * @see #SPLASH_IMAGE
 * @see #SPLASH_MESSAGE
 * @see #SPLASH_FONT
 *
 * @param args List of command line arguments. 
 */
public abstract class RApp {
    /**
     * The property <I>RApp.dialogTitle</I> is a String that is
     * used as the title on all dialogs displayed by this application.
     */
    public static final String DIALOG_TITLE = "RApp.dialogTitle";

    /**
     * The property set <i>RApp.uidefs[n]</i> is a array of strings
     * that control the UIDefaults for the RApp.  The value of each
     * of the Strings should be then name of a UIDefault to set.  For
     * each value, there should be two additional properties:
     * <i>RApp.uidefs[n].type</i> and <i>RApp.uidefs[n].value</i>
     * These three keys make up a set:
     * <pre>
     * RApp.uidefs0 = Button.font
     * RApp.uidefs0.type=FONT
     * RApp.uidefs0.value= SansSerif-12 
     * </pre>
     * The <i>type</i> can be either the String "FONT", or the String
     * "COLOR".  If the <i>type</i> is FONT, a
     * javax.swing.plaf.FontUIResource will be added to the UI
     * Defaults, with the font as specified by the <i>value</i> key.
     * If the <i>type</i> is COLOR, a javax.swing.plaf.ColorUIResource
     * will be added to the UI Defaults, with the color as specified by
     * the <i>value</i> key.
     * <p>
     * Defaults are set in com.sgi.sysadm.ui.SysadmUIP.properties, and
     * the values can be overridden in the products's PackageP file,
     * where the product given by the -p flag in the args to the app.
     *
     * @see javax.swing.UIDefaults
     */
    public static final String UIDEFS = "RApp.uidefs";

    /**
     * The resource <I>RApp.Error.unknownOption</I> is a format String that
     * is displayed by unknownOptionMsg() to indicate to the user that
     * a specific option was not found.  Argument {0} is the option
     * that was not recognized.
     */
    public static final String UNKNOWN_OPTION =
	"RApp.Error.unknownOption";

    /**
     * The resource <I>RApp.Error.tooManyArgs</I> is a String that is
     * displayed by tooManyArgsMsg() to indicate to the User that too
     * many arguments were specified on the command line.
     */
    public static final String TOO_MANY_ARGS = "RApp.Error.tooManyArgs";

    /**
     * The resource <I>SplashImage</I> in the file <CODE>&lt;package
     * name&gt;/SplashP.properties</I> is the name of an image that
     * should be displayed on the splash screen.  eg: <CODE>
     * com/sgi/my-package/splash.gif</CODE>
     */
    public static final String SPLASH_IMAGE = "SplashImage";

    /**
     * The resource <I>SplashMessage</I> in the file <CODE>&lt;package
     * name&gt;/SplashP.properties</I> is the text that
     * should be displayed on the splash screen.  eg: <CODE>
     * Now loading ...</CODE>
     */
    public static final String SPLASH_MESSAGE = "SplashMessage";

    /**
     * The resource <I>SplashFont</I> in the file <CODE>&lt;package
     * name&gt;/SplashP.properties</I> is the name of the font used for
     * the text displayed on the splash screen.  eg: <CODE>
     * SansSerif-Bold-14</CODE>
     */
    public static final String SPLASH_FONT = "SplashFont";

    /**
     * RAppLaunchEvent is passed to RAppLaunchListeners when the attempted
     * launch succeeds, fails, or is aborted because the application
     * is already running.
     */
    public static class RAppLaunchEvent extends EventObject {
	private Object _resultObject;

	/**
	 * Constructor.
	 *
	 * @param source The Object generating this RAppLaunchEvent.
	 * @param result The result of the attempted launch.
	 *               <UL>
	 *               <LI>On success, <TT>result</TT> will be a
	 *                   JComponent that can be used as the dialog
	 *                   parent for the login panel in case the
	 *                   connection to the server is lost.
	 *               <LI>On failure, <TT>result</TT> should be a
	 *                   localized message explaining the failure.
	 *               <LI>If the application is already running,
	 *                   <TT>result</TT> may be null or any Object
	 *                   that the application wishes to pass back
	 *                   to the caller.  See the documentation for
	 *                   the specific application being launched.
	 *               </UL>
	 */
	public RAppLaunchEvent(Object source, Object result) {
	    super(source);
	    _resultObject = result;
	}

	/**
	 * For a successful launch, get the component that can be used
	 * as the login panel dialog parent.  If the connection is
	 * lost while the application is running, the login panel will
	 * be brought up centered over the dialog parent.
	 *
	 * @return The JComponent that can be used as the login panel
	 *         dialog parent.
	 */
	public JComponent getDialogParent() {
	    if (_resultObject == null ||
		    !(_resultObject instanceof JComponent)) {
		return null;
	    } else {
		return (JComponent)_resultObject;
	    }
	}

	/**
	 * For a failed launch, get the localized string that
	 * describes why the launch failed.
	 *
	 * @return A localized string that describes why the launch
	 *         failed; <TT>null</TT> if the event does not
	 *         represent a failed launch.
	 */
	public String getMessage() {
	    if (_resultObject == null ||
		    !(_resultObject instanceof String)) {
		return null;
	    } else {
		return (String)_resultObject;
	    }
	}

	/**
	 * If the application was already running, get the Object
	 * passed back from the application (if any).
	 *
	 * @return The Object passed back by the already-running
	 *         application.  May be <TT>null</TT>.
	 */
	public Object getResultObject() {
	    return _resultObject;
	}
    }

    /**
     * RAppLaunchListener is used as a parameter to the run() and
     * launchApp() methods.
     */
    public static interface RAppLaunchListener extends EventListener {
	/**
	 * If run() or launchApp() succeeds, launchSucceeded() will
	 * be called.
	 *
	 * @param event Use event.getDialogParent() to get the
	 *              JComponent which can be used as the login
	 *              dialog parent in case the connection to the
	 *              server is lost.
	 */
	public void launchSucceeded(RAppLaunchEvent event);

	/**
	 * If run() or launchApp() fails, launchFailed() will be
	 * called.
	 *
	 * @param event Use event.getMessage() to get the String which
	 *              contains a localized error message explaining
	 *              the failure.
	 */
	public void launchFailed(RAppLaunchEvent event);

	/**
	 * If the application supports run-once and determines that
	 * the application is already running, launchAlreadyRunning()
	 * is called.
	 *
	 * @param event Use event.getResultObject() to get the Object
	 *              passed back by the already-running
	 *              application.  To determine what information is
	 *              passed back in the event, check the
	 *              documentation of the application you are launching.
	 */
	public void launchAlreadyRunning(RAppLaunchEvent event);
    }

    private RFrame _frame;
    private UIContext _uic;
    private ResourceStack _rs;
    private String _appClassName;
    private String _serverName;
    private String _loginName;
    private boolean _noPanel = false;
    private HostContext.Client _rAppClient;
    private String _productName;
    private String _preScanError;
    private Window _splashWindow;
    private static boolean _showedSplash = false;
    
    
    static private final String NEWLINES = "\n\n";
    private static final String CLASS_NAME = "RApp";
    private static boolean _installedRhinoLook;

    // Listener used as an argument to post*() to close the
    // application after a message dialog has been confirmed by the
    // User.
    private ActionListener _confirmListener = new ActionListener() {
	public void actionPerformed(ActionEvent event) {
	    if (_frame != null) {
		_frame.dispose();
	    }
            System.exit(1);
	}
    };
    
    private static class RhinoButtonBorder extends MetalBorders.ButtonBorder {
	public void paintBorder(Component c, Graphics g,
				int x, int y, int width, int height) {
	    ButtonModel model = ((JButton)c).getModel();
	    if (model.isEnabled()) {
		super.paintBorder(c, g, x, y, width, height);
	    } else {
		g.translate(x, y);
		g.setColor(new Color(153, 153, 153));
		g.drawRect(0, 0, width - 2, height - 2);
	    }
	}
    }
    
    private static class RhinoTextFieldBorder
                     extends MetalBorders.TextFieldBorder {
	public void paintBorder(Component c, Graphics g,
				int x, int y, int width, int height) {
	    if (c.isEnabled() && ((JTextComponent)c).isEditable()) {
		super.paintBorder(c, g, x, y, width, height);
	    } else {
		g.translate(x, y);
		g.setColor(new Color(153, 153, 153));
		g.drawRect(0, 0, width - 1, height - 1);
	    }
	}
    }

    /**
     * This is ripped right out of MetalUtils.java.  The only
     * difference between this class and MetalUtils.TableHeaderBorder
     * is that here we have a border width on the right, so that
     * right-aligned headers look nice.
     */
    private static class RhinoTableHeaderBorder extends AbstractBorder {
        protected Insets editorBorderInsets = new Insets( 2, 2, 2, 2 );

        public void paintBorder(Component c, Graphics g,
				int x, int y, int w, int h) {
	    g.translate( x, y );
	    
	    g.setColor( MetalLookAndFeel.getControlDarkShadow() );
	    g.drawLine( w-1, 0, w-1, h-2 );
	    g.drawLine( 1, h-2, w-1, h-2 );
	    g.setColor( MetalLookAndFeel.getControlHighlight() );
	    g.drawLine( 0, 0, w-2, 0 );
	    g.drawLine( 0, 0, 0, h-3 );
	    g.setColor( MetalLookAndFeel.getControlShadow() );
	    g.drawLine( 0, h-1, w-1, h-1 );


	    g.translate( -x, -y );
	}

        public Insets getBorderInsets( Component c ) {
	    return editorBorderInsets;
	}
    }	

    private static class RhinoMetalTheme extends DefaultMetalTheme {
	
	private ResourceStack _rs;

	RhinoMetalTheme(ResourceStack rs) {
	    _rs = rs;
	}

	public ColorUIResource getControlShadow() {
	    return new javax.swing.plaf.ColorUIResource(
		255, 255, 255);
	}

	public void addCustomEntriesToTable(UIDefaults uidefs) {
	    String[] resources = _rs.getStringArray(UIDEFS, 
						    new String[0]);
	    String type;
	    try {
		for (int ii = 0; ii < resources.length; ii++) {
		    try {
			type = _rs.getString(UIDEFS + ii + ".type");
		    } catch (MissingResourceException mre) {
			Log.error(CLASS_NAME,
				  "\"" + UIDEFS + ii + "\" was set to \"" +
				  resources[ii] + "\", but \"" +
				  UIDEFS + ii + ".type\" was not set.  Skipping.\n"
				  + mre.toString());
			continue;
		    }
		    if (type.equals("FONT")) {
			try {
			    uidefs.put(
				resources[ii],
				_rs.getFont(UIDEFS + ii
						+ ".value"));
			} catch (MissingResourceException mre) { 
			    Log.error(CLASS_NAME,
				       "\"" + UIDEFS + ii + "\" was set to \"" +
				      resources[ii] + "\", and \"" +
				      UIDEFS + ii + ".type\" was set to \"" +
				      type + "\" but \"" + UIDEFS + 
				      ii + ".value\" was not set.  Skipping.\n"
				      + mre.toString());
			}
			} else if (type.equals("COLOR")) {
			try {
			    uidefs.put(resources[ii],
				       new
				       javax.swing.plaf.ColorUIResource(
					   _rs.getColor(UIDEFS + ii
							+ ".value")));
			} catch (MissingResourceException mre) { 
			    Log.error(CLASS_NAME,
				       "\"" + UIDEFS + ii + "\" was set to \"" +
				      resources[ii] + "\", and \"" +
				      UIDEFS + ii + ".type\" was set to \"" +
				      type + "\" but \"" + UIDEFS +
				      + ii + ".value\" was not set.  Skipping.\n"
				      + mre.toString());
			}
			}
		}
		
		uidefs.put("Button.border",
			   new javax.swing.plaf.BorderUIResource(
			       new CompoundBorder(
				   new RhinoButtonBorder(),
				   new BasicBorders.MarginBorder())));
		uidefs.put("TextField.border",
			   new javax.swing.plaf.BorderUIResource(
			       new CompoundBorder(
				   new RhinoTextFieldBorder(),
				   new BasicBorders.MarginBorder())));
		uidefs.put("TableHeader.cellBorder",
			   new javax.swing.plaf.BorderUIResource(
			       new RhinoTableHeaderBorder()));
		// This is the same border that
		// DefaultTableCellRenderer uses when the cell does
		// not have focus.  This keeps labels from jumping
		// when you select them.
		uidefs.put("Table.focusCellHighlightBorder",
			   new javax.swing.plaf.BorderUIResource(
			       new EmptyBorder(1, 2, 1, 2)));
	    } catch (Exception e) {
		Log.fatal("Exception while reading uidefs from properties: " + e);
	    }
	}
    }

    /**
     * Install the Rhino colors and fonts.
     * 
     * @param rs The ResourceStack that contains the fonts and colors
     * to use in the Rhino look.
     *
     * @param addEntries A flag that controls whether this method
     * makes an explicit call to <tt>addCustomEntriesToTable</tt>.
     * When launched as an application, this call is not necessary, as
     * the call to <tt>setCurrentTheme<tt> automatically calls
     * <tt>addCustomEntriesToTable</tt>.  But when launched as an
     * Applet, the call is necessary, as
     * <tt>addCustomEntriesToTable</tt> is not automatically called.
     *
     */
    public static void installRhinoLook(ResourceStack rs,
					boolean addEntries) {
	if (_installedRhinoLook) {
	    return;
	}
	_installedRhinoLook = true;
	RhinoMetalTheme theme = new RhinoMetalTheme(rs);
	MetalLookAndFeel.setCurrentTheme(theme);
	if (addEntries) {
	    UIDefaults uidefs = UIManager.getLookAndFeelDefaults();
	    theme.addCustomEntriesToTable(uidefs);
	}
/*	try {
	    String lf = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";
	    String lf = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";
	    String lf = "javax.swing.plaf.metal.MetalLookAndFeel";
	    System.out.println("Setting look and feel to: " + lf);
	    UIManager.setLookAndFeel(lf);
	} catch (Exception e) {
	    System.out.println(e);
	}
*/	
	
    }

    /**
     * Constructor.
     *
     * Sets up the UIContext and ResourceStack for the application,
     * sets the application dialog title.
     *
     * @param appClassName The package-qualified name of the
     *                     application class.
     *                     ResourceStack.BUNDLE_SUFFIX will be
     *                     appended to this name to generate the
     *                     application ResourceBundle name.
     * @param productName A String containing the
     *                    <A HREF="glossary.html#CLASSPATHRelative">
     *                    CLASSPATH relative</A> name of the product
     *                    containing the package properties file.
     *                    May be <TT>null</TT>.  The value of this
     *                    parameter will be overridden if the -p
     *                    option is specified on the command line.
     * @param args The commandLine args that RApp should parse to get
     *             the arguments it needs.
     */
    public RApp(String appClassName, String productName, String[] args) {
	_productName = productName;
	preScanArgs(args);

        if (_productName != null && _showedSplash == false) {
            _showedSplash = true;
            
            // We use SplashP.properties instead of the normal
            // PackageP.properties to assure that that file will be
            // small and fast to read.
            
            StringBuffer fileNameBuf = new StringBuffer(
                _productName.replace('.', '/')).append("/SplashP.properties");
            if (fileNameBuf.charAt(0) != '/') {
                fileNameBuf.insert(0, '/');
            }
            String fileName = fileNameBuf.toString();

            try {
                java.io.InputStream in =
                    getClass().getResourceAsStream(fileName);
                if (in == null) {
                    Log.info(CLASS_NAME,
                             "Didn't find splash screen properties file: " +
                             fileName);
                } else {
                    ResourceBundle bundle = new PropertyResourceBundle(in);
                    String splashImage = null;
                    String splashMessage = null;
                    String splashFont = null;
                    
                    try {
                        splashImage = bundle.getString(SPLASH_IMAGE);
                        splashMessage = bundle.getString(SPLASH_MESSAGE);
                        splashFont = bundle.getString(SPLASH_FONT);
                    } catch (MissingResourceException e) {
                        String fatalStr =
                        "Failed to load a property necessary for "+
                        "the splash screen from " + fileName + ": ";
                        
                        if (splashImage == null) {
                            Log.fatal(fatalStr + SPLASH_IMAGE);
                        } else if (splashMessage == null) {
                            Log.fatal(fatalStr + SPLASH_MESSAGE);
                        } else {
                            Log.fatal(fatalStr + SPLASH_FONT);
                        }
                    }
                    
                    URL url = getClass().getResource(splashImage);
                    if (url == null) {
                        Log.fatal("Tried to show a splash sreen, but " +
                                  "couldn't find: " + splashImage +
                                  "which was specified in " + fileName);
                    }

                    Font font = Font.decode(splashFont);
                    if (font == null) {
                        Log.fatal("Tried to show a splash sreen, but " +
                                  "couldn't parse: " + splashFont +
                                  "which was specified in " + fileName);
                    }
                    
                    
                    new SplashWindowFrame(url, splashMessage, font);
                }
                
            } catch (java.io.IOException e) {}
        }
        
        try {
	    FinalizerElem.workAroundAWTFinalizerHang();
	} catch (Throwable err) {
	}
	// Make sure the Releaser class is loaded before we actually
	// need it; otherwise we can deadlock if the finalizer thread
	// is trying to load the Releaser class when another thread
	// has the Class lock.
	Releaser.release(new Releaseable() {
		    public void release() {
		    }
	});
// Code for figuring out which default to change.
//  	UIDefaults uidefs = UIManager.getLookAndFeelDefaults();
//  	Enumeration enum = uidefs.keys();
//  	while (enum.hasMoreElements()) {
//  	    String key = (String)enum.nextElement();
//  	    if (key.indexOf("ScrollBar") >= 0) 
//  		{
//  		    System.out.println(key+ "\t\t" + uidefs.get(key));
//  		}
//  	}

	_appClassName = appClassName;

	// Set up our UIContext
	_uic = new UIContext();

	_rs = _uic.getResourceStack();
	_rs.pushBundle(RApp.class.getName() +
		       ResourceStack.BUNDLE_SUFFIX);
	_rs.pushBundle(appClassName + ResourceStack.BUNDLE_SUFFIX);
	if (_productName != null) {
	    _rs.pushPackageBundles(_productName);
	}
	installRhinoLook(_rs, false);

	// Set the dialog title.
	String title = _rs.getString(DIALOG_TITLE);
	_uic.setDialogTitle(title);
    }

    /**
     * Initialize the application frame by calling createFrame() and
     * register the frame with HostContext.  RApp clients that use
     * run() or launchApp() to launch another RApp must first call initApp().
     * initApp() will be called automatically when the RApp is run
     * from the command-line via loginAndRun().
     */
    public final void initApp() {
	// Create a frame for displaying the application.
	_frame = createFrame();

	// Create a HostContext.Client for registering with HostContext
	_rAppClient = new HostContext.Client(_frame, _uic);
    }

    /**
     * Login and run the sysadm application.  This method should only
     * be called if the application is being run from the command
     * line.  Clients that have already logged in that want to run the
     * application should call the appropriate constructor for that
     * application and then call run().
     *
     * @param args The command-line arguments received by the
     *             application's main() method.
     */
    public final void loginAndRun(String[] args) {
	// If there was an error during pre-scanning of arguments, we
	// can post that error here.
	if (_preScanError != null) {
	    _uic.postError(errorMsg(_preScanError), _confirmListener);
	    return;
	}
	
	// Check the remaining command-line arguments
	parseArgs(args, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		login();
	    }

	    public void failed(ResultEvent event) {
                if (_splashWindow != null) {
                    _splashWindow.dispose();
                    _splashWindow = null;
                }
		_uic.postError(event.getReason(), _confirmListener);
	    }
	});
    }

    /**
     * Run a sysadm application.  This method should be used when one
     * RApp runs another RApp.  Since the first RApp
     * already has a HostContext, there is no need to log in again.
     * <P>
     * RApp clients that call run() must first call initApp() to make
     * sure that the application frame is created and registered with
     * the HostContext.  Failure to call initApp() will result in a
     * call to Log.fatal().
     * <P>
     * This method registers the application frame with the host
     * context, sets up a window listener that unregisters the frame
     * when the window is closed, and then calls launchApp().
     * Applications that wish to implement run-once behavior should
     * override this method.  Be sure to check the return value of
     * getFrame() in run() before dereferencing it; it will be
     * <TT>null</TT> if the caller neglected to call initApp().
     * 
     * @param hc HostContext
     * @param listener Notified of the result of calling launchApp().
     */
    public void run(final HostContext hc, RAppLaunchListener listener) {
	if (_frame == null) {
	    Log.fatal("initApp() must be called before run()");
	}
	hc.registerClient(_rAppClient);
	_frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
	_frame.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		hc.unregisterClient(_rAppClient);
		_frame.removeWindowListener((WindowListener)this);
	    }
	});

	launchApp(hc, listener);
    }

    /**
     * Get the RFrame in which this application is displayed.  The
     * RFrame will only be available after initApp() has been called.
     *
     * @return the RFrame in which this application is displayed;
     *         <TT>null</TT> if initApp() has not yet been called.
     */
    public RFrame getFrame() {
	return _frame;
    }

    /**
     * Get the UIContext associated with this application.
     *
     * @return the UIContext associated with this application.
     */
    public UIContext getUIContext() {
	return _uic;
    }

    /**
     * Get the product name that was specified on the command line, if
     * any.
     *
     * @return A String representing the product name specified on the
     *         command line with the -p option or null if no -p option
     *         was specified.
     */
    public String getProductName() {
	return _productName;
    }

    /**
     * Set the product name when running the RApp from another RApp.
     *
     * @param productName A String containing the
     *                    <A HREF="glossary.html#CLASSPATHRelative">
     *                    CLASSPATH relative</A> name of the product
     *                    containing the package properties file.
     */
    public void setProductName(String productName) {
	_productName = productName;
    }

    /**
     * Called by initApp() to initiate creation of the
     * application frame.  This frame is used as the parent of the
     * login dialog.  The subclass can access the returned RFrame
     * using the getFrame() method.
     *
     * @return the application RFrame.
     */
    public abstract RFrame createFrame();

    /**
     * Called by the base class to forward command-line arguments to
     * the subclass.  The subclass should parse <TT>appArgs</TT> and
     * notify <TT>listener</TT> of success or failure.  On failure,
     * the subclass should pass a localized error message to
     * <TT>listener</TT> via ResultEvent.setReason().
     *
     * @param appArgs A Vector of command-line arguments that were not
     *                consumed during command-line parsing by the RApp class.
     * @param listener The listener to call upon success or failure.
     */
    public abstract void setArgs(Vector appArgs,
				 ResultListener listener);

    /**
     * Called by run() to launch the application.  The
     * subclass should start the application and make it
     * visible. Clients that call launchApp() directly must first call
     * initApp() to ensure that the application frame has been created.
     *
     * @param hc HostContext The HostContext generated at login.
     * @param listener A listener to notify when the launch succeeds
     *                 or fails.  On success, the subclass should put
     *                 a JComponent into the ResultEvent result that
     *                 can be used as the dialog parent for the login
     *                 panel, in case the connection needs to be
     *                 restarted.
     */
    protected abstract void launchApp(HostContext hc,
				      RAppLaunchListener listener);

    /**
     * Called by run() to register the application's RFrame
     * with the HostContext before calling launchApp().
     * Subclasses should override this method if they want to
     * implement run-once behavior.
     * <P>
     * The base class will automatically unregister the application's
     * RFrame when the frame is closed.
     *
     * @param hc The HostContext to register the RFrame with.
     * @param listener The listener to notify when registration has
     *                 completed.
     */
    protected void registerClient(HostContext hc,
				  ResultListener listener) {
	hc.registerClient(_rAppClient);
	listener.succeeded(new ResultEvent(this));
    }


    private void login() {
	initApp();

	final LoginDialog loginDialog = new LoginDialog(_frame, _uic,
							_productName);
	final RemoteHostPanel panel = loginDialog.getPanel();
	if (_serverName != null) {
	    panel.setHost(_serverName);
	}
	if (_loginName != null) {
	    panel.setLogin(_loginName);
	}
	loginDialog.setResultListener(new ResultListener() {
	    public void succeeded(ResultEvent event) {
		run((HostContext)event.getResult(),
		    new RAppLaunchListener() {
		    public void launchSucceeded(RAppLaunchEvent event) {
			JComponent parent = event.getDialogParent();
			if (parent != null) {
			    panel.setDialogParent(parent);
			}
		    }

		    public void launchFailed(RAppLaunchEvent event) {
			// We can get here if the user cancels the
			// privilege dialog for a task.  In this case,
			// there's no message and we should just exit.
			String msg = event.getMessage();
			if (msg != null) {
			    _uic.postError(msg, _confirmListener);
			} else {
			    _frame.dispose();
			}
		    }

		    public void launchAlreadyRunning(RAppLaunchEvent event) {
			String message = event.getMessage();
			if (message != null) {
			    _uic.postInfo(message, _confirmListener);
			} else {
			    _frame.dispose();
			}
		    }
		});
	    }

	    public void failed(ResultEvent event) {
		_frame.dispose();
		System.exit(1);
	    }
	});

	// If we are not using the connection method the user intended,
	// do not send password/login over without the user having
	// an opportunity to at least cancel after being warned
        if (_splashWindow != null) {
            _splashWindow.dispose();
            _splashWindow = null;
        }
	if (_serverName != null && _noPanel && 
	    panel.usingIntendedConnection()) {
	    panel.login();
	} else {
	    loginDialog.setVisible(true);
	}
    }

    /** 
     * Parse the command line arguments.  This method first looks for
     * RApp arguments and then puts the remaining arguments into a
     * vector that is passed to the subclass for parsing.
     * <p>
     * Sets the following class fields:
     *      _serverName
     *      _loginName
     *
     * @param args List of command line arguments
     * @param listener Listener to call on success or failure.
     */
    private void parseArgs(String[] args,
			   final ResultListener listener) {
	Vector appArgs = new Vector();
	final ResultEvent ev = new ResultEvent(this);

	// Parse the RApp arguments
	for (int ii = 0; ii < args.length; ii++) {
	    if (args[ii].charAt(0) != '-') {
		// Pass this argument to the application
		appArgs.addElement(args[ii]);
		continue;
	    }
	    
	    switch (args[ii].charAt(1)) {
	    case 'L': // Set the locale.
	    case 'p': // Set the product name.
		// This argument has already been handled by
		// preScanArgs()
		ii++;
		continue;

	    case 's': // Set the name of the server to run the task on.
		if (ii + 1 == args.length) {
		    ev.setReason(errorMsg(
			_rs.getString("RApp.Error.missingServer")));
		    listener.failed(ev);
		    return;
		}
		_serverName = args[ii+1];
		ii++;
		break;
		
	    case 'l': // Set the login name to use on the server
		if (ii + 1 == args.length) {
		    ev.setReason(errorMsg(
			_rs.getString("RApp.Error.missingLogin")));
		    listener.failed(ev);
		    return;
		}
		_loginName = args[ii+1];
		ii++;
		break;

	    case 'n': // Do not bring up the login dialog, if possible
		_noPanel = true;
		break;

	    default:
		// Pass this unrecognized argument to the application.
		appArgs.addElement(args[ii]);
	    }
	}

	setArgs(appArgs, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		listener.succeeded(ev);
	    }

	    public void failed(ResultEvent event) {
		ev.setReason(errorMsg(event.getReason()));
		listener.failed(ev);
	    }
	});
    }

    private ResourceStack getDefaultRS() { 
	ResourceStack rs = new ResourceStack();
	rs.pushBundle(RApp.class.getName() +
		      ResourceStack.BUNDLE_SUFFIX);
	return rs;
    }
    
    
    // Pre scan the argument list for the options that set the locale
    // and product name since we need these options at initialization time.
    private void preScanArgs(String[] args) {
	if (args == null) {
	    return;
	}


	for (int ii = 0; ii < args.length; ii++) {
	    if (args[ii].charAt(0) != '-') {
		// skip this argument; we're looking for options
		continue;
	    }
	    
	    switch (args[ii].charAt(1)) {
	    case 'L': // Set the locale.
		if (ii + 1 == args.length) {
		    // Use a default ResourceStack for error messages
		    // at this stage.
		   
		    _preScanError =
			getDefaultRS().getString("RApp.Error.missingLocale");
		    return;
		}
		_preScanError = setLocale(args[ii+1]);
		if (_preScanError != null) {
		    return;
		}

		ii++;
		break;

	    case 'p': // Set the product name.
		if (ii + 1 == args.length ||
		    args[ii+1].charAt(0) == '-') {
		    _preScanError =
			getDefaultRS().getString(
			    "RApp.Error.missingProductName");
		    return;
		}
		_productName = args[ii+1];
		ii++;
		break;

	    default: // Ignore all other args for now
		break;
	    }
	}

    }

    /**
     * Generate a localized error message using the resource
     * UNKNOWN_OPTION stating that <TT>option</TT>
     * is not recognized by the application
     *
     * @param option The option from the command line that is not
     *               recognized.
     * @return A localized error message.
     */
    public String unknownOptionMsg(String option) {
	return MessageFormat.format(
	    _rs.getString(UNKNOWN_OPTION),
	    new Object[] { option });
    }

    /**
     * Generate a localized error message using the resource
     * TOO_MANY_ARGS stating that there were too
     * many arguments specified on the command line.
     *
     * @return A localized error message.
     */
    public String tooManyArgsMsg() {
	return _rs.getString(TOO_MANY_ARGS);
    }

    // Form an error message by concatenating an error string with the
    // usage string.
    private String errorMsg(String error) {
	return error + NEWLINES + MessageFormat.format(
		_rs.getString("RApp.Error.usageFmt"),
		new Object[] { _appClassName,
			       _rs.getString("RApp.Error.appUsage")});
    }

    /*
     * Parse the locale specifier and set the default locale.
     * 
     * If the specifier is invalid, return an error message in the default
     * locale, otherwise return null.
     *
     * @param localeString The locale as specified on the command line.
     * @return null if the specifier is valid, otherwise an error message.
     */
    private String setLocale(String localeString) {
	boolean error = false;
	String language = "";
	String country = "";
	String variant = "";
	int localeLen = localeString.length();
	
	switch (localeLen) {
	case 8: // locale should look like: ll_CC_variant
	    if (localeString.charAt(5) != '_') {
		error = true;
		break;
	    }
	    variant = localeString.substring(6);
	    // fall through to get country specifier
	    
	case 5: // locale looks like: ll_CC
	    if (localeString.charAt(2) != '_') {
		error = true;
		break;
	    }
	    country = localeString.substring(3,5);
	    // fall through to get language specifier
	    
	case 2: // locale looks like: ll
	    language = localeString.substring(0,2);
	    break;
	    
	case 1: // LANG = C
	    if (localeString.charAt(0) == 'C') {
		language = "C";
		break;
	    }
	    // All other 1-character locales are invalid. Fall through
	    
	default: // Invalid locale
	    // The locale specifier was invalid
	    error = true;
	    break;
	}
	
	if (error) {
	    // Use a default ResourceStack for error messages at this stage.
	    return MessageFormat.format(
		getDefaultRS().getString("RApp.Error.illegalLocale"),
		new Object[] { localeString });
	}
	
	Locale.setDefault(new Locale(language, country, variant));
	
	return null;
    }


    private class SplashWindowFrame extends Frame {
	SplashWindowFrame(URL imageURL, String message, Font font) {
	    super();
	    MediaTracker mt = new MediaTracker(this);
            Image splashIm = Toolkit.getDefaultToolkit().getImage(imageURL);
	    mt.addImage(splashIm, 0);
	    try {
		mt.waitForID(0);
	    } catch(InterruptedException ie){
            }
	    _splashWindow = new SplashWindow(this, splashIm, message, font);
	}
    }

    private class SplashWindow extends Window {
	private Image splashIm;

        private int stringLocation;
        private String message;
        
	SplashWindow(Frame parent, Image splashIm, String message,
                     Font font) {
	    super(parent);
            this.splashIm = splashIm;
            setSize(splashIm.getWidth(null),
                    splashIm.getHeight(null));
	    /* Center the window */
	    Dimension screenDim = 
                Toolkit.getDefaultToolkit().getScreenSize();
	    Rectangle winDim = getBounds();
	    setLocation((screenDim.width - winDim.width) / 2,
			(screenDim.height - winDim.height) / 2);

            stringLocation =
                splashIm.getHeight(null) -
                getFontMetrics(font).getDescent() - 10;
            
            setFont(font);
            this.message = message;
	    setVisible(true);
            paint(getGraphics());
	}

        public void paint(Graphics g) {
            g.drawImage(splashIm,0,0,this);
            g.drawString(message, 10, stringLocation);
	}
    }
}
