//
// RApplet.java
//
//	A base class for sysadm-based applets.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.manager.login.*;

/**
 * RApplet is an abstract base class for sysadm-based Applets.  A
 * sysadm-based Applet presents a user interface element on a web page
 * that when activated launches a sysadm application (typically a
 * subclass of RApp such as TaskManager) in an independent frame.
 * <p>
 * A subclass of RApplet calls its <tt>activate</tt> method when the
 * user makes a gesture such as clicking the mouse.  The first time
 * <tt>activate</tt> is called, it attempts to establish a connection
 * with the server from which the Applet was downloaded, and if
 * successful calls the <tt>launchFrame</tt> method with the resulting
 * HostContext.  Subsequent calls to <tt>activate</tt> bring the
 * RFrame created by <tt>launchFrame</tt> to the front of the window
 * stacking order.
 * <p>
 * If a call to <tt>HostContext.exit</tt> is made, then the connection
 * with the server will be dropped.  A subsequent call to
 * <tt>activate</tt> will result in an attempt to establish a new
 * connection and if successful <tt>launchFrame</tt> will be called
 * again.
 * <p>
 * <tt>HostContext.exit</tt> will be called when the last application
 * window is closed.
 * 
 * @see HostContext
 * @see RApp
 * @see RFrame
 */
public abstract class RApplet extends JApplet {
    private UIContext _uic = new UIContext();
    private RFrame _clientFrame;
    private LoginDialog _loginDialog;
    private HostContext _hc;
    private Cursor _saveCursor;
    private static final String CLASS_NAME = "RApplet";
    private String _productName;

    /**
     * The property <I>RApplet.dialogTitle</I> is a String that is
     * used as the title on dialogs displayed by this applet.
     */
    public static final String DIALOG_TITLE = "RApplet.dialogTitle";

    /**
     * Construct an RApplet.  Initializes internal resources using
     * <tt>appClassName</tt> and <tt>productName</tt>, and interacts
     * with HostContext to provide proper handling for Help buttons.
     * 
     * @param appClassName The package-qualified name of the
     *                     application class.
     *                     ResourceStack.BUNDLE_SUFFIX will be
     *                     appended to this name to generate the
     *                     application ResourceBundle name.
     * @param productName A String containing the
     *                    <A HREF="glossary.html#CLASSPATHRelative">
     *                    CLASSPATH relative</A> name of the product
     *                    containing the package properties file.
     *                    May be <TT>null</TT>.  The value of this
     */
    public RApplet(String appClassName, String productName) {
	_productName = productName;
	getRootPane().putClientProperty("defeatSystemEventQueueCheck", "true");
	ResourceStack rs = _uic.getResourceStack();
	try {
	    rs.pushBundle(appClassName + ResourceStack.BUNDLE_SUFFIX);
	} catch (MissingResourceException ex) {
	    Log.debug(CLASS_NAME, ex.getMessage());
	    Log.debug(CLASS_NAME, SysUtil.stackTraceToString(ex));
	}
	rs.pushPackageBundles(productName);
	RApp.installRhinoLook(rs, true);
	_uic.setDialogTitle(rs.getString(DIALOG_TITLE));
    }

    /**
     * Called when the user activates (typically by clicking the
     * mouse) the Applet.  The first time <tt>activate</tt> is called,
     * it attempts to establish a connection with server from which
     * the web page was downloaded.  If successfull,
     * <tt>launchFrame</tt> will be called.
     * <p>
     * Subsequent calls to <tt>activate</tt> will result in the frame
     * returned by <tt>launchFrame</tt> being brought to the front of
     * the window stacking order (via a call to <tt>toFront</tt>).
     * <p>
     * If <tt>HostContext.exit</tt> is called, then the connection
     * with the server will be dropped and subsequent calls to
     * <tt>activate</tt> will start the process over again.
     * @see RFrame#toFront
     */
    public void activate() {
	// If we already have a client, bring it to the
	// top.  Otherwise, if we have a HostContext start the
	// client.  Otherwise, log in.
	if (_clientFrame != null) {
	    _clientFrame.toFront();
	} else if (_hc != null) {
	    startClient();
	} else {
	    login();
	}
    }

    /**
     * Query whether this RApplet is connected to the server.  A
     * connection is established by calling <tt>activate</tt>.
     * 
     * @return <tt>true</tt> if RApplet is connected, <tt>false</tt>
     *         otherwise.
     */
    public boolean isConnected() {
	return _hc != null;
    }

    /**
     * Called by RApplet after a connection has been established.  The
     * subclass should create and display the frame that the
     * application runs in.  If the RFrame returned by
     * <tt>launchFrame</tt> is closed and then <tt>activate</tt> is
     * called, <tt>launchFrame</tt> will be called again.
     * 
     * @param hc HostContext for server connection.
     * 
     * @return RFrame in which application will run.
     */
    protected abstract RFrame launchFrame(HostContext hc);

    /**
     * Log in to the server.  If successful, start TaskManager.
     */
    private void login() {
	if (_loginDialog != null) {
	    _loginDialog.toFront();
	    return;
	}
	_saveCursor = getCursor();
	setCursor(new Cursor(Cursor.WAIT_CURSOR));
	_loginDialog = new LoginDialog(null, _uic, _productName);
	final RemoteHostPanel hostPanel = _loginDialog.getPanel();
	hostPanel.setHost(getDocumentBase().getHost());
	hostPanel.setHostEditable(false);
	_loginDialog.setResultListener(new ResultListener() {
	    public void succeeded(ResultEvent event) {
		_hc = (HostContext)event.getResult();
		startClient();
	    }
	    public void failed(ResultEvent event) {
		if (_hc != null) {
		    _hc.exit(1);
		} else {
		    hideAll();
		}
	    }
	});
	_loginDialog.setVisible(true);
	_loginDialog.setDefaultCloseOperation(
	    WindowConstants.DISPOSE_ON_CLOSE);
	_loginDialog.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		hideAll();
	    }
	    public void windowOpened(WindowEvent event) {
		if (_saveCursor != null) {
		    setCursor(_saveCursor);
		    _saveCursor = null;
		}
	    }
	});
    }

    /**
     * Start the client.
     */
    private void startClient() {
	Log.assert(_hc != null,
		   "Attempt to start client w/o a HostContext");
	_hc.setExitHandler(new HostContext.ExitHandler() {
	    public void exit(int status) {
		hideAll();
	    }
	});
	_clientFrame = launchFrame(_hc);
	_clientFrame.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		_clientFrame = null;
	    }
	});
    }

    /**
     * Called in response to HostContext exit or login failure; get
     * rid of all of our windows.
     */
    private void hideAll() {
	if (_loginDialog != null) {
	    _loginDialog.setVisible(false);
	    _loginDialog.dispose();
	    _loginDialog = null;
	}
	if (_clientFrame != null) {
	    _clientFrame.setVisible(false);
	    _clientFrame.dispose();
	    _clientFrame = null;
	}
	_hc = null;
    }
}
