//
// RunItemView.java
//
//	Launch an ItemView from the command line.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager;

import java.awt.event.*;
import java.text.*;
import java.util.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;

/**
 * A class that will run an ItemView in a stand alone frame.  This
 * class has a main() function, so it can be started from the command
 * line.
 */
public class RunItemView extends RApp {

    /**
     * A resource <I>RunItemView.Error.missingCategory</i> is a string
     * that is displayed if the category is not specified on the
     * command line.
     */
    static public final String MISSING_CATEGORY = ".Error.missingCategory";

     /**
     * A resource <I>RunItemView.Error.missingItem</i> is a string
     * that is displayed if the item is not specified on the
     * command line.
     */
    static public final String MISSING_ITEM = ".Error.missingItemName";

    static private final String CLASS_NAME = "RunItemView";

    private RFrame _frame;
    private UIContext _uic;
    private ResourceStack _rs;
    private String _category;
    private String _itemName;

    /**
     * Launch an ItemView from the command line.  Additional RApp
     * operands may be specified.
     * <p>
     * <PRE>
     * Usage: java sysadm.ui.RunItemView <category> <item name>
     *
     *	      category
     *	         The package-qualified category containing the item to view
     *	      item name
     *		 The name of the item
     * </PRE>
     *
     * @param args List of command line arguments.
     * @see com.sgi.sysadm.manager.RApp
     */
    static public void main(String[] args) {
	new RunItemView(args).loginAndRun(args);
    }

    /**
     * Constructor.
     *
     */
    public RunItemView(String[] args) {
	super(RunItemView.class.getName(), null, args);
	_uic = getUIContext();
	_rs = _uic.getResourceStack();
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#createFrame
     */
    public RFrame createFrame() {
	_frame = new ItemViewFrame();
	return _frame;
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#launchApp
     */
    protected void launchApp(HostContext hostContext,
			     RAppLaunchListener listener) {
	try {
	    ItemView iv = ItemView.createItemView(hostContext,
						  _category);
	    ((ItemViewFrame)_frame).setItemView(iv);
	    iv.setItem(_itemName);
	    _frame.pack();
	    _frame.setVisible(true);
	    listener.launchSucceeded(
		new RAppLaunchEvent(this, iv.getPanel()));
	}
	catch (ItemViewException ive) {
	    listener.launchFailed(
		new RAppLaunchEvent(this, ive.getMessage()));
	}
    }

    /**
     * Parse the command line arguments.
     * <p>
     * Sets the following class fields:
     *      _category
     *      _itemName
     *
     * @param args List of command line arguments
     * @param listener Listener to notify of success or failure.
     */
    public void setArgs(Vector args, ResultListener listener)
    {
	Enumeration argList = args.elements();
	ResultEvent ev = new ResultEvent(this);

	while (argList.hasMoreElements()) {
	    String arg = (String)argList.nextElement();

	    if (arg.charAt(0) != '-') {
		if (_category == null) {
		    _category = arg;
		} else if (_itemName == null) {
		    _itemName = arg;
		} else {
		    ev.setReason(tooManyArgsMsg());
		    listener.failed(ev);
		    return;
		}
		continue;
	    }

	    ev.setReason(unknownOptionMsg(arg));
	    listener.failed(ev);
	    return;
	}

	if (_category == null) {
	    ev.setReason(
		_rs.getString(CLASS_NAME + MISSING_CATEGORY));
	    listener.failed(ev);
	    return;
	}

	if (_itemName == null) {
	    ev.setReason(_rs.getString(CLASS_NAME + MISSING_ITEM));
	    listener.failed(ev);
	    return;
	}

	listener.succeeded(ev);
    }
}
