//
// RunTask.java
//
//	Launch a Task from the command line.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager;

import java.awt.event.*;
import java.text.*;
import java.util.*;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;

public class RunTask extends RApp {
    private RFrame _frame;
    private UIContext _uic;
    private ResourceStack _rs;
    private String _taskName;
    private TaskLoader _taskLoader;
    private TaskData _dataAttrs = new TaskData();
    private Vector _operands = new Vector();
    private Task _task;

    /**
     * Launch a Task from the command line.  Additional RAapp
     * operands may be specified.
     * <p>
     * <PRE>
     * Usage: java sysadm.manager.RunTask \
     *             [-d key=value] <task name> [operands]
     *	      -d key=value
     *           Set the value of public TaskData attribute 'key'
     *	      task name
     *	         package.taskname to launch.
     *	      operands
     *		 Task specific operands, if any.
     * </PRE>
     * @param args List of command line arguments.
     * @see com.sgi.sysadm.manager.RApp
     */
    static public void main(String[] args) {
	new RunTask(args).loginAndRun(args);
    }

    /**
     * Constructor.
     *
     */
    public RunTask(String[] args) {
	super(RunTask.class.getName(), null, args);
	_uic = getUIContext();
	_rs = _uic.getResourceStack();
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#createFrame
     */
    public RFrame createFrame() {
	_frame = new TaskFrame();
	return _frame;
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#launchApp
     */
    protected void launchApp(HostContext hostContext,
			     final RAppLaunchListener listener) {
	// Create a loader for the Task.
	try {
	    _taskLoader = new TaskLoader(hostContext, _taskName);
	} catch (MissingResourceException exception) {
	    listener.launchFailed(
		new RAppLaunchEvent(this, exception.getMessage()));
	    return;
	}

	// Attempt to load the Task.
	_taskLoader.loadTask(_dataAttrs, _operands, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		_task = (Task)event.getResult();
		_uic.setDialogParent(_task);

		// Tell the TaskFrame to display the loaded task.
		TaskFrame taskFrame = (TaskFrame)_frame;
		taskFrame.setTask(_task);
		taskFrame.addTaskDoneListener(new TaskDoneListener() {
		    public void taskDone(TaskResult result) {
			_frame.dispose();
		    }
		});

		_frame.pack();
		_frame.setVisible(true);
		listener.launchSucceeded(new RAppLaunchEvent(this, _task));
	    }

	    public void failed(ResultEvent event) {
		listener.launchFailed(
		    new RAppLaunchEvent(this, event.getReason()));
	    }
	});
    }

    /**
     * Parse the command line arguments.
     * <p>
     * Sets the following class fields:
     *	    _taskName
     *	    _dataAttrs
     *	    _operands
     *
     * @param args List of command line arguments
     * @param listener Listener to notify of success or failure.
     */
    public void setArgs(Vector args, ResultListener listener)
    {
	Enumeration argList = args.elements();
	ResultEvent ev = new ResultEvent(this);

	while (argList.hasMoreElements()) {
	    String arg = (String)argList.nextElement();

	    if (arg.charAt(0) != '-') {
		if (_taskName == null) {
		    _taskName = arg;
		} else {
		    _operands.addElement(arg);
		}
		continue;
	    }

	    switch (arg.charAt(1)) {
	    case 'd': // Set a TaskData attribute
		if (!argList.hasMoreElements()) {
		    ev.setReason(
			_rs.getString("RunTask.Error.missingTaskDataAttr"));
		    listener.failed(ev);
		    return;
		}

		String parsingError =
		    parseDataAttr((String)argList.nextElement());
		if (parsingError != null) {
		    ev.setReason(parsingError);
		    listener.failed(ev);
		    return;
		}
		break;

	    default:
		ev.setReason(unknownOptionMsg(arg));
		listener.failed(ev);
		return;
	    }
	}

	if (_taskName == null) {
	    ev.setReason(
		_rs.getString("RunTask.Error.missingTaskName"));
	    listener.failed(ev);
	    return;
	}

	listener.succeeded(ev);
    }

    /**
     * Parse a TaskData attribute specification of the form "key=value"
     *
     * @param dataArg Attribute specification following the -D option.
     * @return null if the parsing was successful, otherwise an error message
     *              suitable for posting to the user.
     */
    private String parseDataAttr(String dataArg) {
	int equalSign = dataArg.indexOf('=');
	if (equalSign < 0) {
	    return
		MessageFormat.format(
		    _rs.getString("RunTask.Error.badDataAttrFormat"),
		    new Object [] { dataArg });
	}

	_dataAttrs.setAttr(
	    new Attribute(dataArg.substring(0, equalSign),
			  dataArg.substring(equalSign+1, dataArg.length())));

	return null;
    }
}
