//
// TaskManager.java
//
//	Display a set of tasks for a product.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager;

import com.sgi.sysadm.manager.taskManager.*;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.lang.reflect.*;
import java.text.*;
import java.util.*;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;

/**
 * TaskManager is a User interface intended to be the front end of a
 * sysadm product.  It is customized for a specific product by
 * overriding the generic resources in a file called
 * {product}.TaskManagerP.properties.  See TaskManagerProperties for
 * details about the properties that can be overridden.
 * <P>
 * It is also possible to write subclasses of TaskManagerFrame,
 * TaskManagerAction, and TaskManagerPanel that can be plugged into
 * TaskManager, again by referring to those classes in the
 * TaskManagerP.properites file.
 * <P>
 * XXX more details coming soon to a comment near you.
 */
public class TaskManager extends RApp
                         implements TaskManagerProperties
{
    private RFrame _frame;
    private UIContext _uic;
    private ResourceStack _rs;
    private String _productName;
    private TableOfContents _toc;
    private DisplayArea _displayArea;
    private JSplitPane _tmPane;
    private HostContext _hc;

    static private String _class =
	ResourceStack.getClassName(TaskManager.class);

    /**
     * Display a set of tasks for a product.
     * <p>
     * Usage: java sysadm.manager.TaskManager -p <product name>
     *	      product name
     *	         package.productname to launch.
     *
     * @param args List of command line arguments.
     */
    static public void main(String[] args) {
	new TaskManager(args).loginAndRun(args);
    }

    /**
     * Constructor.
     * <P>
     * This constructor should be called when TaskManager is started
     * from the command line.  The product name must be specified via
     * the -p option.
     */
    public TaskManager(String[] args) {
	super(TaskManager.class.getName(), null, args);
	_uic = getUIContext();
	_rs = _uic.getResourceStack();
    }

    /**
     * Constructor.
     * <P>
     * This constructor should be called when TaskManager is being launched
     * from another application that has already logged in.  The
     * caller should next call initApp() and then run() to start the
     * application.
     *
     * @param productName <A HREF="glossary.html#CLASSPATHRelative">
     *                    CLASSPATH relative</A> path where
     *                    TaskManagerP.properties has been defined.
     */
    public TaskManager(String productName) {
	super(TaskManager.class.getName(), productName, (String[])null);
	_uic = getUIContext();
	_rs = _uic.getResourceStack();
	_productName = getProductName();
	loadProductResources();
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#createFrame
     */
    public RFrame createFrame() {
	_frame = new RFrame();
	return _frame;
    }

    /**
     * Override RApp.run in order to implement run-once behavior.
     * @see com.sgi.sysadm.manager.RApp#run
     */
    public void run(final HostContext hc, RAppLaunchListener listener) {
	_hc = hc;

	if (_frame == null) {
	    Log.fatal("initApp() must be called before run()");
	}

	String key = getClass().getName();
	if (_productName != null) {
	    key += key + "." + _productName;
	}
	Vector vec = hc.getClients(key);
	if (vec.size() > 0) {
	    ((HostContext.Client)vec.elementAt(0)).getFrame().toFront();
	    _frame.dispose();
	    listener.launchAlreadyRunning(
		new RAppLaunchEvent(this, null));
	    return;
	}
	final HostContext.Client taskManagerClient =
	    new HostContext.Client(_frame, _uic);
	hc.registerClient(taskManagerClient, key);
	_frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
	_frame.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		hc.unregisterClient(taskManagerClient);
	    }
	});

	launchApp(hc, listener);
    }

    /**
     * @see com.sgi.sysadm.manager.RApp#launchApp
     */
    protected void launchApp(final HostContext hc,
			     final RAppLaunchListener listener) {
	final JPanel panel = new JPanel();
	_uic.setDialogParent(panel, new ResultListener() {
	    public void failed(ResultEvent e) {
	    }

	    public void succeeded(ResultEvent e) {
		// Start the TaskManager title renderer or set the window
		// title based on a property.
		String className = null;
		try {
		    String[] initPlugins = _rs.getStringArray(INIT_PLUGINS);
		    Class initClass;
		    for (int ii = 0; ii < initPlugins.length; ii++) {
			className =  initPlugins[ii];
			initClass = SysUtil.forName(className);
			TaskManagerPlugin plugin =
			    TaskManagerUtil.loadPlugin(hc, _uic,
						       initClass,
						       _productName);
			if (!(plugin instanceof TaskManagerInitPlugin)) {
			    Log.error(_class, className +
				      " is not a TaskManagerInitPlugin");
			}
		    }
		} catch (MissingResourceException ex) {
		    // The init plugins are optional
		} catch (ClassNotFoundException exception) {
		    Log.error(_class,
			      "Missing TaskManagerInitPlugin class: " +
			      className);
		}

		TaskManagerTitleRenderer titleRenderer = null;
		String titleRendererName = _rs.getString(TITLE_RENDERER, null);

		if (titleRendererName != null) {
		    Class rendererClass;
		    try {
			rendererClass = SysUtil.forName(titleRendererName);
			TaskManagerPlugin plugin =
			    TaskManagerUtil.loadPlugin(hc, _uic,
						       rendererClass,
						       _productName);
			if (plugin instanceof TaskManagerTitleRenderer) {
			    titleRenderer = (TaskManagerTitleRenderer)plugin;
			} else {
			    Log.error(_class, titleRendererName +
				      " is not a TaskManagerTitleRenderer");
			}
		    } catch (ClassNotFoundException exception) {
			Log.error(_class,
				  "Missing TaskManagerTitleRenderer class: " +
				  titleRendererName);
		    }
		}

		if (titleRenderer == null) {
		    _frame.setTitle(MessageFormat.format(
			_rs.getString(FRAME_TITLE),
			new Object[] { hc.getHostName() }));
		} else {
		    titleRenderer.startRendering(_frame);
		}

		boolean noMenu = _rs.getBoolean(TASKMANAGER_NOMENU);

		// set up the menu bar
		JMenuBar menuBar = null;
		if (! noMenu) {
		    menuBar = new JMenuBar();
		    _frame.setJMenuBar(menuBar);
		}

		panel.setBorder(new EmptyBorder(_rs.getPixels(MARGIN_TOP),
						_rs.getPixels(MARGIN_LEFT),
						_rs.getPixels(MARGIN_BOTTOM),
						_rs.getPixels(MARGIN_RIGHT)));
		panel.setLayout(new BorderLayout());
		panel.setPreferredSize(new Dimension(_rs.getPixels(WIDTH),
						     _rs.getPixels(HEIGHT)));

		_displayArea = new DisplayArea(hc, _uic);
		_toc = new TableOfContents(_uic, _displayArea);
		_tmPane = new
		    JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
			       true,
			       _toc,
			       _displayArea);
		_tmPane.setDividerSize(_rs.getPixels(DIVIDER_WIDTH));

		// Set up the Options menu
		JMenu optMenu = null;
		if (! noMenu) {
		    optMenu =
			new JMenu(_rs.getString(TASKMANAGER_OPTIONSMENU +
						".label"));
		    setMnemonic(optMenu,
				TASKMANAGER_OPTIONSMENU + ".mnemonic");
		    menuBar.add(optMenu);
		}

		// Create the button bar
		JPanel buttonBar = new JPanel();

		// Add user defined buttons
		String[] buttonList;
		try {
		    buttonList = _rs.getStringArray(BUTTON_ITEM);

		    for (int ii = 0; ii < buttonList.length; ii++) {
			final String target;
			try {
			    target = _rs.getString(
				BUTTON_ITEM + ii + BUTTON_TARGET);
			} catch (MissingResourceException exception) {
			    Log.error(_class, "Missing button target property: " +
				      BUTTON_ITEM + ii + BUTTON_TARGET);
			    continue;
			}

			final Class targetClass;
			try {
			    targetClass = SysUtil.forName(target);
			} catch (ClassNotFoundException exception) {
			    Log.error(_class,
				      "Missing button class: " + target);
			    continue;
			}
			JButton button = new JButton(buttonList[ii]);
			ActionListener pluginLoader = new ActionListener() {
			    public void actionPerformed(ActionEvent event) {
				TaskManagerPlugin plugin =
				    TaskManagerUtil.loadPlugin(hc, _uic,
							       targetClass,
							       _productName);
				if (plugin == null) {
				    _uic.postWarning(
					TaskManagerUtil.getLoadPluginError());
				}
			    }
			};
			button.addActionListener(pluginLoader);
			buttonBar.add(button);

			// add "button" to Options menu
			if (! noMenu) {
			    JMenuItem menuItem = new JMenuItem(buttonList[ii]);
			    setMnemonic(menuItem,
					BUTTON_ITEM + ii + BUTTON_TARGET +
					MENU_MNEMONIC);
			    menuItem.addActionListener(pluginLoader);
			    optMenu.add(menuItem);
			}
		    }
		} catch (MissingResourceException exception) {
		    // User defined buttons are optional.
		}

		// Add the close button last.
		JButton closeButton =
		    new JButton(_rs.getString(CLOSE_BUTTON_LABEL));
		closeButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			_frame.dispose();
		    }
		});
		buttonBar.add(closeButton);

		// Add the close button to the Options menu.
		if (! noMenu) {
		    JMenuItem closeItem =
			new JMenuItem(_rs.getString(CLOSE_BUTTON_LABEL));
		    setMnemonic(closeItem,
				CLOSE_BUTTON_LABEL + MENU_MNEMONIC);
		    closeItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
			    _frame.dispose();
			}
		    });
		    optMenu.addSeparator();
		    optMenu.add(closeItem);
		}

		panel.add(_tmPane, BorderLayout.CENTER);
		panel.add(buttonBar, BorderLayout.SOUTH);

		_frame.setContentPane(panel);
		_frame.pack();
		_tmPane.setDividerLocation((double)_rs.getFloat(SPLIT_RATIO));
		_frame.setVisible(true);

		String[] productAttributes =
		    _rs.getStringArray(PRODUCT_ATTRIBUTES, null);

		if (productAttributes != null) {
		    _uic.busy();
		    loadProductAttributes(productAttributes, 0, hc, _uic,
					  new TaskData(),
					  new ResultListener() {
			public void succeeded(ResultEvent event) {
			    _uic.notBusy();
			    listener.launchSucceeded(
				new RAppLaunchEvent(TaskManager.this, panel));
			}

			public void failed(ResultEvent event) {
			    _uic.notBusy();
			    listener.launchFailed(
				new RAppLaunchEvent(TaskManager.this,
						    event.getReason()));
			}
		    });
		} else {
		    listener.launchSucceeded(
			new RAppLaunchEvent(TaskManager.this, panel));
		}

		// add Help menu to menubar last
		if (! noMenu) {
		    buildHelpMenu(menuBar);
		}
	    }});
    }

    /**
     * Parse the command line arguments.
     * <p>
     * Sets the following class fields:
     *	    _productName
     *
     * @param args List of command line arguments.
     * @param listener Listener to notify of success or failure.
     */
    public void setArgs(Vector args, ResultListener listener)
    {
	_productName = getProductName();

	Enumeration argList = args.elements();
	ResultEvent ev = new ResultEvent(this);

	while (argList.hasMoreElements()) {
	    String arg = (String)argList.nextElement();

	    if (arg.charAt(0) != '-') {
		ev.setReason(tooManyArgsMsg());
		listener.failed(ev);
		return;
	    }

	    ev.setReason(unknownOptionMsg(arg));
	    listener.failed(ev);
	    return;
	}

	if (_productName == null) {
	    ev.setReason(
		_rs.getString("TaskManager.Error.missingProductName"));
	    listener.failed(ev);
	    return;
	}

	loadProductResources();
	listener.succeeded(ev);
    }

    /**
     * Loads the productAttributes for the products specified
     *
     * @param products The list of products for which to load the
     *                 product Attributes
     * @param numProduct The index into the products array where this
     *                   method should start looking.
     * @param hc The HostContext to use.
     * @param uic The UIContext to use.
     * @param taskData The TaskData in which to place the product
     *                 attributes.
     * @param listener The listener to notify about the success of the
     *                 getProductAttributes() calls.
     */
    private void loadProductAttributes(final String[] products,
				       final int numProduct,
				       final HostContext hc,
				       final UIContext uic,
				       final TaskData taskData,
				       final ResultListener listener) {
	hc.getProductAttributes(
	    products[numProduct], taskData, uic,
	    HostContext.DONT_REFRESH_ATTRIBUTES,
	    null,
	    new ResultListener() {
	    	public void succeeded(ResultEvent event) {
		    if (numProduct == products.length-1) {
		    	listener.succeeded(event);
		    } else {
		    	loadProductAttributes(products, numProduct+1, hc, uic,
					      taskData, listener);
		    }
	    	}
	    	public void failed(ResultEvent event) {
		    listener.failed(event);
	    	}
	});
    }

    private void loadProductResources() {
	try {
	    _rs.pushBundle(_productName + ".TaskManager" +
			   ResourceStack.BUNDLE_SUFFIX);
	} catch (MissingResourceException exception) {
	    Log.error("TaskManager", MessageFormat.format(
		_rs.getString("TaskManager.Error.cantFindProduct"),
		new Object[] { _productName }));
	}

	// Reset the dialog title now that we have product-specific
	// resources.
	String title = _rs.getString(RApp.DIALOG_TITLE);
	_uic.setDialogTitle(title);
    }

    /**
     * Add "Help" menu to our menu bar.  Help menu items are
     * completely specified in the properties file, so the writer can
     * control them.
     *
     * @param menuBar Menu bar to add "Help" menu to.
     */
    private void buildHelpMenu(JMenuBar menuBar) {
	JMenu menu;
	try {
	    menu = new JMenu(_rs.getString(TASKMANAGER_HELPMENU + ".label"));
	    setMnemonic(menu, TASKMANAGER_HELPMENU + MENU_MNEMONIC);
	} catch (MissingResourceException ex) {
	    return;
	}
	menuBar.add(Box.createHorizontalGlue());
	menuBar.add(menu);

	int ii = 0;
	while (true) {
	    try {
		final String itemPrefix = TASKMANAGER_HELPMENU_ITEM + ii + ".";
		JMenuItem item = new JMenuItem(
		    _rs.getString(itemPrefix + "label"));
		setMnemonic(item, itemPrefix + "mnemonic");
		item.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent ev) {
			_hc.help(_rs.getString(HostContext.HELP_SET),
                                 _rs.getString( itemPrefix + "helpKey"));
		    }
		});
		menu.add(item);
	    } catch (MissingResourceException ex) {
		break;
	    }
	    ii++;
	}
    }

    /**
     * Set the mnemonic for a menu item.  A mnemonic is an int, but we
     * don't want to ask people to specify integers in resource files
     * for mnemonics, so we get the mnemonic as a string and use the
     * int value of the first character.
     *
     * @param menuItem The menu item to set mnemonic for.
     * @param mnemonicResource resource string for mnemonic.
     */
    private void setMnemonic(JMenuItem menuItem, String mnemonicResource) {
// Comment out mnemonics for now since they don't work on Irix.
// 	String mnemonic = _rs.getString(mnemonicResource);
// 	if (mnemonic.length() > 0) {
// 	    menuItem.setMnemonic(mnemonic.charAt(0));
// 	}
    }
}
