//
// HostPanelEvent.java
//
//	HostPanel actions.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.login;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.comm.*;
import java.util.*;

/**
 * HostPanelEvent carries notification information about an action
 * which has occurred in a HostPanel.  Clients receive HostPanelEvents
 * via the HostPanelListener interface.
 * <p>
 * A <b>SUCCEEDED</b> HostPanelEvent is sent when a Connection has been
 * established.  A HostContext representing the Connection can be
 * retrieved by calling HostPanelEvent.getHostContext().
 * <p>
 * A <b>CANCELLED</b> HostPanelEvent is sent when the user presses the
 * Cancel button or a fatal error occurs.  The application should exit
 * in this case.
 * <p>
 * A <b>RETRY</b> HostPanelEvent is sent when the Connection is
 * disrupted and an attempt to login using the previous parameters
 * specified by the user has failed.  The application should make the
 * HostPanel visible again  and expect to receive subsequent
 * HostPanelEvents.
 * <p>
 * A <b>HELP</b> HostPanelEvent is sent when the user presses the Help
 * button.
 */
public class HostPanelEvent extends EventObject {

    // Event types.
    public static final int SUCCEEDED = 1;
    public static final int CANCELLED = 2;
    public static final int SHOW = 3;
    public static final int HIDE = 4;
    public static final int HELP = 5;
    public static final int ERROR = 6;

    private int _type = 0;
    private HostContext _context = null;
    private ConnectionEvent _connEvent = null;

    /**
     * Construct a HostPanelEvent.
     * 
     * @param source Source of this event.
     * @param type <b>SUCCEEDED</b>, <b>CANCELLED</b>, <b>RETRY</b>,
     *             or <b>HELP</b>.
     */
    public HostPanelEvent(Object source, int type) {
	super(source);
	_type = type;
	Log.assert(SUCCEEDED <= type && type <= ERROR,
		   "Invalid HostPanelEvent type");
    }

    /**
     * Construct a HostPanelEvent of type <b>SUCCEEDED</b>.
     * 
     * @param source Source of this event.
     * @param context HostContext for a successfully established
     *                Connection.
     */
    public HostPanelEvent(Object source, HostContext context) {
	super(source);
	_type = SUCCEEDED;
	_context = context;
    }

    /**
     * Construct a HostPanelEvent of type <b>ERROR</b>.
     * 
     * @param source Source of this event.
     * @param connEvent ConnectionEvent containing more detail about
     *                  the error.
     */
    public HostPanelEvent(Object source, ConnectionEvent connEvent) {
	super(source);
	_type = ERROR;
	_connEvent = connEvent;
    }

    /**
     * Get the HostContext from a <b>SUCCEEDED</b> HostPanelEvent.
     * 
     * @return HostContext.
     */
    public HostContext getHostContext() {
	Log.assert(_type == SUCCEEDED && _context != null,
		   "Attempt to getHostContext() from wrong event");
	return _context;
    }

    /**
     * Get the ConnectionEvent from a <b>ERROR</b> HostPanelEvent.
     * 
     * @return ConnectionEvent.
     */
    public ConnectionEvent getConnectionEvent() {
	Log.assert(_type == ERROR && _connEvent != null,
		   "Attempt to getConnectionEvent from wrong event");
	return _connEvent;
    }
}
