//
// LoginDialog.java
//
//	Dialog for logging in to the server.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.login;

import java.awt.*;
import java.awt.event.*;
import java.text.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;

/**
 * LoginDialog encapsulates the logic of prompting the user for login
 * parameters and logging in to the server.  If the connection gets
 * disrupted, LoginDialog also handles asking the user if a retry is
 * desired and prompting again if the password changes.
 */
public class LoginDialog extends RDialog {
    /**
     * The resource <i>LoginDialog.title</i> is the String displayed
     * in the title bar of the login dialog.
     */
    public static final String DIALOG_TITLE = "LoginDialog.title";

    /**
     * The resource <i>LoginDialog.Error.connection</i> is the String
     * displayed in the error dialog posted when a connection error
     * occurs.
     */
    public static final String CONNECTION_ERROR = 
	                                     "LoginDialog.Error.connection";

    /**
     * The resource <i>LoginDialog.fatalErrorFormat</i> is a String
     * passed to Message.format along with one String argument.  The
     * format string contains a localized version of "Fatal Error {0}".
     * {0} will get replaced with a localized message detailing the
     * cause of the fatal error.
     */
    public static final String FATAL_ERROR_FORMAT =
                       "LoginDialog.fatalErrorFormat";

    /**
     * The resource <i>LoginDialog.helpKey</i> is a String that
     * is passed to HostContext.help().  It it should be set to the
     * name of the help key that contains info about how to use the
     * LoginDialog.
     */
    public static final String HELP_KEY = "LoginDialog.helpKey";

    private RemoteHostPanel _panel;
    private UIContext _uic;
    private ResultListener _listener;
    private boolean _checkedConnection = false;
    private boolean _inputBlocked;

    /**
     * Construct a LoginDialog.
     * 
     * @param parentFrame Owner of this dialog.
     * @param uic UIContext for posting dialogs and getting resources.
     */
    public LoginDialog(Frame owner, UIContext uic, String productName) {
	super(owner);
//	setResizable(false);
	_uic = uic;
	_panel = new RemoteHostPanel(productName);
	setTitle(_panel._rs.getString(DIALOG_TITLE));
	_uic.setDialogParent(_panel);
	_panel.addHostPanelListener(new HostPanelListener() {
	    public void hostConnectionSucceeded(HostPanelEvent event) {
		HostContext hc = event.getHostContext();
		_uic.setHostContext(hc);
		_listener.succeeded(new ResultEvent(LoginDialog.this, hc));
	    }
	    public void hostConnectionError(HostPanelEvent event) {
		String message = MessageFormat.format(
		    _panel._rs.getString(FATAL_ERROR_FORMAT),
		    new Object[] {event.getConnectionEvent().getErrorString() });
		_uic.postGlobalError(message, new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			_listener.failed(new ResultEvent(LoginDialog.this));
		    }
		});
	    }
	    public void hostPanelShow(HostPanelEvent event) {
		setVisible(true);
	    }
	    public void hostPanelHide(HostPanelEvent event) {
		setVisible(false);
	    }
            public void hostPanelHelp(HostPanelEvent event) {
                HostContext.help(_panel._rs.getString(HostContext.HELP_SET),
                                 _panel._rs.getString(HELP_KEY));
            }
	    public void hostPanelCancel(HostPanelEvent event) {
		_listener.failed(new ResultEvent(LoginDialog.this));
	    }
	});
	setDefaultCloseOperation(DISPOSE_ON_CLOSE);
	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent event) {
		if (isShowing()) {
		    _listener.failed(new ResultEvent(LoginDialog.this));
		}
	    }
	});
	setContentPane(_panel);
	pack();
    }

    /**
     * Get the RemoteHostPanel that prompts the user for login
     * parameters.
     * 
     * @return RemoteHostPanel
     */
    public RemoteHostPanel getPanel() {
	return _panel;
    }

    /**
     * Set the ResultListener that will get notified when a connection
     * is successfully established.  The getResult() method on the
     * ResultEvent passed to listener.succeeded() will return the
     * HostContext for the Connection.
     * <p>
     * listener.succeeded() will only be called once.
     * listener.failed() will be called if the cancel button is
     *              pressed, or if the connection fails completely.
     *
     * @param listener Gets informed when a connection is established.
     */
    public void setResultListener(ResultListener listener) {
	_listener = listener;
    }

    /**
     * Override base class setVisible in order to block input to our
     * parent and center ourselves with respect to our parent.
     * 
     * @param visible true to make us visible, false otherwise.
     */
    public void setVisible(final boolean visible) {
	// We need to do our stuff from an invokeLater because of our
	// test of _uic.getDialogParent().  _uic.setDialogParent()
	// itself does an invokeLater(), so that
	// _uic.getDialogParent() does not return the value passed to
	// _uic.setDialogParent() right away.
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		Log.assert(_listener != null,
			   "Must setResultListener before calling setVisible");

		if (_uic.getDialogParent() != _panel && visible) {
		    _uic.blockInput(true);
		    _inputBlocked = true;
		}

		if (!visible && _inputBlocked) {
		    _uic.blockInput(false);
		    _inputBlocked = false;
		}

		if (visible) {
		    JComponent parent = _uic.getDialogParent();
		    // It's OK if parent is NULL: that puts us in the middle
		    // of the screen.
		    setLocationRelativeTo(parent);	    
		}

		makeVisible(visible);

		if (visible && !_checkedConnection) {
		    _panel.checkConnectionMethod();
		    _checkedConnection = true;
		}
	    }
	});
    }

    /**
     * Workaround the fact that there's no way in Java to call
     * LoginDialog.super.setVisible() from within a nested class.
     * 
     * @param visible whether or not we should be visible.
     */
    public void makeVisible(boolean visible) {
	super.setVisible(visible);
    }
}
