//
// RemoteHostPanel.java
//
//	UI for logging into a remote system by connecting to sysadmd.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.login;

import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.proxy.comm.*;
import com.sgi.sysadm.proxy.ui.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.Event.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;
import java.util.*;

/**
 * RemoteHostPanel presents a UI for logging into a remote system for doing
 * system administration.  The client of RemoteHostPanel must add a
 * HostPanelListener to get notification of HostPanelEvents.
 * <p>
 * All of the default properties for this class are in
 * com.sgi.sysadm.ui.SysadmUIP.properties, and
 * the values can be overridden by any resource files in the
 * ResourceStack passed to the constructor.
 * If this class is created by RApp, then the products's PackageP file,
 * where the product given by the -p flag in the args to the RApp,
 * will be on the ResourceStack.
 */
public class RemoteHostPanel extends JPanel {

    /**
     * The resource <i>RemoteHostPanel.textFieldWidth</i> is an
     * Integer specifying the width of the text fields.
     */
    public static final String TEXT_FIELD_WIDTH =
	"RemoteHostPanel.textFieldWidth";

    /**
     * The resource <i>RemoteHostPanel.dynamicSize</i> is a boolean
     * that detemine whether to use dynamic sizing for this component
     * or not.  If true, then it is only necessary to specify the
     * resource described in WIDTH.  If false, you must specify both
     * WIDTH and HEIGHT.
     */
    public static final String DYNAMIC_SIZE = "RemoteHostPanel.dynamicSize";

    /**
     * The resource <i>RemoteHostPanel.width</i> is an int
     * specifying the width, in points, of the RemoteHostPanel
     */
    public static final String WIDTH = "RemoteHostPanel.width";

    /**
     * The resource <i>RemoteHostPanel.height</i> is an int
     * specifying the height, in points, of the RemoteHostPanel
     */
    public static final String HEIGHT = "RemoteHostPanel.height";

    /**
     * The resource <i>RemoteHostPanel.title</i> is a String specifying
     * the title of the RemoteHostPanel component.
     */
    public static final String TITLE = "RemoteHostPanel.title";

    /**
     * The resource <i>RemoteHostPanel.icon</i> is a String specifying
     * the name of the icon to use.
     */
    public static final String ICON = "RemoteHostPanel.icon";

    /**
     * The resource <i>RemoteHostPanel.introText</i> is a String specifying
     * the introText to be used on the RemoteHostPanel.  You can
     * include any HTML markup that RichTextArea understands.
     */
    public static final String INTRO_TEXT = "RemoteHostPanel.introText";

    /**
     * The resource <i>RemoteHostPanel.hostLabel</i> is the label
     * String for the host text field.
     */
    public static final String HOST_LABEL = "RemoteHostPanel.hostLabel";

    /**
     * The resource <i>RemoteHostPanel.loginLabel</i> is the label
     * String for the login text field.
     */
    public static final String LOGIN_LABEL = "RemoteHostPanel.loginLabel";

    /**
     * The resource <i>RemoteHostPanel.passwordLabel</i> is the label
     * String for the password text field.
     */
    public static final String PASSWORD_LABEL =
	"RemoteHostPanel.passwordLabel";

    /**
     * The resource <i>RemoteHostPanel.proxyPortLabel</i> is the label
     * String for the proxy port text field.  This textfield appears
     * only when the user has selected the proxy item on the
     * connection combobox.
     */
    public static final String PROXY_PORT_LABEL =
	"RemoteHostPanel.proxyPortLabel";

    /**
     * The resource <i>RemoteHostPanel.proxyHostLabel</i> is the label
     * String for the proxy host text field.  This textfield appears
     * only when the user has selected the proxy item on the
     * connection combobox.
     */
    public static final String PROXY_HOST_LABEL =
	"RemoteHostPanel.proxyHostLabel";

    /**
     * The resource <i>RemoteHostPanel.remoteShellLabel</i> is the
     * label String for the remote shell text field.  This textfield
     * appears only when the user has selected the remote shell item
     * on the connection combobox.
     */
    public static final String REMOTE_SHELL_LABEL =
	"RemoteHostPanel.remoteShellLabel";

    /**
     * The resource <i>RemoteHostPanel.connectionTypeLabel</i> is the
     * label String for the connection combobox.
     */
    public static final String CONNECTION_TYPE_LABEL =
	"RemoteHostPanel.connectionTypeLabel";

    /**
     * The resource <i>RemoteHostPanel.connectionType.tcpmux</i> is
     * the label String for the direct item in the connection
     * combobox.
     */
    public static final String CONNECTION_TYPE_TCPMUX =
	"RemoteHostPanel.connectionType.tcpmux";

    /**
     * The resource <i>RemoteHostPanel.connectionType.remoteshell</i>
     * is the label String for the remote shell item in the connection
     * combobox.
     */
    public static final String CONNECTION_TYPE_REMOTESHELL =
	"RemoteHostPanel.connectionType.remoteshell";

    /**
     * The resource <i>RemoteHostPanel.connectionType.proxy</i> is the
     * label String for the proxy item in the connection combobox.
     */
    public static final String CONNECTION_TYPE_PROXY =
	"RemoteHostPanel.connectionType.proxy";

    /**
     * The resource <i>RemoteHostPanel.okButtonLabel</i> is the label
     * string for the "OK" button.
     */
    public static final String OK_BUTTON_LABEL =
	"RemoteHostPanel.okButtonLabel";

    /**
     * The resource <i>RemoteHostPanel.cancelButtonLabel</i> is the
     * label string for the "Cancel" button.
     */
    public static final String CANCEL_BUTTON_LABEL =
	"RemoteHostPanel.cancelButtonLabel";

    /**
     * The resource <i>RemoteHostPanel.helpButtonLabel</i> is the
     * label string for the "Cancel" button.
     */
    public static final String HELP_BUTTON_LABEL =
	"RemoteHostPanel.helpButtonLabel";

    /**
     * The resource <i>RemoteHostPanel.Question.restart</i> is the
     * String that gets posted to the user in a question dialog when
     * the connection is disrupted.
     */
    public static final String QUESTION_RESTART =
	"RemoteHostPanel.Question.restart";

    /**
     * The resource <i>RemoteHostPanel.restartTield</i> is the String
     * used in the title bar of the question dialog that asks the user
     * if connection restart is desired.
     */
    public static final String RESTART_TITLE = "RemoteHostPanel.restartTitle";

    /**
     * The resource <i>RemoteHostpanel.errorTitle</i> is the
     * String that gets displayed in the title bar of error dialogs.
     */
    public static final String ERROR_TITLE = "RemoteHostPanel.errorTitle";

    /**
     * The resource <i>RemoteHostpanel.warningTitle</i> is the
     * String that gets displayed in the title bar of warning dialogs.
     */
    public static final String WARNING_TITLE = "RemoteHostPanel.warningTitle";

    /**
     * The resource <i>RemoteHostPanel.Error.login</i> is the String
     * that gets posted to the user when a login error occurs.
     */
    public static final String ERROR_LOGIN = "RemoteHostPanel.Error.login";

    /**
     * The resource <i>RemoteHostPanel.Error.version</i> is the String
     * that gets posted to the user when a login error occurs.
     */
    public static final String ERROR_VERSION = "RemoteHostPanel.Error.version";

    /**
     * The resource <i>RemoteHostPanel.loginTitle</i> is the String
     * that gets appears as the title of the login dialog
     */
    public static final String LOGIN_TITLE = "RemoteHostPanel.loginTitle";

    /**
     * The resource <i>RemoteHostPanel.missingProduct</i> is the
     * String that is displayed if the product being administered in
     * not installed on the server.
     */
    public static final String MISSING_PRODUCT =
	"RemoteHostPanel.missingProduct";

    /**
     * The resource <i>RemoteHostPanel.productVersionMismatch</i> is the
     * String that is displayed if the product being administered has
     * a version mismatch between the server and the client
     */
    public static final String PRODUCT_VERSION_MISMATCH =
	"RemoteHostPanel.productVersionMismatch";

    /**
     * The resource <i>ServerInstName</i> is a String representing
     * Inst's short name of the server portion of the product.
     */
    public static final String SERVER_INST_NAME = "ServerInstName";

    /**
     * The resource <i>ClientInstName</i> is a String representing
     * Inst's short name of the client portion of the product.
     */
    public static final String CLIENT_INST_NAME = "ClientInstName";

    /**
     * The resource <i>ProductName</i> is the String representing the
     * name of the product.
     */
    public static final String PRODUCT_NAME = "ProductName";

    /**
     * The resource <i>ProductVersion</i> is the String representing
     * the client's version of the product.
     */
    public static final String REQUIRED_PRODUCT_VERSION = "ProductVersion";

    // Task data attribute names.
    private static final String HOST = "host";
    private static final String LOGIN = "login";
    private static final String PASSWORD = "password";
    private static final String PROXYHOST = "proxyhost";
    private static final String PROXYPORT = "proxyport";
    private static final String REMOTESHELLCMD = "remoteshell";

    // Connection methods
    private static final String TCPMUX      = "tcpmux";
    private static final String PORTFWD     = "proxy:";
    private static final String REMOTESHELL = "remoteshell:";

    /**
     * The resource <i>RemoteHostPanel.Warning.unintendedConnectionMethod</i>
     * is the String displayed when a malformed Connection.method property
     * is read.   A warning is posted so the user knows that a fallback
     * connection method is being used to send data across the network.
     */
    public static final String FALLBACK_CONNECTION =
	"RemoteHostPanel.Warning.unintendedConnectionMethod";

    private static final String CLASS_NAME = "RemoteHostPanel";

    interface Notifier {
	public void notify(HostPanelListener listener);
    }

    // _rs has package visibility so that LoginDialog can use it.
    private TaskData _td = new TaskData();
    private Vector _listeners = new Vector();
    private ServerConnectionFactory _factory = null;
    private Authenticator _auth = null;
    private Connection _conn;
    private UIContext _uic = new UIContext(this);
    ResourceStack _rs = _uic.getResourceStack();

    private boolean _trying = false;
    private RTextField _hostText;
//    private RTextField _proxyPortText;
//    private RTextField _proxyHostText;
//    private RTextField _remoteShellText;
    private RPasswordField _passwordText;
    private RTextField _loginText;
//     private JComboBox _connType;
//     private Object  _tcpmuxItem;
//     private Object  _remoteShellItem;
//     private Object  _proxyItem;
    private JLabel _hostLabel;
    private boolean _busy = false;
    private boolean _connEstablished = false;
    private JComponent _dialogParent;
//    private boolean _usingIntendedConnection = false;
    private boolean _hostEditable = true;
    private ConnectionErrorListener _errorListener;
    private ConnectionDisruptedListener _disruptedListener;
    private boolean _ignoreInput = false;
    private String _productName;

     /**
     * The version of the Rhino protocol that this GUI requires.  This
     * number should be updated each time this product is released.
     */
    private static final String REQUIRED_VERSION = "1.4";

    private class HostPanelAuthListener implements AuthListener {

	private SysadmProxy _proxy;

	public void productVersionSucceeded() {
	    if (_busy) {
		_uic.notBusy();
		_busy = false;
	    }
	    if (isShowing()) {
		requestHide();
	    }
	    _conn.notifyStarted();
	    if (!_connEstablished) {
		_connEstablished = true;
		notifySucceeded();
	    }
	}

	public void versionSucceeded() {
	    final String version =
		_rs.getString(REQUIRED_PRODUCT_VERSION);
	    _proxy.checkProductVersion(_productName,
				       version, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    productVersionSucceeded();
		}

		public void failed(ResultEvent event) {
		    String serverInstName =
			_rs.getString(SERVER_INST_NAME);
		    String clientInstName =
			_rs.getString(CLIENT_INST_NAME);
		    String productName = _rs.getString(PRODUCT_NAME);
		    if (event.getResult().equals("")) {
			postError(MessageFormat.format(
			    _rs.getString(MISSING_PRODUCT),
			    new String[] {productName, serverInstName,
					      version}),
				  true);
		    } else {
			postError(MessageFormat.format(
			    _rs.getString(PRODUCT_VERSION_MISMATCH),
			    new String[] {productName, clientInstName, version,
					      (String)event.getResult()}),
				  true);
		    }
		}
	    });
	}

	/**
	 * Called when authentication succeeds.
	 *
	 * @param event
	 */
	public void authSucceeded(AuthEvent event) {
	    // If _conn is null, then this is the first time the
	    // connection has succeeded and the ResultListener created by
	    // connect() will take care of notification.  Otherwise this
	    // is a retry, and we need to do the notification ourselves.

	    _proxy = SysadmProxy.get(_conn);
	    _proxy.checkVersion(REQUIRED_VERSION, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    versionSucceeded();
		}
		public void failed(ResultEvent event) {
		    Vector versions = (Vector)event.getResult();
		    StringBuffer versionsStr = new StringBuffer();
		    String separator = _rs.getString(ERROR_VERSION +
						     ".separator");
		    String conjunction = _rs.getString(ERROR_VERSION +
						     ".conjunction");

		    for (int ii = 0; ii < versions.size(); ii++) {
			if (ii != 0) {
			    if (versions.size() != 2) {
				versionsStr.append(separator);
			    }
			    if (ii == versions.size()-1) {
				versionsStr.append(conjunction);
			    }
			}
			versionsStr.append((String)versions.elementAt(ii));
		    }
		    if (versionsStr.length() == 0) {
			//  If we didn't get a version number back
			//  from the server, then it must be an old
			//  (1.1) server.
			versionsStr.append("1.1");
		    }
		    postError(MessageFormat.format(
			_rs.getString(ERROR_VERSION),
			new String[] {REQUIRED_VERSION,
					  versionsStr.toString()}), true);
		}
	    });
	}

	/**
	 * Called when we fail to login.  We want to prompt the user to
	 * re-enter username/password.
	 *
	 * @param event
	 */
	public void authRetry(AuthEvent event) {
	    // This blanks the password text field.
	    _td.setString(PASSWORD, "");

	    // Only post an error message if we're showing because
	    // it's obnoxious to post more than one dialog; if we're
	    // not showing our client is about to show us.
	    postError(_rs.getString(ERROR_LOGIN), false);
	}

	private void postError(String message, boolean postIfNotShowing) {
	    if (_busy) {
		_uic.notBusy();
		_busy = false;
	    }
	    boolean showing = isShowing();
	    if (!showing) {
		requestShow();
	    }
	    if ((!showing && postIfNotShowing) || showing) {
		_uic.setDialogParent(RemoteHostPanel.this);
		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));
		_uic.postError(message);
	    }
	}
    }

    /**
     * Construct a HostPanel.
     */
    public RemoteHostPanel(String productName) {
	super(new BorderLayout());
	_productName = productName;
	_uic.setDialogParent(this);
	_rs.pushBundle(ResourceStack.getPackageName(UIContext.class) +
		       "SysadmUI" + ResourceStack.BUNDLE_SUFFIX);
	_rs.pushBundle(TaskContext.class.getName()
		       + ResourceStack.BUNDLE_SUFFIX);
	_rs.pushBundle(getClass().getName() + ResourceStack.BUNDLE_SUFFIX);

	if (productName != null) {
	    _rs.pushPackageBundles(productName);
	}
	_td.setString(HOST, "");
	_td.setString(LOGIN, "root");
	_td.setString(PASSWORD, "");
// 	_td.setString(PROXYPORT, "");
// 	_td.setString(PROXYHOST, "localhost");
// 	_td.setString(REMOTESHELLCMD, "");

	TaskPage panel = new TaskPage(_rs);

	panel.setupSizing(_rs, DYNAMIC_SIZE, WIDTH, HEIGHT);

	panel.setTitle(_rs.getString(TITLE));

	panel.setIcon(ICON);

	RichTextArea intro = new
	    RichTextArea("RemoteHostPanel.introText", _rs);
	intro.setMargins(new Insets(0,0,0,0));
	intro.setText(_rs.getString(INTRO_TEXT));
	panel.setIntroText(intro);

 	int textFieldWidth =_rs.getInt(TEXT_FIELD_WIDTH);

// 	_connType = new JComboBox();
// 	_tcpmuxItem = _rs.getString(CONNECTION_TYPE_TCPMUX);
// 	_connType.addItem(_tcpmuxItem);
// 	_remoteShellItem = _rs.getString(CONNECTION_TYPE_REMOTESHELL);
// 	_connType.addItem(_remoteShellItem);
// 	_proxyItem = _rs.getString(CONNECTION_TYPE_PROXY);
// 	_connType.addItem(_proxyItem);

// 	panel.addTaskComponent(_connType,
// 			       CONNECTION_TYPE_LABEL);

//  	_remoteShellText = new RTextField(textFieldWidth);
// 	panel.addTaskComponent(_remoteShellText,
// 			       REMOTE_SHELL_LABEL);
// 	StringJTextComponentBinder.bind(_td, REMOTESHELLCMD, _remoteShellText);

	// _hostText is for editing when we're initially displayed.
 	_hostText = new RTextField(textFieldWidth);
	panel.addTaskComponent(_hostText, HOST_LABEL);
	StringJTextComponentBinder.bind(_td, HOST, _hostText);

	// When we get re-displayed when a connection goes down, we
	// display the host in a label instead of a text field because
	// we don't let the user switch to another host.
	//
	// XXX Maybe the user *would* want to switch!!  If one node in
	// XXX a cluster goes down, it would make sense to switch to
	// XXX another node in the cluster.
	_hostLabel = new JLabel("");
	panel.addTaskComponent(_hostLabel, HOST_LABEL);
	StringJLabelBinder.bind(_td, HOST, _hostLabel);

//  	_proxyHostText = new RTextField(textFieldWidth);
// 	panel.addTaskComponent(_proxyHostText,
// 			       PROXY_HOST_LABEL);
// 	StringJTextComponentBinder.bind(_td, PROXYHOST, _proxyHostText);

//  	_proxyPortText = new RTextField(textFieldWidth);
// 	panel.addTaskComponent(_proxyPortText,
// 			       PROXY_PORT_LABEL);
// 	StringJTextComponentBinder.bind(_td, PROXYPORT, _proxyPortText);

  	_loginText = new RTextField(textFieldWidth);
 	panel.addTaskComponent(_loginText, LOGIN_LABEL);
 	StringJTextComponentBinder.bind(_td, LOGIN, _loginText);

 	_passwordText = new RPasswordField(textFieldWidth);
	panel.addTaskComponent(_passwordText, PASSWORD_LABEL);
	StringJTextComponentBinder.bind(_td, PASSWORD, _passwordText);

	add(panel, BorderLayout.CENTER);

	JPanel buttonPanel = new JPanel();
	int numButtons = 0;

	final JButton okButton = new JButton(_rs.getString(OK_BUTTON_LABEL));
	okButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent ev) {
		if (_busy || _ignoreInput) {
		    Toolkit.getDefaultToolkit().beep();
		    return;
		}
		login();
	    }
	});
	numButtons++;
	buttonPanel.add(okButton);

	ActionListener actionListener = new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		okButton.doClick();
	    }
	};
// 	_remoteShellText.addActionListener(actionListener);
	_hostText.addActionListener(actionListener);
// 	_proxyHostText.addActionListener(actionListener);
// 	_proxyPortText.addActionListener(actionListener);
	_passwordText.addActionListener(actionListener);
 	_loginText.addActionListener(actionListener);

	JButton cancelButton = new JButton(_rs.getString(CANCEL_BUTTON_LABEL));
	cancelButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyCancelled();
	    }
	});
	try {
	    UIContext.addKeyCodeShortCut(
		cancelButton,
		_rs.getInt("RemoteHostPanel.cancelButtonKeyCode"));
	} catch (MissingResourceException ex) {
	}
	try {
	    UIContext.addKeyCodeShortCut(
		okButton,
		_rs.getInt("RemoteHostPanel.okButtonKeyCode"));
	} catch (MissingResourceException ex) {
	}
	numButtons++;
	buttonPanel.add(cancelButton);

	JButton helpButton = new JButton(_rs.getString(HELP_BUTTON_LABEL));
	helpButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		final HostPanelEvent hostEvent =
		    new HostPanelEvent(RemoteHostPanel.this,
				       HostPanelEvent.HELP);
		panelNotify(new Notifier() {
		    public void notify(HostPanelListener listener) {
			listener.hostPanelHelp(hostEvent);
		    }
		});
	    }
	});
	numButtons++;
	buttonPanel.add(helpButton);

	add(buttonPanel, BorderLayout.SOUTH);

	if (!_hostEditable) {
	    setHostEditable(false);
	}

	setHostEditable(_hostEditable);
	_factory = new TcpmuxServerConnectionFactory();
	createConnection(_factory);
	_auth = new UnixClientAuth();
 	_auth.setConnection(_conn);
 	_auth.addAuthListener(new HostPanelAuthListener());
//	determineConnectionFactory();

// 	_connType.addItemListener(new ItemListener() {
// 	    public void itemStateChanged(ItemEvent event) {
// 		if (event.getStateChange() == ItemEvent.SELECTED) {
// 		    setConnectionMethod(event.getItem());
// 		    validate();
// 		    try {
// 			((Window)getTopLevelAncestor()).pack();
// 		    } catch (ClassCastException ex) {
// 		    }
// 		}
// 	    }
// 	});
    }

    /**
     * Set the initial value of the text field that displays the host
     * name.  This should be called before RemoteHostPanel is
     * displayed.
     *
     * @param host intial value of host text field.
     */
    public void setHost(String host) {
	_td.setString(HOST, host);
    }

    /**
     * Control whether the host field is editable.
     *
     * @param editable true if host field should be editable, false
     *                 otherwise.
     */
    public void setHostEditable(boolean editable) {
	_hostLabel.setVisible(!editable);
	_hostText.setVisible(editable);
	_hostEditable = editable;
    }

    /**
     * Set the initial value of the text field that displays the login
     * name.  This should be called before RemoteHostPanel is
     * displayed.
     *
     * @param login initial value of login text field.
     */
    public void setLogin(String login) {
	_td.setString(LOGIN, login);
    }

    /**
     * Set the initial value of the text field that displays the
     * password.  This should be called before RemoteHostPanel is
     * displayed.
     *
     * @param password initial value of password text field.
     */
    public void setPassword(String password) {
	_td.setString(PASSWORD, password);
    }

    /**
     * Set the parent to use for posting dialogs when the
     * RemoteHostPanel itself is not visible.
     *
     * @param dialogParent parent to sue for posting dialogs when the
     *                     RemoteHostPanel itself is not visible.
     */
    public void setDialogParent(JComponent dialogParent) {
	_dialogParent = dialogParent;
    }

    /**
     * Login to server using current UI parameters.  Can be used
     * instead of setVisible() for UI-less logins.
     */
    public void login() {
	final String host;

// 	if (_factory instanceof RshServerConnectionFactory) {
// 	    host = null;
// 	    String rshCommand = _td.getString(REMOTESHELLCMD);
// 	    if (rshCommand.length() == 0) {
// 		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));
// 		_uic.postError(_rs.getString(
// 		    "RemoteHostPanel.Error.needRemoteShell"));
// 		return;
// 	    }
// 	    ((RshServerConnectionFactory)_factory).setCmdAndArgs(rshCommand);
// 	} else if (_proxyPortText.isVisible()) {
// 	    String proxyHost = _td.getString(PROXYHOST);
// 	    String proxyPort = _td.getString(PROXYPORT);
// 	    if (proxyHost.length() == 0 || proxyPort.length() == 0) {
// 		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));
// 		_uic.postError(_rs.getString(
// 		    "RemoteHostPanel.Error.needHostProxy"));
// 		return;
// 	    }
// 	    host = proxyHost + ":" + proxyPort;
// 	    ((TcpmuxServerConnectionFactory)_factory).setProxy(proxyHost,
// 							       proxyPort);
// 	} else {
	    host = _td.getString(HOST);
	    if (host.length() == 0) {
		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));
		_uic.postError(_rs.getString(
		    "RemoteHostPanel.Error.needHost"));
		return;
	    }
	    ((TcpmuxServerConnectionFactory)_factory).clearProxy();
// 	}

// 	if (_auth instanceof UnixClientAuth) {
	    String login = _td.getString(LOGIN);
	    if (login.length() == 0) {
		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));
		_uic.postError(_rs.getString(
		    "RemoteHostPanel.Error.needLogin"));
		return;
	    }
// 	}

	_busy = true;
	_uic.setDialogTitle(_rs.getString(LOGIN_TITLE));
	String message;
	if (host != null) {
	    message = MessageFormat.format(
		    _rs.getString("RemoteHostPanel.connectTitle"),
		    new Object[] { host });
	} else {
	    message = _rs.getString("RemoteHostPanel.connectTitleNoHost");
	}
	_uic.busy(message, new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		Log.assert(_busy, "How did _busy get reset?");
	       	_uic.notBusy();
		_busy = false;
		_factory.cancelConnection();
	    }
	});

	// connect will start a new thread and return since we do
	// not want the actual connection logic to execute in the
	// event dispatch thread, since the busy dialog must run in
	// the event dispatch thread.
	_factory.connect(_conn, _td.getString(HOST), new ResultListener() {
	    public void succeeded(ResultEvent event) {
		// remove busy dialog and put up busy cursor
		_uic.notBusy();
		_uic.busy();
		try {
		    ((UnixClientAuth)_auth).setParams(_td.getString(LOGIN),
						      _td.getString(PASSWORD));
		} catch (ClassCastException ex) {
		}
		_auth.authenticate();
	    }
	    public void failed(ResultEvent event) {
		if (_busy) {
		    _uic.notBusy();
		    _busy = false;
		}
		_uic.setDialogParent(RemoteHostPanel.this);
		_uic.setDialogTitle(_rs.getString(ERROR_TITLE));

		String errorMessage;
		if (host != null) {
		    errorMessage = MessageFormat.format(
			_rs.getString(
			    "RemoteHostPanel.Error.connectionFailed"),
			new Object[] { host });
		} else {
		    errorMessage = _rs.getString(
			"RemoteHostPanel.Error.connectionFailedNoHost");
		}
		String reason = event.getReason();
		if (reason != null) {
		    errorMessage = MessageFormat.format(
			_rs.getString("RemoteHostPanel.Error.connSpecific"),
			new Object[] { errorMessage, reason });
		}
		if (!isShowing()) {
		    requestShow();
		}
		_uic.postError(errorMessage);
	    }
	});
    }

    /**
     * Add a listener to get notified of HostPanelEvents.
     *
     * @param listener Gets notified of HostPanelEvents.
     */
    public void addHostPanelListener(HostPanelListener listener) {
	_listeners.addElement(listener);
    }

    /**
     * Remove a listener from the listeners that get notified of
     * HostPanelEvents.
     *
     * @param listener Will no longer be notified of HostPanelEvents.
     */
    public void removeHostPanelListener(HostPanelListener listener) {
	_listeners.removeElement(listener);
    }

    /**
     *  Return true if current connection type was the one requested.
     *  Return false if using fallback tcpmux connection.
     */
    public boolean usingIntendedConnection() {
	return true;
//	return _usingIntendedConnection;
    }

    /**
     * Display warning message if a fallback connection method is
     * being used because the Connection.method property was
     * not specified properly
     */
    public void checkConnectionMethod() {

// 	if (isShowing() && !_usingIntendedConnection) {
// 	    _uic.setDialogParent(RemoteHostPanel.this);
// 	    _uic.setDialogTitle(_rs.getString(WARNING_TITLE));
// 	    _uic.postWarning(_rs.getString(FALLBACK_CONNECTION));
// 	}
    }

//     /**
//      *  Create the correct ServerConnectionFactory based on
//      *  the server connection menu item chosen by the user.
//      *  This procedure gets called back when choosing any
//      *  of the Connection menu items.
//      *
//      * @param item The connection menu item choosen
//      */
//     public void setConnectionMethod(Object item) {

// 	if (item == _tcpmuxItem) {
// 	    setHostEditable(_hostEditable);
// 	    _auth = new UnixClientAuth();
// 	    if (! (_factory instanceof TcpmuxServerConnectionFactory)) {
// 		_factory = new TcpmuxServerConnectionFactory();
// 		createConnection(_factory);
// 	    }
// 	}
// 	else if (item == _remoteShellItem) {
// 	    setHostEditable(_hostEditable);
// 	    _auth = new NullClientAuth();
// 	    if (! (_factory instanceof RshServerConnectionFactory)) {
// 		_factory = new RshServerConnectionFactory();
// 		createConnection(_factory);
// 	    }
// 	}
// 	else if (item == _proxyItem) {
// 	    _hostText.setVisible(false);
// 	    _hostLabel.setVisible(false);
// 	    _auth = new UnixClientAuth();
// 	    if (! (_factory instanceof TcpmuxServerConnectionFactory)) {
// 		_factory = new TcpmuxServerConnectionFactory();
// 		createConnection(_factory);
// 	    }
// 	} else {
// 	    Log.fatal("Unknown connection method");
// 	}
// 	_auth.setConnection(_conn);
// 	_auth.addAuthListener(new HostPanelAuthListener());

// 	_hostText.setVisible(item != _remoteShellItem && item != _proxyItem);
// 	_passwordText.setVisible(item != _remoteShellItem);
// 	_loginText.setVisible(item != _remoteShellItem);
// 	_remoteShellText.setVisible(item == _remoteShellItem);
// 	_proxyPortText.setVisible(item == _proxyItem);
// 	_proxyHostText.setVisible(item == _proxyItem);
//     }

    /**
     * Request HostPanelListeners to make this RemoteHostPanel visible.
     */
    private void requestShow() {
	_ignoreInput = false;
	final HostPanelEvent hostEvent
	    = new HostPanelEvent(this, HostPanelEvent.SHOW);
	panelNotify(new Notifier() {
	    public void notify(HostPanelListener listener) {
		listener.hostPanelShow(hostEvent);
	    }
	});
    }

    /**
     * Request HostPanelListeners to hide this RemoteHostPanel.
     */
    private void requestHide() {
	_ignoreInput = true;
	final HostPanelEvent hostEvent
	    = new HostPanelEvent(this, HostPanelEvent.HIDE);
	panelNotify(new Notifier() {
	    public void notify(HostPanelListener listener) {
		listener.hostPanelHide(hostEvent);
	    }
	});
    }

    /**
     * Notify HostPanelListeners that something has happened.
     *
     * @param notif Provides a notify() method to notify each
     *              listener.
     */
    private void panelNotify(Notifier notif) {
	Enumeration enum = _listeners.elements();
	while (enum.hasMoreElements()) {
	    HostPanelListener listener
		= (HostPanelListener)enum.nextElement();
	    notif.notify(listener);
	}
    }

    /**
     * Notify HostPanelListeners that the user has pressed Cancel or a
     * fatal error has occurred.
     */
    private void notifyCancelled() {
	final HostPanelEvent hostEvent =
	    new HostPanelEvent(RemoteHostPanel.this, HostPanelEvent.CANCELLED);
	panelNotify(new Notifier() {
	    public void notify(HostPanelListener listener) {
		listener.hostPanelCancel(hostEvent);
	    }
	});
    }

    /**
     * Notify HostPanelListeners that a Connection has been
     * established.
     */
    private void notifySucceeded() {
	final HostPanelEvent hostEvent
	    = new HostPanelEvent(this, new RemoteHostContext(_conn));
	_uic.setHostContext(hostEvent.getHostContext());
	panelNotify(new Notifier() {
	    public void notify(HostPanelListener listener) {
		listener.hostConnectionSucceeded(hostEvent);
	    }
	});
    }

    /**
     * Notify HostPanelListeners that a Connection has encountered an
     * error.
     *
     * @param event ConnectionEvent containing info about the error.
     */
    private void notifyError(ConnectionEvent event) {
	// Game over.  Notifications from Connection don't matter
	// anymore, because we're going to exit when the user presses
	// "OK".
	_conn.removeErrorListener(_errorListener);
	_conn.removeDisruptedListener(_disruptedListener);
	final HostPanelEvent hostEvent
	    = new HostPanelEvent(this, event);
	panelNotify(new Notifier() {
	    public void notify(HostPanelListener listener) {
		listener.hostConnectionError(hostEvent);
	    }
	});
    }

    /**
     * Determine what type of connection the user wants based on
     * system properties that are set.  Default to using tcpmux.
     */
//     private void determineConnectionFactory() {

// 	final String CONNECTION_TYPE   = "Connection.method";
// 	final String SYSADM_PROPERTIES = ".sysadm.properties";

//	Object connType;
//
//	String factoryName;
// 	// This could fail with a SecurityException for an
// 	// unsigned applet or a  FileNotFoundException if the
// 	// properties file does not exist or less likely
// 	// an IoException if the file read fails.
// 	try {
// 	    // Command line specification overrides property file
// 	    factoryName = System.getProperty(CONNECTION_TYPE,"");
// 	    if (factoryName.length() == 0) {
// 		String pname = System.getProperty("user.home", "") +
// 		               System.getProperty("file.separator", "") +
// 			       SYSADM_PROPERTIES;
// 		FileInputStream pfile = new FileInputStream(pname);
// 		Properties props = new Properties();
// 		props.load(new BufferedInputStream(pfile));
// 		factoryName = props.getProperty(CONNECTION_TYPE,"");
// 	    }
// 	}
// 	catch (Exception ex) {
// 	    factoryName = null;
// 	}

// 	if (factoryName == null || factoryName.length() == 0) {
// 	    factoryName = TCPMUX;
// 	}

// 	// It is extremely important that we detect the case where
// 	// we are falling back to tcpmux because of an erroneous
// 	// Connection.method property specification.  This will
// 	// give the user an opportunity to quit before logging in.
// 	_usingIntendedConnection = true;
// 	if (factoryName.equalsIgnoreCase(
// 	             REMOTESHELL.substring(0,REMOTESHELL.length()-1))) {
// 	    connType = _remoteShellItem;
// 	} else if (factoryName.toLowerCase().startsWith(REMOTESHELL) &&
// 		   factoryName.length() > REMOTESHELL.length()) {
// 	    String path = factoryName.substring(REMOTESHELL.length());
// 	    _td.setString(REMOTESHELLCMD, path);
// 	    connType = _remoteShellItem;
// 	} else {
// 	    connType = _tcpmuxItem;

// 	    if (factoryName.equalsIgnoreCase(
// 		                 PORTFWD.substring(0,PORTFWD.length()-1))) {
// 		connType = _proxyItem;
// 	    } else if (factoryName.toLowerCase().startsWith(PORTFWD)) {
// 		int idx;
// 		try {
// 		    String portStr = factoryName.substring(PORTFWD.length());
// 		    if ((idx=portStr.lastIndexOf(':')) < 0) {
// 			_td.setString(PROXYPORT, portStr);
// 		    }
// 		    else {
// 			_td.setString(PROXYPORT, portStr.substring(idx+1));
// 			_td.setString(PROXYHOST, portStr.substring(0,idx));
// 		    }
// 		    connType = _proxyItem;
// 		}
// 		catch (Exception ex) {
// 		    _usingIntendedConnection = false;
// 		}
// 	    }
// 	    else if (!factoryName.equalsIgnoreCase(TCPMUX)) {
// 		_usingIntendedConnection = false;
// 	    }
// 	}
// 	_connType.setSelectedItem(connType);
//	setConnectionMethod(connType);
//    }

    /**
     *  Create the correct Connection based on the ServerConnectionFactory.
     *  Assign appropriate listeners.
     * @param factory The current ServerConnectionFactory
     */
    private void createConnection(ServerConnectionFactory factory) {

	_conn = factory.createConnection();

	_errorListener = new ConnectionErrorListener() {
	    public void connectionError(ConnectionEvent event) {
		notifyError(event);
	    }
	};
	_conn.addErrorListener(_errorListener);
	_disruptedListener = new ConnectionDisruptedListener() {
	    public void connectionDisrupted(ConnectionEvent event) {
		if (_busy) {
		    _uic.notBusy();
		    _busy = false;
		}
		if (_dialogParent != null) {
		    _uic.setDialogParent(_dialogParent);
		}
		_uic.setDialogTitle(_rs.getString(RESTART_TITLE));
		_uic.postGlobalQuestion(
		    MessageFormat.format(
			_rs.getString(QUESTION_RESTART), new Object[] {
			    _conn.getRemoteHostName() }),
		    new ActionListener() {
			public void actionPerformed(ActionEvent event) {
			    login();
			}
		    }, new ActionListener() {
			public void actionPerformed(ActionEvent event) {
			    notifyCancelled();
			}
		    });
	    }
	};
	_conn.addDisruptedListener(_disruptedListener);
    }
}
