//
// DisplayArea.java
//
//	The scrolled area on the right-hand side of TaskManager.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.text.*;
import java.util.*;

/**
 * The DisplayArea is the scrolled panel on the
 * right hand side of the TaskManager window.  The contents of the
 * DisplayArea change in response to the selection of a new category
 * in the Table of Contents.
 */
public class DisplayArea extends JScrollPane
                         implements TaskManagerProperties {
    private HostContext _hc;
    private UIContext _uic;
    private ResourceStack _rs;

    private DisplayPage _currentPage;
    private Color _background;
    private Color _foreground;
    private Font _font;
    private Color _titleColor;
    private Font _titleFont;
    private int _marginTop;
    private int _marginLeft;
    private int _marginBottom;
    private int _marginRight;
    private int _titleBottomOffset;
    private TaskManagerLinkListener _linkListener;

    /**
     * Constructor.
     *
     * @param hc HostContext shared with TaskManager.
     * @param uic UIContext shared with TaskManager
     * @param pages The list of pages that can be requested by the
     *              TableOfContents.
     */
    public DisplayArea(HostContext hc, UIContext uic) {
	_hc = hc;
	_uic = uic;
	_rs = uic.getResourceStack();
	_linkListener = new TaskManagerLinkListener(_hc, _uic);

	_background = _rs.getColor(DISPLAY_AREA_BACKGROUND);
	_foreground = _rs.getColor(DISPLAY_AREA_FOREGROUND);
	_font = _rs.getFont(DISPLAY_AREA_FONT);
	_titleColor = _rs.getColor(DISPLAY_AREA_TITLE_COLOR);
	_titleFont = _rs.getFont(DISPLAY_AREA_TITLE_FONT);
	_marginTop = _rs.getPixels(DISPLAY_AREA_MARGIN_TOP);
	_marginLeft = _rs.getPixels(DISPLAY_AREA_MARGIN_LEFT);
	_marginBottom = _rs.getPixels(DISPLAY_AREA_MARGIN_BOTTOM);
	_marginRight = _rs.getPixels(DISPLAY_AREA_MARGIN_RIGHT);
	_titleBottomOffset = _rs.getPixels(DISPLAY_AREA_TITLE_BOTTOM_OFFSET);
	getVerticalScrollBar().setUnitIncrement(
	    _rs.getPixels(DISPLAY_AREA_VERTICAL_UNIT_INCREMENT));
    }

    /**
     * showPage is called when the user selects a category in the
     * TableOfContents.  The DisplayArea will create the new page if
     * it has not been shown yet and will then display the page.
     *
     * @param pageNumber The number of the page to display.
     * @param listener The listener to notify when the page has been
     *                 displayed.
     */
    public void showPage(final DisplayPage newPage,
			 final ResultListener listener) {
	if (_currentPage == newPage) {
	    listener.succeeded(new ResultEvent(this));
	    return;
	}
	    
	newPage.getPanel(new ResultListener() {
		public void succeeded(ResultEvent event) {
		    _currentPage = newPage;
		    setPanel((JPanel)event.getResult());
		    listener.succeeded(new ResultEvent(this));
		}

		public void failed(ResultEvent event) {
		    listener.failed(event);
		}
	});
    }

    /**
     * Get the HostContext associated with the DisplayArea.
     *
     * @return The HostContext associated with the DisplayArea.
     */
    public HostContext getHostContext() {
	return _hc;
    }

    /**
     * Get the UIContext associated with the DisplayArea.
     *
     * @return The UIContext associated with the DisplayArea.
     */
    public UIContext getUIContext() {
	return _uic;
    }

    /**
     * Create a component that will display a title string.  The
     * font and color of this panel will match the rest of the DisplayArea.
     *
     * @param title The title to display in the title panel.
     *
     * @return A JComponent that displays <TT>title</TT>
     */
    public JComponent createTitleComponent(String title) {
	JLabel titleLabel = new JLabel(title);
	setComponentAppearance(titleLabel);
	titleLabel.setForeground(_titleColor);
	titleLabel.setOpaque(false);
	titleLabel.setFont(_titleFont);
	titleLabel.setBorder(BorderFactory.createEmptyBorder(
	    0, 0, _titleBottomOffset, 0));

	return titleLabel;
    }

    /**
     * Set the font, foreground, and background of the component to
     * match the rest of the DisplayArea.
     *
     * @param comp The component whose appearance is to be set.
     */
    public void setComponentAppearance(Component comp) {
	comp.setBackground(_background);
	comp.setForeground(_foreground);
	comp.setFont(_font);
    }

    /**
     * Set the standard DisplayArea border for <TT>panel</TT>.
     *
     * @param panel The panel whose border is to be set.
     */
    public void setBorder(JPanel panel) {
	panel.setBorder(new EmptyBorder(_marginTop, _marginLeft,
					_marginBottom, _marginRight));
    }

    /**
     * Get the shared DisplayArea TaskManagerLinkListener.
     *
     * @return The shared DisplayArea TaskManagerLinkListener.
     */
    public TaskManagerLinkListener getLinkListener() {
	return _linkListener;
    }

    // Set the page to display and make sure the DisplayArea gets
    // repainted.
    private void setPanel(JPanel panel) {
	setViewportView(panel);
	panel.repaint();
	getViewport().revalidate();
    }
}
