//
// DisplayPage.java
//
//	A page to be displayed in the DisplayArea of TaskManager.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.text.*;

/**
 * DisplayPage is a page that will be displayed in the DisplayArea of
 * the TaskManager window.  The User interface for the page will not
 * be created until the User requests the page.
 */
public class DisplayPage implements TaskManagerProperties {
    private DisplayArea _displayArea;
    private HostContext _hc;
    private UIContext _uic;
    private ResourceStack _rs;
    private int _type;
    private String _title;
    private JLabel _iconLabel = new JLabel();
    private Icon _icon;
    private String _target;
    private JPanel _panel;

    /**
     * Use this value as an argument to DisplayPage.setType() if the
     * page to display will only contain text.
     */
    static public final int TEXT_PAGE_TYPE = 0;

    /**
     * Use this value as an argument to DisplayPage.setType() if the
     * page to display is a tasklist, which will contain a list
     * of Metatasks and Tasks that the User can launch.
     */
    static public final int TASKLIST_PAGE_TYPE = 1;

    /**
     * Use this value as an argument to DisplayPage.setType() if the
     * page to display is a TaskManagerPanel.
     */
    static public final int CLASS_PAGE_TYPE = 2;

    /**
     * Constructor.
     *
     * @param displayArea The DisplayArea that will be displaying this
     *                    page.
     */
    public DisplayPage(DisplayArea displayArea) {
	_displayArea = displayArea;
	_hc = _displayArea.getHostContext();
	_uic = _displayArea.getUIContext();
	_rs = _uic.getResourceStack();
    }
    
    /**
     * Set the page type.  Must be one of DisplayPage.TEXT_PAGE_TYPE,
     * DisplayPage.TASKLIST_PAGE_TYPE, or DisplayPage.CLASS_PAGE_TYPE.
     *
     * @param type The page type.
     */
    public void setType(int type) {
	Log.assert(type == TEXT_PAGE_TYPE || type == TASKLIST_PAGE_TYPE ||
		   type == CLASS_PAGE_TYPE, "Illegal page type");
	_type = type;
    }

    /**
     * Get the page type.
     *
     * @return The type of this page.
     */
    public int getType() {
	return _type;
    }
	
    /**
     * Set the title of this page.
     *
     * @param title The user-readable title of this page.
     */
    public void setTitle(String title) {
	_title = title;
    }
	

    /**
     * Get the title of this page.
     *
     * @return The title of this page.
     */
    public String getTitle() {
	return _title;
    }
	

    /**
     * Set the icon that will be displayed along with the title of
     * this page in the TableOfContents.
     *
     * @param icon The icon to display next to the title of this page
     *             in the TableOfContents.
     */
    public void setIcon(Icon icon) {
	_icon = icon;
	_iconLabel.setIcon(_icon);
    }

    /**
     * Get the icon to display next to the title of this page in the
     * TableOfContents.
     *
     * @return The icon to display next to the title of this page in
     *         the TableOfContents.
     */
    public Icon getIcon() {
	return _icon;
    }

    /**
     * Get the label on which the icon for this page is displayed.
     *
     * @param iconLabel The label used to display the icon for this
     *                  page.
     */
    public void setIconLabel(JLabel iconLabel) {
	_iconLabel = iconLabel;
    }

    /**
     * Set the target to launch when this page is selected in
     * the TableOfContents.
     *
     * @param target The target to launch when this page is selected.
     *               If the page type is TEXT, then the target is the
     *               actual text to display.
     *		     If the page type is METATASK, the target is the
     *               package-qualified name of a metatask.
     *               If the page type is CLASS, the target is the
     *               package-qualified name of a TaskManagerPlugin.
     *               
     */
    public void setTarget(String target) {
	_target = target;
    }

    /**
     * Get the target to launch when this page is
     * selected in the TableOfContents.
     *
     * @return The target to launch when this page is selected.
     *         If the page type is TEXT, then the target is the
     *         actual text to display.
     *	       If the page type is METATASK, the target is the
     *         package-qualified name of a metatask.
     *         If the page type is CLASS, the target is the
     *         package-qualified name of a TaskManagerPlugin.
     */
    public String getTarget() {
	return _target;
    }

    /**
     * Set the panel to display when this page is selected.
     *
     * @param panel The panel to display when this page is selected.
     */
    public void setPanel(JPanel panel) {
	_panel = panel;
    }
	
    /**
     * Get the panel to display when this page is selected.  If the
     * page has not yet been created, create it.
     *
     * @param listener The listener to notify when the page has been
     *                 retrieved or created.
     */
    public void getPanel(ResultListener listener) {
	if (_panel != null) {
	    listener.succeeded(new ResultEvent(this, _panel));
	} else {
	    createPage(listener);
	}
    }

    private void createPage(final ResultListener listener) {
	_panel = new JPanel(new BorderLayout());
	_displayArea.setComponentAppearance(_panel);
	_displayArea.setBorder(_panel);
	_panel.add(_displayArea.createTitleComponent(_title),
		   BorderLayout.NORTH);
	
	JPanel pageContents = null;
	
	switch(_type) {
	case TEXT_PAGE_TYPE:
	    RichTextComponent richText =
		new RichTextComponent(DISPLAY_AREA_TEXT, _rs);
	    richText.setText(_target);
	    richText.addLinkListener(_displayArea.getLinkListener());
	    _panel.add(richText, BorderLayout.CENTER);
	    listener.succeeded(new ResultEvent(this, _panel));
	    break;

	case TASKLIST_PAGE_TYPE:
	     pageContents =
		 new MetataskGroupPanel(_target, _hc, _uic,
					new ResultListener() {
		     public void succeeded(ResultEvent event) {
			 JPanel panel = (JPanel)event.getResult();
			 _displayArea.setComponentAppearance(panel);
			 _panel.add(panel, BorderLayout.CENTER);
			 listener.succeeded(new ResultEvent(this, _panel));
		     }

		     public void failed(ResultEvent event) {
			 listener.failed(event);
		     }
	         });
	     break;

        case CLASS_PAGE_TYPE:
	    final Class targetClass;
	    try {
		targetClass = SysUtil.forName(_target);
	    } catch (ClassNotFoundException exception) {
		ResultEvent event = new ResultEvent(this);
		event.setReason(
		    MessageFormat.format(_rs.getString(MISSING_CLASS_ERROR),
					 new Object[] { _target, _title }));
		listener.failed(event);
		return;
	    }

	    TaskManagerPlugin plugin =
		TaskManagerUtil.loadPlugin(_hc, _uic, targetClass, null);
	    if (plugin == null) {
		ResultEvent event = new ResultEvent(this);
		event.setReason(TaskManagerUtil.getLoadPluginError());
		listener.failed(event);
		return;
	    } else if (!(plugin instanceof TaskManagerPanel)) {
		ResultEvent event = new ResultEvent(this);
		event.setReason(
		    MessageFormat.format(
			_rs.getString(BAD_PAGE_TYPE),
			new Object[] { _target, _title }));
		listener.failed(event);
		return;
	    }

	    final TaskManagerPanel tmPanel = (TaskManagerPanel)plugin;
	    tmPanel.taskManagerSetDisplayArea(_displayArea);
	    tmPanel.taskManagerGetPanel(new ResultListener() {
		public void succeeded(ResultEvent event) {
		    final JPanel panel = (JPanel)event.getResult();

		    _displayArea.setComponentAppearance(panel);
		    _panel.add(panel, BorderLayout.CENTER);
		    listener.succeeded(new ResultEvent(this, _panel));
		}

		public void failed(ResultEvent event) {
		    listener.failed(event);
		}
	    });
	    break;
	}
    }
}
    
