//
// MetataskGroupPanel.java
//
//	A JPanel that displays a tasklist page in Task Manager.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
 * MetataskGroupPanel is a JPanel that displays a tasklist page in
 * the DisplayArea.  A tasklist is a list of Metatasks and Tasks
 * that are logically related in some manner.  The name
 * "MetataskGroup" was the historical name for a "tasklist" but we
 * decided not to spend the time changing the class name since it is
 * not visible to Rhino developers.
 */
public class MetataskGroupPanel extends JPanel
				implements TaskManagerProperties {
    private ResourceStack _rs;
    private UIContext _uic;
    private JComponent _spacing = new JLabel();
    private TaskManagerLinkListener _linkListener;
    private HostContext _hc;
    private TaskRegistry _tr;
    private String _class =
	ResourceStack.getClassName(MetataskGroupPanel.class);

    // This is a member so we can access it inside a listener (see the
    // MetataskGroupPanel constructor that takes the
    // metataskGroupName).
    String taskGroups[] = null;

    // The TaskIconListener is the class that enables TaskIcons to
    // respond to mouse clicks.
    static private class TaskIconListener implements MouseListener,
						     MouseMotionListener  {
	private Cursor _cursor;
	private String _target;
	private JLabel _label;
	private HostContext _hc;
	private UIContext _uic;
	private int _clickX;
	private int _clickY;
	private RichTextComponent _rt;
	private int _type;
	private TaskManagerLinkListener linkListener;
	private int _currentLinkState;
	
	// This is the square of the distance in pixels that the mouse can
	// move while held down over a link and still activate the link
	// when released.
	private final int CLICK_THRESHHOLD_SQUARED = 16;
	public static final int TASK = 0;
	public static final int METATASK = 1;

	public TaskIconListener(String target, HostContext hc,
				UIContext uic, JLabel label,
				RichTextComponent rt, int type,
				TaskManagerLinkListener listener) {
	    _target = target;
	    _hc = hc;
	    _uic = uic;
	    _label = label;
	    _rt = rt;
	    _type = type;
	    linkListener = listener;
	}

	public void mouseEntered(MouseEvent event) {
	    _cursor = _label.getCursor();
	    _label.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
	    _currentLinkState = _rt.getLinkState(0);
	}

	public void mouseExited(MouseEvent event) {
	    _label.setCursor(_cursor);
	    _rt.setLinkState(
		0, _currentLinkState);
	    _rt.repaint();
	}

	public void mousePressed(MouseEvent event) {
	    _clickX = event.getX();
	    _clickY = event.getY();
	    _rt.setLinkState(
		0, RichTextComponent.LINK_STATE_ACTIVE);
	    _rt.repaint();
	}

	public void mouseReleased(MouseEvent event) {
	    if (inClickThreshhold(event.getX(), event.getY())) {
		if (_type == TASK) {
		    linkListener.launchTask(_target);
		} else if (_type == METATASK) {
		    linkListener.launchMetatask(_target);
		}
		_rt.setLinkState(
		    0, RichTextComponent.LINK_STATE_VISITED);
		_currentLinkState =
		    RichTextComponent.LINK_STATE_VISITED;
		_rt.repaint();
	    } else {
		_rt.setLinkState(
		    0, _currentLinkState);
		_rt.repaint();
	    }
	}

	public void mouseClicked(MouseEvent event) {
	    // Ignored.
	}

	public void mouseDragged(MouseEvent event) {
	    if (!inClickThreshhold(event.getX(), event.getY())) {
		_rt.setLinkState(
		    0, _currentLinkState);
		_rt.repaint();
	    }
	}

	public void mouseMoved(MouseEvent event) {
	    // Ignored.
	}

	/**
	 * Determine whether an (x, y) position is close enough to the
	 * mouse down event we got to constitute a click.
	 *
	 * @param x horizontal position.
	 * @param y vertical position.
	 *
	 * @return true if it's close enough for a click, false otherwise.
	 */
	private boolean inClickThreshhold(int x, int y) {
	    int xdelta = _clickX - x;
	    int ydelta = _clickY - y;
	    return xdelta * xdelta + ydelta * ydelta <
		CLICK_THRESHHOLD_SQUARED;
	}
    }

    /**
     * Constructor.
     *
     * @param metataskGroupName The name of the metataskGroup, which
     *				is used to form the set of properties
     *				that list the Metatasks, Tasks, and
     *				any text to display in the metatask
     *				group Panel.
     * @param hc The HostContext to be associated with this panel.
     * @param uic The UIContext to be associated with this panel.
     * @param listener The ResultListener to notify when the panel is
     *		       ready to display or cannot be created.
     */
    public MetataskGroupPanel(final String metataskGroupName,
			      HostContext hc,
			      UIContext uic,
			      final ResultListener listener) {
	initPanel(hc, uic);

	// XXX for now, get the list of Metatasks from a properties
	// XXX file.  Later get them out of the TaskRegistry using
	// XXX metataskGroupName.

	// Add the set of task groups, if any.
	try {
	    taskGroups = _rs.getStringArray(
		TASKMANAGER_PROPERTY_PREFIX +
		metataskGroupName +
		TASK_GROUP);

	    addTaskGroups(taskGroups, 0, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    boolean foundItem = false;
		    MetataskGroupPanel panel =
			(MetataskGroupPanel) event.getResult();
		    if (panel.taskGroups.length > 0) {
			foundItem = true;
		    }

		    // Then put the introductory metatasks text, if any.
		    //
		    String metatasksText = "";
		    try {
			metatasksText = _rs.getString(
			    TASKMANAGER_PROPERTY_PREFIX +
			    metataskGroupName +
			    METATASKS_TEXT);
		    } catch (MissingResourceException exception) {
			// Metatasks text is optional.
		    }
		    if ("" != metatasksText && foundItem) {
			// Put a separator after task groups, before
			// metatask text.
			JSeparator separator = new JSeparator();
			add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
			add(separator, LinkPageLayout.SEPARATOR);
			add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);

			RichTextComponent nameComponent =
			    new RichTextComponent(DISPLAY_AREA_TEXT, _rs);
			nameComponent.setText(metatasksText);
			add(nameComponent, LinkPageLayout.TEXT);
			add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		    }

// XXX any way to use metatask icons for the smart metatasks?

		    // Finally list Metatasks in this group, if any.
		    String[] metatasks = null;
		    try {
			metatasks = _rs.getStringArray(
			    TASKMANAGER_PROPERTY_PREFIX +
			    metataskGroupName +
			    METATASK_ITEM);

			int numMetatasks = metatasks.length;

			if (numMetatasks > 0) {
			    foundItem = true;
			}

			if (metatasksText == "" && foundItem) {
			    // Make sure to put a separator after task
			    // groups, if any, and before metatasks.
			    JSeparator separator = new JSeparator();
			    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
			    add(separator, LinkPageLayout.SEPARATOR);
			    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
			}

			add(_spacing, LinkPageLayout.INDENT);

			for (int ii = 0; ii < numMetatasks; ii++) {
			    addMetatask(metatasks[ii]);
			}

			add(_spacing, LinkPageLayout.OUTDENT);

		    } catch (MissingResourceException exception) {
			// Metatasks are optional.
		    }

		    if (!foundItem) {
			Log.warning("MetataskGroupPanel",
				    "no metatasks or tasks found " +
				    "in metatask group " +
				    metataskGroupName);
		    }
		    listener.succeeded(event);
		}

		public void failed(ResultEvent event) {
		    Log.warning("MetataskGroupPanel",
				"failed to add task groups");
		    listener.succeeded(event);
		}
	    });
	} catch (MissingResourceException exception) {
	    // Task groups are optional.
	    listener.succeeded(new ResultEvent(this, this));
	}
    }

    /**
     * Constructor.
     *
     * Allow TaskManagerPlugins to create Metatask group panels.  For
     * example, the SearchPanel class uses this method to create a
     * panel that shows which Metatasks and Tasks match a particular
     * set of keywords.
     *
     * @param itemList The list of Metatasks and Tasks to display.
     * @param hc The HostContext to associate with this panel.
     * @param uic The UIContext to associate with this panel.
     */
    public MetataskGroupPanel(Vector itemList, HostContext hc,
			      UIContext uic) {
	initPanel(hc, uic);

	int numItems = itemList.size();
	for (int ii = 0; ii < numItems; ii++) {
	    Object item = itemList.elementAt(ii);
	    if (item instanceof TaskLoader) {
		// Add a task item
		addTask((TaskLoader)item);
	    } else if (item instanceof String) {
		// Add a metatask item
		addMetatask((String)item);
	    } else {
		Log.fatal("unknown item type");
	    }

	    if (ii < numItems - 1) {
		add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
	    }
	}
    }

    private void initPanel(HostContext hc, UIContext uic) {
	_hc = hc;
	_uic = uic;
	_tr = _hc.getTaskRegistry();
	_rs = _uic.getResourceStack();

	if (_linkListener == null) {
	    _linkListener = new TaskManagerLinkListener(_hc, _uic);
	}

	setLayout(new LinkPageLayout(this, _rs));

	_spacing.setPreferredSize(
	    new Dimension(0,
			_rs.getPixels(DISPLAY_AREA_EXTRA_VERTICAL_SPACE)));
    }

    /**
     * Set the title of the panel.
     *
     * @param title The component which contains the panel title.
     */
    public void setTitle(Component title) {
	add(title, LinkPageLayout.TITLE);
    }

     private void addMetatask(String metataskName) {
	ResourceStack mtrs = new ResourceStack();
	mtrs.pushStack(_rs);
	mtrs.pushBundle(metataskName + ResourceStack.BUNDLE_SUFFIX);

	// Add the Metatask icon and link
	Icon icon = mtrs.getIcon(METATASK_ICON);
	JLabel iconLabel = new JLabel(icon);
	add(iconLabel, LinkPageLayout.ICON);

	String name = mtrs.getString(METATASK_NAME);
	RichTextComponent nameComponent =
	    new RichTextComponent(DISPLAY_AREA_METATASK_LINK, mtrs);
	nameComponent.setText(TARGET_PREFIX +
			      METATASK_TARGET +
			      metataskName +
			      TARGET_INFIX + name +
			      TARGET_SUFFIX);
	nameComponent.addLinkListener(_linkListener);
	add(nameComponent, LinkPageLayout.LINK);
	TaskIconListener til = new TaskIconListener(
	    metataskName, _hc, _uic, iconLabel,
	    nameComponent, TaskIconListener.METATASK, _linkListener);
	iconLabel.addMouseListener(til);
	iconLabel.addMouseMotionListener(til);
    
	// Add Metatask intro text, if any
	String introText = "";
	try {
	    introText = mtrs.getString(METATASK_INTRO_TEXT);
	    RichTextComponent introTextComponent =
		new RichTextComponent(DISPLAY_AREA_TEXT, mtrs);
	    introTextComponent.setText(introText);
	    introTextComponent.addLinkListener(_linkListener);
	    add(introTextComponent, LinkPageLayout.INDENT);
	    add(introTextComponent, LinkPageLayout.INDENT);
	    add(introTextComponent, LinkPageLayout.TEXT);
	    add(introTextComponent, LinkPageLayout.OUTDENT);
	    add(introTextComponent, LinkPageLayout.OUTDENT);
	} catch (MissingResourceException exception) {
	    // introText is optional
	}
    }

    private void addTaskGroup(final String taskGroupName,
			      final ResultListener listener) {
	_tr.getTaskList(taskGroupName, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		// Add the intro text for the group, if any
		String introTextResource =
		    TASKMANAGER_PROPERTY_PREFIX + taskGroupName +
		    TASK_GROUP_INTRO_TEXT;
		try {
		    String introText = _rs.getString(introTextResource);
// XXX can we make this text column wider in a property?
		    RichTextComponent introTextComponent =
			new RichTextComponent(DISPLAY_AREA_TEXT,
					      _rs);
		    introTextComponent.setText(introText);
		    introTextComponent.addLinkListener(_linkListener);
		    add(introTextComponent, LinkPageLayout.TEXT);
		    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		} catch (MissingResourceException exception) {
		    // The intro text is optional, but the developer
		    // may want to know that it is missing.
		    Log.debug(_class, "Missing property: " +
			      introTextResource);
		}

		TaskLoader[] taskList =
		    (TaskLoader[])event.getResult();

		if (taskList.length == 0) {
		    listener.succeeded(event);
		    return;
		}

		// Add the tasks in the group
		addTasks(taskList);
		listener.succeeded(event);
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }

    private void addTaskGroups(final String[] taskGroups,
			       final int groupNumber,
			       final ResultListener listener) {
	if (groupNumber >= taskGroups.length) {
	    listener.succeeded(new ResultEvent(this, this));
	    return;
	}

	addTaskGroup(taskGroups[groupNumber], new ResultListener() {
	    public void succeeded(ResultEvent event) {
		if (groupNumber < taskGroups.length - 1) {
		    JSeparator separator = new JSeparator();
		    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		    add(separator, LinkPageLayout.SEPARATOR);
		    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		}
		addTaskGroups(taskGroups, groupNumber+1, listener);
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }

    private void addTasks(TaskLoader[] taskList) {
	add(_spacing, LinkPageLayout.INDENT);

	for (int ii = 0; ii < taskList.length; ii++) {
	    addTask(taskList[ii]);
	    if (ii < taskList.length - 1) {
		add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
	    }
	}

	add(_spacing, LinkPageLayout.OUTDENT);
    }

    private void addTask(TaskLoader loader) {
	RichTextComponent nameComponent =
	    new RichTextComponent(DISPLAY_AREA_TASK_LINK, _rs);
	ResourceStack trs = loader.getResourceStack();

	nameComponent.setText(TARGET_PREFIX +
			      TASK_TARGET +
			      loader.getFullTaskName() +
			      TARGET_INFIX +
			      trs.getString(Task.LONG_NAME) +
			      TARGET_SUFFIX);
	nameComponent.addLinkListener(_linkListener);

	// Add the Task icon and link
	Icon icon = trs.getIcon(Task.TASK_SHELF_ICON);
	JLabel iconLabel = new JLabel(icon);
	TaskIconListener til = new TaskIconListener(
	    loader.getFullTaskName(), _hc, _uic, iconLabel,
	    nameComponent, TaskIconListener.TASK, _linkListener); 
	iconLabel.addMouseListener(til);
	iconLabel.addMouseMotionListener(til);

	add(iconLabel, LinkPageLayout.ICON);
	add(nameComponent, LinkPageLayout.LINK);
    }
}
