//
// SearchPanel.java
//
//	A TaskManagerPanel that provides keyword searching of the
//	Metatasks and Tasks that are displayed by TaskManager.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;

/**
 * The SearchPanel is provided as a service by the sysadm
 * architecture for doing keyword searches across Metatasks and
 * Tasks.  The client need only specify this class as one of the items
 * in the TableOfContents and define the keyword property for each
 * Task and Metatask in the product.
 *
 * The first time this panel is displayed there will be a delay while
 * all of the Tasks and Metatasks are indexed.
 */
public class SearchPanel implements TaskManagerPanel, TaskManagerProperties {

    /**
     * The property <I>SearchPanel.introText</I> is the text which
     * will describe the purpose of the Search page.  It may be
     * overridden in the TaskManagerP.properties file.
     */
    static public final String INTRO_TEXT =
	"SearchPanel.introText";

    /**
     * The property <I>SearchPanel.keywordsLabel</I> is the string to
     * display to the left of the keywords field.  It may be
     * overridden in the TaskManagerP.properties file.
     */
    static public final String KEYWORDS_LABEL =
	"SearchPanel.keywordsLabel";

    /**
     * The property <I>SearchPanel.keywordFieldColumns</I> is the
     * number of columns wide the keyword text field should be.  It
     * may be overridden in the TaskManagerP.properties file.
     */
    static public final String KEYWORD_FIELD_COLUMNS =
	"SearchPanel.keywordFieldColumns";

    /**
     * The property <I>SearchPanel.searchButtonLabel</I> is the String
     * to display on the button the User presses to initiate a search
     * on the keywords entered into the keyword field.  It may be
     * overridden in the TaskManagerP.properties file.
     */
    static public final String SEARCH_BUTTON_LABEL =
	"SearchPanel.searchButtonLabel";

    /**
     * The property <I>SearchPanel.clearButtonLabel</I> is the String
     * to display on the button the user presses to clear the keyword
     * field.  It may be overridden in the TaskManagerP.properties
     * file.
     */
    static public final String CLEAR_BUTTON_LABEL =
	"SearchPanel.clearButtonLabel";

    private HostContext _hc;
    private TaskRegistry _tr;
    private UIContext _uic;
    private ResourceStack _rs;
    private DisplayArea _displayArea;
    private RTextField _keywords;
    private JPanel _resultPanel;
    private Vector _metataskGroupItems = new Vector();
    private boolean _firstSearch = true;
    private String _invalidParent =
	"parent parameter does not match value passed to constructor";

    private ActionListener _searchListener = new ActionListener() {
	public void actionPerformed(ActionEvent event) {

	    if (_firstSearch) {
		_uic.busy(_rs.getString("SearchPanel.Busy.indexingKeywords"));
		// Create the keyword->task index.
		indexKeywords(new ResultListener() {
		    public void succeeded(ResultEvent event) {
			_firstSearch = false;
			_uic.notBusy();
			search();
		    }

		    public void failed(ResultEvent event) {
			_uic.notBusy();
			_uic.postError(event.getReason());
		    }
		});

		return;
	    } else {
		search();
	    }
	}
    };

    private void search() {
	_uic.busy();
	_resultPanel.removeAll();
	int panelsAdded = 0;
	Vector keywords = parseKeywordList(_keywords.getText());

	Enumeration mgis = _metataskGroupItems.elements();
	while (mgis.hasMoreElements()) {
	    Vector itemList = new Vector();
	    MetataskGroupItem mgi = (MetataskGroupItem)mgis.nextElement();
	    Enumeration items = mgi._items.elements();
	    nextItem: while (items.hasMoreElements()) {
		Object item = items.nextElement();
		if (item instanceof TaskItem) {
		    TaskItem taskItem = (TaskItem)item;
		    Enumeration searchKeys = keywords.elements();
		    while (searchKeys.hasMoreElements()) {
			String searchKey = (String)searchKeys.nextElement();
			Enumeration itemKeywords =
			    taskItem._keywords.elements();
			while (itemKeywords.hasMoreElements()) {
			    String itemKeyword =
				(String)itemKeywords.nextElement();
			    if (itemKeyword.startsWith(searchKey)) {
				itemList.addElement(taskItem._loader);
				continue nextItem;
			    }
			}
		    }
		} else if (item instanceof MetataskItem) {
		    MetataskItem metataskItem = (MetataskItem)item;
		    Enumeration searchKeys = keywords.elements();
		    while (searchKeys.hasMoreElements()) {
			String searchKey = (String)searchKeys.nextElement();
			Enumeration itemKeywords =
			    metataskItem._keywords.elements();
			while (itemKeywords.hasMoreElements()) {
			    String itemKeyword =
				(String)itemKeywords.nextElement();
			    if (itemKeyword.startsWith(searchKey)) {
				itemList.addElement(metataskItem._name);
				continue nextItem;
			    }
			}
		    }
		} else {
		    Log.fatal("unknown item type");
		}
	    }

	    if (itemList.size() > 0) {
		if (panelsAdded > 0) {
		    _resultPanel.add(new JSeparator());
		}
		MetataskGroupPanel panel =
		    new MetataskGroupPanel(itemList, _hc, _uic);
		_displayArea.setComponentAppearance(panel);
		panel.add(_displayArea.createTitleComponent(mgi._title),
			  LinkPageLayout.TITLE);

		_resultPanel.add(panel);
		panelsAdded++;
	    }
	}

	if (panelsAdded == 0) {
	    JLabel notFoundLabel =
		new JLabel(MessageFormat.format(
		    _rs.getString("TaskManager.Info.noMatchesFound"),
		    new Object[] { _keywords.getText() } ));
	    _displayArea.setComponentAppearance(notFoundLabel);
	    _resultPanel.add(notFoundLabel);
	}

	_displayArea.revalidate();
	_uic.notBusy();
    }

    private class SearchPanelLayout implements LayoutManager2 {
	//private Vector _components = new Vector();
	private Container _parent;
	private int _verticalSpacing;

	public SearchPanelLayout(Container parent) {
	    _parent = parent;
	    _verticalSpacing =
		_rs.getPixels(LinkPageLayout.VERTICAL_SPACING);
	}

	public void addLayoutComponent(String name, Component comp) {
	    Log.fatal("Use add(Component, Object) instead.");
	}

	public void removeLayoutComponent(Component comp) {
	    // no action is needed.
	}

	private Dimension getSize(Container parent) {
	    Dimension size = new Dimension(0, 0);
	    Component[] items = parent.getComponents();

	    for (int ii = 0; ii < items.length; ii++) {
		Component item = items[ii];
		Dimension compSize = item.getPreferredSize();
		size.width = Math.max(size.width, compSize.width);
		size.height += compSize.height + _verticalSpacing;
		// 642942: vertical spacing needed before and after
		// separators
		if (item instanceof JSeparator) {
		    size.height += 2 * _verticalSpacing;
		}
	    }

	    if (size.height > 0) {
		size.height -= _verticalSpacing;
	    }

	    Insets insets = _parent.getInsets();
	    size.width += insets.left + insets.right;
	    size.height += insets.top + insets.bottom;

	    return size;
	}

	public Dimension preferredLayoutSize(Container parent) {
	    Log.assert(parent == _parent, _invalidParent);

	    return getSize(parent);
	}

	public Dimension minimumLayoutSize(Container parent) {
	    Insets insets = _parent.getInsets();
	    return new Dimension(insets.left + insets.right,
				 insets.top + insets.bottom);
	}

	public void layoutContainer(Container parent) {
	    Dimension parentSize = new Dimension(parent.getSize());
	    if (parentSize.width <= 0) {
		// Don't bother doing the layout if the parent isn't
		// a reasonable size yet.
		return;
	    }

	    // Calculate the parent width after insets
	    Insets insets = parent.getInsets();
	    parentSize.width -= (insets.left + insets.right);

	    int currentYPos = insets.top;
	    int availWidth = parentSize.width;
	    Component[] items = parent.getComponents();

	    for (int ii = 0; ii < items.length; ii++) {
		Component item = items[ii];
		Dimension compSize = item.getPreferredSize();
		if (compSize.width == 0 || item instanceof JPanel) {
		    // This is a separator.  Give it the full
		    // available width.
		    compSize.width = availWidth;
		}

		int XPos = insets.left;

		// 642942: extra spacing needed before separators
		if (item instanceof JSeparator) {
		    currentYPos += _verticalSpacing;
		}

		int YPos = currentYPos;
		int width = Math.min(availWidth, compSize.width);
		int height = compSize.height;

		item.setBounds(XPos, YPos, width, height);
		currentYPos +=  height + _verticalSpacing;

		// 642942: extra spacing needed after separators.
		if (item instanceof JSeparator) {
		    currentYPos += _verticalSpacing;
		}
	    }
	}

	public void addLayoutComponent(Component comp, Object
				       constraints) {
	    // no action is needed
	}

	public Dimension maximumLayoutSize(Container parent) {
	    Log.assert(parent == _parent, _invalidParent);

	    return getSize(parent);
	}

	public float getLayoutAlignmentX(Container parent) {
	    Log.assert(parent == _parent, _invalidParent);

	    return java.awt.Component.CENTER_ALIGNMENT;
	}

	public float getLayoutAlignmentY(Container parent) {
	    Log.assert(parent == _parent, _invalidParent);

	    return java.awt.Component.CENTER_ALIGNMENT;

	}

	public void invalidateLayout(Container parent) {
	    Log.assert(parent == _parent, _invalidParent);
	}
    }

    private class MetataskGroupItem {
	String _title;
	Vector _items = new Vector();
    }

    private class MetataskItem {
	String _name;
	Vector _keywords;
    }

    private class TaskItem {
	Vector _keywords;
	TaskLoader _loader;
    }

    // ----- TaskManagerPlugin methods -----
    /**
     * @see
     *
     * com.sgi.sysadm.manager.taskManager.TaskManagerPlugin#initTaskManagerPlugin
     */
    public void initTaskManagerPlugin(HostContext hc, UIContext uic) {
	_hc = hc;
	_tr = _hc.getTaskRegistry();
	_uic = uic;
	_rs = _uic.getResourceStack();
    }

    // ----- TaskManagerPanel methods -----

    /**
     * @see
     *
     * com.sgi.sysadm.manager.taskManager.TaskManagerPanel#taskManagerGetPanel
     */
    public void taskManagerGetPanel(final ResultListener listener) {
	_resultPanel = new JPanel();
	_resultPanel.setLayout(new SearchPanelLayout(_resultPanel));
	_displayArea.setComponentAppearance(_resultPanel);

	final JPanel panel = new JPanel();
	panel.setLayout(new SearchPanelLayout(panel));

	final String introText = _rs.getString(INTRO_TEXT);
	final JPanel introPanel = new JPanel();
	_displayArea.setComponentAppearance(introPanel);
	RichTextComponent introArea =
	    new RichTextComponent(DISPLAY_AREA_TEXT, _rs);
	introArea.setText(introText);
	introPanel.add(introArea);

	JPanel keywordsPanel = new JPanel();
	_displayArea.setComponentAppearance(keywordsPanel);
	keywordsPanel.add(
	    new JLabel(_rs.getString(KEYWORDS_LABEL)));
	_keywords =
	    new RTextField(_rs.getInt(KEYWORD_FIELD_COLUMNS));
	_keywords.addActionListener(_searchListener);
	keywordsPanel.add(_keywords);
	JButton searchButton =
	    new JButton(_rs.getString(SEARCH_BUTTON_LABEL));
	searchButton.addActionListener(_searchListener);
	keywordsPanel.add(searchButton);
	JButton clearButton = new
	    JButton(_rs.getString(CLEAR_BUTTON_LABEL));
	clearButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		_keywords.setText("");
	    }
	});
	keywordsPanel.add(clearButton);

	panel.add(introPanel);
	panel.add(keywordsPanel);
	panel.add(new JSeparator());
	panel.add(_resultPanel);

	listener.succeeded(new ResultEvent(this, panel));
    }

    /**
     * @see
     *
     * com.sgi.sysadm.manager.taskManager.TaskManagerPanel#taskManagerSetDisplayArea
     */
    public void taskManagerSetDisplayArea(DisplayArea displayArea) {
	_displayArea = displayArea;
    }

    private void indexKeywords(ResultListener listener) {
	String[] itemTypes =
	    _rs.getStringArray(TOC_ITEMS);

	indexTasklists(0, itemTypes, listener);
    }

    private void indexTasklists(final int itemNumber,
				final String[] itemTypes,
				final ResultListener listener) {
	if (itemNumber >= itemTypes.length) {
	    // We've indexed each of the Metatask groups.
	    listener.succeeded(new ResultEvent(this));
	    return;
	}

	indexTasklist(itemNumber, itemTypes[itemNumber],
		      new ResultListener() {
	    public void succeeded(ResultEvent event) {
		indexTasklists(itemNumber+1, itemTypes,
			       listener);
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }

    private void indexTasklist(int itemNumber, String itemType,
			       final ResultListener listener) {
	if (!itemType.equals(TASKLIST)) {
	    // We're indexing only the tasklists.
	    listener.succeeded(new ResultEvent(this));
	    return;
	}

	final MetataskGroupItem mgi = new MetataskGroupItem();
	mgi._title = _rs.getString(TOC_ITEMS +
				   itemNumber + TOC_ITEM_TITLE);
	final String target =
	    _rs.getString(TOC_ITEMS + itemNumber + TOC_ITEM_TARGET);

	String[] metatasks =_rs.getStringArray(
	    TASKMANAGER_PROPERTY_PREFIX +
	    target + METATASK_ITEM, null);

	// First index the Metatasks, then index the Task groups.
	indexMetatasks(0, metatasks, mgi, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		String[] taskGroups = _rs.getStringArray(
			TASKMANAGER_PROPERTY_PREFIX + target +
			TASK_GROUP, null);
		indexTaskGroups(0, taskGroups, mgi, new ResultListener() {
		    public void succeeded(ResultEvent event) {
			if (mgi._items.size() > 0) {
			    _metataskGroupItems.addElement(mgi);
			}
			listener.succeeded(event);
		    }

		    public void failed(ResultEvent event) {
			listener.failed(event);
		    }
		});
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }

    private void indexMetatasks(final int itemNumber,
				final String[] metatasks,
				final MetataskGroupItem mgi,
				final ResultListener listener) {
	// Making this method asynchronous is not strictly
	// necessary at this time, since metatasks are defined in
	// properties files rather than in the TaskRegistry.  But I am
	// anticipating that the TaskRegistry will support additional
	// items in the future and I want the code to reflect the more
	// complex logic that would be required.  CTB 7/15/98
	if (metatasks == null || itemNumber >= metatasks.length) {
	    // All of the metatasks have been indexed.
	    listener.succeeded(new ResultEvent(this));
	    return;
	}

	ResourceStack mtrs = new ResourceStack();
	mtrs.pushBundle(metatasks[itemNumber] + ResourceStack.BUNDLE_SUFFIX);
	String keywords = mtrs.getString(METATASK_KEYWORDS, null);
	if (keywords != null) {
	    MetataskItem mti = new MetataskItem();
	    mti._name = metatasks[itemNumber];
	    mti._keywords = parseKeywordList(keywords);
	    mgi._items.addElement(mti);
	}
	indexMetatasks(itemNumber+1, metatasks, mgi, listener);
    }

    private void indexTaskGroups(final int itemNumber,
				 final String[] taskGroups,
				 final MetataskGroupItem mgi,
				 final ResultListener listener) {
	if (taskGroups == null || itemNumber >= taskGroups.length) {
	    // All of the task groups have been indexed.
	    listener.succeeded(new ResultEvent(this));
	    return;
	}

	_tr.getTaskList(taskGroups[itemNumber], new ResultListener() {
	    public void succeeded(ResultEvent event) {
		TaskLoader[] taskList =
		    (TaskLoader[])event.getResult();

		if (taskList.length == 0) {
		    listener.succeeded(event);
		    return;
		}

		// Index the tasks in the current group
		indexTasks(taskList, mgi);

		// Now index the subsequent group.
		indexTaskGroups(itemNumber+1, taskGroups, mgi, listener);
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }

    private void indexTasks(TaskLoader[] taskList, MetataskGroupItem mgi) {
	String keywords;
	for (int ii = 0; ii < taskList.length; ii++) {
	    ResourceStack trs = taskList[ii].getResourceStack();
	    keywords = trs.getString(Task.KEYWORDS, null);
	    if (keywords != null) {
		TaskItem ti = new TaskItem();
		ti._keywords = parseKeywordList(keywords);
		ti._loader = taskList[ii];
		mgi._items.addElement(ti);
	    }
	}
    }

    private Vector parseKeywordList(String keywordList) {
	Vector keywordVector = new Vector();
	String keywords = keywordList.trim().toLowerCase();

	int start = 0;
	int index = 0;
	while (index < keywords.length()) {
	    if (Character.isWhitespace(keywords.charAt(index))) {
		keywordVector.addElement(keywords.substring(start, index));
		index++;

		// Skip any additional whitespace.
		while (index < keywords.length() &&
		       Character.isWhitespace(keywords.charAt(index))) {
		    index++;
		}

		start = index;
	    } else {
		index++;
	    }
	}

	if (start < index) {
	    keywordVector.addElement(keywords.substring(start, index));
	}

	return keywordVector;
    }
}
