//
// TableOfContents.java
//
//	Display the left-hand side of TaskManager as a categorized
//	Table of Contents.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
 * The TableOfContents displays a title and list of categories along
 * the left-hand side of the TaskManager window.  When a category is
 * selected, the icon next to the category text is changed to indicate that
 * the category is active, and the DisplayArea is directed to display
 * the page associated with that category.
 */
public class TableOfContents extends JPanel
                             implements TaskManagerProperties {
    private UIContext _uic;
    private ResourceStack _rs;
    private DisplayArea _displayArea;
    private Vector _pages = new Vector();
    private Vector _iconLabels = new Vector();

    private Icon _activeIcon;
    private Icon _inactiveIcon;
    private JComponent _spacing = new JLabel();
    private DisplayPage _currentPage;

    // 635567: The TOCIconListener is the class that enables
    // TableOfContents icons to respond to mouse clicks.  It's based
    // on the TaskIconListener in MetataskGroupPanel.java.  We add one
    // of these listeners for each task manager LED icon that we want
    // to act like a link; the target of the link (i.e., which page to
    // show) is sent in via the constructor (targetPage).
    private class TOCIconListener implements MouseListener,
	                                     MouseMotionListener  {
	private Cursor _cursor;
	private int _targetPage;
	private JLabel _LED;
	private UIContext _uic;
	private int _clickX;
	private int _clickY;

	// This is the square of the distance in pixels that the mouse can
	// move while held down over a link and still activate the link
	// when released.  It's more generous than in
	// TaskIconListener, because the LED icons tend to be smaller
	// and thus a little harder to click on.
	private final int CLICK_THRESHHOLD_SQUARED = 36;

	public TOCIconListener(int targetPage, 
			       UIContext uic, JLabel LED) {
	    _targetPage = targetPage;
	    _uic = uic;
	    _LED = LED;
	}

	public void mouseEntered(MouseEvent event) {
	    _cursor = _LED.getCursor();
	    _LED.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
	}

	public void mouseExited(MouseEvent event) {
	    if (_cursor != null) {
		_LED.setCursor(_cursor);
		_cursor = null;
	    }
	}

	public void mousePressed(MouseEvent event) {
	    _clickX = event.getX();
	    _clickY = event.getY();
	}

	public void mouseReleased(MouseEvent event) {
	    if (_cursor != null &&
		inClickThreshhold(event.getX(), event.getY())) {

		_uic.busy();
		final DisplayPage page =
		    (DisplayPage)_pages.elementAt(_targetPage);
		_displayArea.showPage(page, new ResultListener() {
		    public void succeeded(ResultEvent event) {
			if (page != _currentPage) {
			    _currentPage.setIcon(_inactiveIcon);
			    page.setIcon(_activeIcon);
			    _currentPage = page;
			}
			_uic.notBusy();
		    }
		    
		    public void failed(ResultEvent event) {
			_uic.notBusy();
			_uic.postError(event.getReason());
		    }
		});
	    }
	}

	public void mouseClicked(MouseEvent event) {
	    // Ignored.
	}

        public void mouseDragged(MouseEvent event) {
	    if (_cursor != null &&
		    !inClickThreshhold(event.getX(), event.getY())) {
		_LED.setCursor(_cursor);
		_cursor = null;
	    }
	}

	public void mouseMoved(MouseEvent event) {
	    // Ignored.
	}

	/**
	 * Determine whether an (x, y) position is close enough to the
	 * mouse down event we got to constitute a click.
	 * 
	 * @param x horizontal position.
	 * @param y vertical position.
	 * 
	 * @return true if it's close enough for a click, false otherwise.
	 */
	private boolean inClickThreshhold(int x, int y) {
	    int xdelta = _clickX - x;
	    int ydelta = _clickY - y;
	    return xdelta * xdelta + ydelta * ydelta <
		CLICK_THRESHHOLD_SQUARED;
	}
    }


    /**
     * Constructor.
     *
     * @param uic UIContext to be associated with the TableOfContents.
     * @param displayArea The DisplayArea where pages will be
     *                    displayed.
     * @param pages The list of pages, one per category, that can be
     *              displayed.
     */
    public TableOfContents(UIContext uic, DisplayArea displayArea) {
	_uic = uic;
	_rs = uic.getResourceStack();
	_displayArea = displayArea;

	_activeIcon = _rs.getIcon(TOC_ACTIVE_ICON);
	_inactiveIcon = _rs.getIcon(TOC_INACTIVE_ICON);

	setLayout(new LinkPageLayout(this, _rs));
	setBorder(new EmptyBorder(_rs.getPixels(TOC_MARGIN_TOP),
				  _rs.getPixels(TOC_MARGIN_LEFT),
				  _rs.getPixels(TOC_MARGIN_BOTTOM),
				  _rs.getPixels(TOC_MARGIN_RIGHT)));
	Color background = _rs.getColor(TOC_BACKGROUND);
	setBackground(background);
	    
	_spacing.setPreferredSize(
	    new Dimension(0, _rs.getPixels(TOC_EXTRA_VERTICAL_SPACE)));
	    
	// Read in the table of contents properties to determine
	// how to populate the table of contents.
 	RichTextComponent title = new RichTextComponent(TOC_TITLE, _rs,
 	    _rs.getString(TOC_TITLE));
	title.setForeground(_rs.getColor(TOC_TITLE_COLOR));
	title.setOpaque(false);
	add(title, LinkPageLayout.TITLE);
	add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
	    
	String[] itemTypes = _rs.getStringArray(TOC_ITEMS);
	    
	// Initialize the page Vector while creating the
	// Table of Contents components.
	int linkCount = 0;
	for (int ii = 0; ii < itemTypes.length; ii++) {
	    if (itemTypes[ii].equalsIgnoreCase(SEPARATOR)) {
		JSeparator separator = new JSeparator();
		add(separator, LinkPageLayout.SEPARATOR);
		add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		continue;
	    }
		
	    String pageTitle =
		_rs.getString(TOC_ITEMS + ii + TOC_ITEM_TITLE);
		
	    JLabel icon = new JLabel(_inactiveIcon);
	    icon.setAlignmentX(Component.CENTER_ALIGNMENT);
	    icon.setAlignmentY(Component.CENTER_ALIGNMENT);
	    // 635567: Let user click on LED to change pages too.
	    icon.addMouseListener(new TOCIconListener(
		linkCount, _uic, icon));
	    add(icon, LinkPageLayout.ICON);
		
	    RichTextComponent richText =
		new RichTextComponent(TOC_LINK, _rs);
	    richText.setText(TARGET_PREFIX + linkCount + TARGET_INFIX +
			     pageTitle + TARGET_SUFFIX);
	    richText.addLinkListener(new LinkListener() {
		public void linkActivated(LinkEvent event) {
		    final int pageNumber =
			new Integer(event.getTarget()).intValue();
		    _uic.busy();
		    final DisplayPage page =
			(DisplayPage)_pages.elementAt(pageNumber);
		    _displayArea.showPage(page, new ResultListener() {
			public void succeeded(ResultEvent event) {
			    if (page != _currentPage) {
				_currentPage.setIcon(_inactiveIcon);
				page.setIcon(_activeIcon);
				_currentPage = page;
			    }
			    _uic.notBusy();
			}

			public void failed(ResultEvent event) {
			    _uic.notBusy();
			    _uic.postError(event.getReason());
			}
		    });
		}
	    });
	    add(richText, LinkPageLayout.LINK);
	    add(_spacing, LinkPageLayout.EXTRA_VERTICAL_SPACE);
		
	    DisplayPage dp = new DisplayPage(_displayArea);
	    dp.setTarget(_rs.getString(TOC_ITEMS + ii + TOC_ITEM_TARGET));
	    dp.setTitle(pageTitle);
	    dp.setIconLabel(icon);
		
	    if (itemTypes[ii].equalsIgnoreCase(TEXT)) {
		dp.setType(DisplayPage.TEXT_PAGE_TYPE);
	    } else if (itemTypes[ii].equalsIgnoreCase(TASKLIST)) {
		dp.setType(DisplayPage.TASKLIST_PAGE_TYPE);
	    } else if (itemTypes[ii].equalsIgnoreCase(CLASS)) {
		dp.setType(DisplayPage.CLASS_PAGE_TYPE);
	    } else {
		Log.warning("TableOfContents", "Illegal page type");
	    }

	    _pages.addElement(dp);

	    linkCount++;
	}
	    
	if (_pages.size() > 0) {
	    _displayArea.showPage((DisplayPage)_pages.elementAt(0),
				   new ResultListener() {
		public void succeeded(ResultEvent event) {
		    _currentPage = (DisplayPage)_pages.elementAt(0);
		    _currentPage.setIcon(_activeIcon);
		}

		public void failed(ResultEvent event) {
		    _uic.postError(event.getReason());
		}

	    });
	}
    }
}
