//
// TaskManagerLinkListener.java
//
//	A LinkListener that can launch a Task, Metatask, or glossary window.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
 * The TaskManagerLinkListener is shared by several components of
 * TaskManager.  It is intended to be added to a RichTextComponent and
 * launch either a Task, Metatask, or Glossary window depending on the
 * format of the link.
 */
public class TaskManagerLinkListener implements LinkListener,
                                                TaskManagerProperties {
    private HostContext _hc;
    private UIContext _uic;
    private ResourceStack _rs;
    private static RFrame _glossaryFrame;
    private static RichTextArea _glossary;
	

    /**
     * Constructor.
     *
     * @param hc HostContext to be associated with the launched
     *           object.
     * @param uic UIContext to be associated with the launched object.
     */
    public TaskManagerLinkListener(HostContext hc, UIContext uic) {
	_hc = hc;
	_uic = uic;
	_rs = _uic.getResourceStack();
    }

    /**
     * @see com.sgi.sysadm.ui.richText.LinkListener#linkActivated
     */
    public void linkActivated(LinkEvent event) {
	String target = event.getTarget();
	if (target.startsWith(GLOSSARY_TARGET)) {
	    if (_glossaryFrame == null) {
		_glossaryFrame = new RFrame();
		_glossary = new RichTextArea(RichTextArea.GLOSSARY_NAME, _rs);
		_glossaryFrame.getContentPane().add(_glossary);
		_glossaryFrame.setTitle(
		    _rs.getString(RichTextArea.GLOSSARY_TITLE, "Glossary"));
	    }

	    // Pass our resource stack to the glossary window so that
	    // links within glossary entries will work.
	    _glossary.setGlossaryResources(_rs);
	    _glossary.setText(_rs.getString(target, target));
	    _glossaryFrame.pack();
	    _glossaryFrame.setVisible(true);
	} else if (target.startsWith(METATASK_TARGET)) {
	    launchMetatask(target.substring(METATASK_TARGET.length()));
	} else if (target.startsWith(TASK_TARGET)) {
	    launchTask(target.substring(TASK_TARGET.length()));
	} else {
	    Log.fatal("invalid target activated: " + target);
	}
    }

    /**
     * Launches a metatask for the TaskManager
     */
    /* package */ void launchMetatask(String metataskName) {
	ResourceStack mtrs = new ResourceStack();
	mtrs.pushBundle("com.sgi.sysadm.manager.TaskManager" + 
	    ResourceStack.BUNDLE_SUFFIX);
	mtrs.pushPackageBundles(metataskName);
	mtrs.pushBundle(metataskName + ResourceStack.BUNDLE_SUFFIX);
	
	    final RFrame frame = new RFrame();
	    
	    final HostContext.Client linkListenerClient =
		new HostContext.Client(frame, _uic);
	    _hc.registerClient(linkListenerClient);
	    frame.setDefaultCloseOperation(
		WindowConstants.DISPOSE_ON_CLOSE);
	    frame.addWindowListener(new WindowAdapter() {
		public void windowClosed(WindowEvent event) {
		    _hc.unregisterClient(linkListenerClient);
		}
	    });

	    frame.setTitle(mtrs.getString(METATASK_NAME));
	    JPanel panel = new JPanel(new BorderLayout());
	    RichTextComponent rt =
		new RichTextComponent(
		    mtrs.getString(METATASK_TEXT));
	    rt.addLinkListener(this);
	    panel.add(rt, BorderLayout.CENTER);

	    JPanel buttonBar = new JPanel();
	    JButton closeButton = 
		new JButton(mtrs.getString(METATASK_CLOSE_BUTTON_LABEL));
	    closeButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    frame.dispose();
		}
	    });
	    buttonBar.add(closeButton);
	    panel.add(buttonBar, BorderLayout.SOUTH);
	    
	    frame.setContentPane(panel);
	    frame.pack();
	    frame.setVisible(true);
    }
    
    /**
     * Launches a Task for the TaskManager
     */
    /* package */ public void launchTask(String target) { 
	TaskFrame.launchTaskFrame(
	    new TaskLaunchRequestEvent(_hc, new
				       TaskLoader(_hc, target)), 
	    _uic,
	    TaskFrame.POST_RESULT_PANEL);
    }
}
