//
// TaskManagerUtil.java
//
//	Utility methods for TaskManager and supporting classes.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.manager.taskManager;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import java.lang.reflect.*;

/**
 * TaskManagerUtil contains utility methods for TaskManager and its
 * supporting classes.
 */
public class TaskManagerUtil implements TaskManagerProperties {
    static private String _loadPluginError;
    
    /**
     * Attempt to load a TaskManagerPlugin.
     *
     * @param hc The HostContext to pass to the plugin.
     * @param uic The UIContext to pass to the plugin.
     * @param pluginClass The Class to load.
     * @param productName The Package name of the current product
     *
     * @return A TaskManagerPlugin that has been loaded, instantiated,
     *         and initialized.  Null if the plugin could not be
     *         loaded.  If null is returned, the caller should call
     *         getLoadPluginError() to get a User readable error
     *         message explaining the problem.
     */

    static public TaskManagerPlugin loadPlugin(final HostContext hc,
					       final UIContext uic,
					       Class pluginClass,
					       String productName) {
	ResourceStack rs = uic.getResourceStack();
	boolean setProductName = false;
	_loadPluginError = null;
	Constructor[] pluginConstructors = null;

	try {
	    pluginConstructors = pluginClass.getConstructors();
	} catch (SecurityException exception) {
	    _loadPluginError =
		rs.getString("TaskManager.Error.pluginSecurityException");
	}

	if (pluginConstructors == null) {
	    return null;
	}

	// Find the right constructor
	Constructor pluginConstructor = pluginConstructors[0];
	Object[] constructorArgs = null;

	for (int ii = 0; ii < pluginConstructors.length; ii++) {
	    Class[] parameters =
		pluginConstructors[ii].getParameterTypes();
	    if (parameters.length == 0) {
		continue;
	    }
	    if (parameters.length == 1 &&
		    parameters[0] == String[].class) {
		pluginConstructor = pluginConstructors[ii];
		constructorArgs = new Object[] { (String[])null };
		setProductName = true;
	    } else if (parameters.length == 1 && 
		       parameters[0] == String.class) {
		pluginConstructor = pluginConstructors[ii];
		constructorArgs = new Object[] { productName };
	    } else {
		_loadPluginError =
			rs.getString("TaskManager.Error.invalidPluginConstructor");
		return null;
	    }
	}

	// Attempt to instantiate the plugin
	TaskManagerPlugin plugin = null;

	try {
	    plugin =
		(TaskManagerPlugin)(pluginConstructor.newInstance(constructorArgs));
	} catch (InstantiationException exception) {
	    _loadPluginError = rs.getString(
		"TaskManager.Error.pluginInstantiationException");
	} catch (IllegalAccessException exception) {
	    _loadPluginError = rs.getString(
		"TaskManager.Error.pluginIllegalAccess");
	} catch (IllegalArgumentException exception) {
	    _loadPluginError = rs.getString(
		"TaskManager.Error.pluginIllegalArgument");
	} catch (InvocationTargetException exception) {
	    exception.getTargetException().printStackTrace();
	    _loadPluginError = rs.getString(
		"TaskManager.Error.pluginConstructorFailed");
	}
	
	if (plugin == null) {
	    return null;
	}

	if (plugin instanceof com.sgi.sysadm.manager.RApp && setProductName) {
	    ((com.sgi.sysadm.manager.RApp)plugin).setProductName(productName);
	}

	if (plugin instanceof TaskManagerFrame) {
	    plugin.initTaskManagerPlugin(hc, new UIContext());
	    ((TaskManagerFrame)plugin).taskManagerLaunchFrame();
	} else if (plugin instanceof TaskManagerPanel) {
	    plugin.initTaskManagerPlugin(hc, uic);
	    // Allow the client to get the panel when desired.
	} else if (plugin instanceof TaskManagerAction) {
	    final UIContext newUIC = new UIContext();
	    final TaskManagerPlugin pluginCopy = plugin;
	    newUIC.setDialogParent(
		uic.getDialogParent(),
		new ResultListener() {
		public void succeeded(ResultEvent event) {
		    pluginCopy.initTaskManagerPlugin(hc, newUIC);
		    uic.busy();
		    ((TaskManagerAction)pluginCopy).
			taskManagerStartAction(new ResultListener() {
			    public void succeeded(ResultEvent event) {
				uic.notBusy();
			    }
			    
			    public void failed(ResultEvent event) {
				uic.notBusy();
				String reason = event.getReason();
				if (reason != null && reason.length() > 0) {
				    uic.postError(reason);
				}
			    }
			});
		}
		public void failed(ResultEvent event) {
		}
	    });
	} else if (plugin instanceof TaskManagerTitleRenderer) {
	    plugin.initTaskManagerPlugin(hc, uic);
	    // Allow the client to initiate title rendering
	} else if (plugin instanceof TaskManagerInitPlugin) {
	    plugin.initTaskManagerPlugin(hc, uic);
	} else {
	    _loadPluginError = rs.getString(
		"TaskManager.Error.unknownPluginType");
	    return null;
	}
	
	return plugin;
    }

    
    /**
     * Get the error generated by the last call to loadPlugin().
     *
     * @return A User readable error message explaining why the
     *         previous call to loadPlugin() failed.  Null if
     *         loadPlugin() was not called previously or if it succeeded.
     */
    static public String getLoadPluginError() {
	return _loadPluginError;
    }
}
