//
// AssociationProxy.java
//	Proxy class for Association
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.category;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.proxy.comm.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.proxy.util.*;
import java.text.MessageFormat;

/**
 * AssociationProxy is the proxy class for Association.
 */
public class AssociationProxy extends Association 
                              implements PacketListener,
                                         ConnectionStartedListener,
                                         Releaseable {
    
    private int _refCount = 0;
    private boolean _initialized = false;
    private Connection _conn = null;
    private SysadmProxy _sysadmProxy = null;
    private static Hashtable _assocProxies = new Hashtable();
    private ProxyHelper _helper = null;

    private static final String ASSOC_SERVICE = "association";
    private static final String ASSOCPROXY = "assocProxy";
    private static final int CAPACITY_INCR = 5;
    private static ResourceStack _rs = null;

    // restart logic
    private boolean _receivingListOnRestart = false;
    private long _skipEndBlockChanges = 0;
    private Hashtable _listOnRestart = new Hashtable();
    private String _serviceName;
    private boolean _releasing = false;

    // Handshake strings between AssociationProxy and Association service.
    private static final String LOAD = "load";
    private static final String UNLOAD = "unload";
    private static final String ADDASSOCIATIONLISTENER = 
        "addAssociationListener";

    // Handshake strings between AssociationListenerProxy and AssociationProxy.
    private static final String ITEM = "item";
    private static final String ITEMADDED = "itemAdded";
    private static final String ITEMCHANGED = "itemChanged";
    private static final String NEWITEM = "newItem";
    private static final String ITEMREMOVED = "itemRemoved";
    private static final String SELECTOR = "selector";
    private static final String BEGINBLOCKCHANGES = "beginBlockChanges";
    private static final String ENDBLOCKCHANGES = "endBlockChanges";
    private static final String ENDEXISTS = "endExists";
    private static final String ERROR = "error";
    private static final String ERRORSTRING = "errorString";

    /**
     * Protected constructor to ensure that clients get a handle
     * to an AssociationProxy instance via the getAssociationProxy method.
     *
     * @param conn Connection to the server.
     * @param parentCategory The type of the parent Item.
     * @param parentSelector The selector of the parent Item.
     * @param childCategory The type of the related Item(s).
     */
    protected AssociationProxy(Connection conn,
			       String parentCategory, 
			       String parentSelector,
			       String childCategory) {
	super(parentCategory, parentSelector, childCategory);
	Log.assert(conn != null,
		   "Cannot pass null to AssociationProxy constructor");
	_conn = conn;
	_sysadmProxy = SysadmProxy.get(conn);	
	
	_serviceName = 
	    new String(ASSOC_SERVICE + "/" + getSelector());
	_helper = ProxyHelper.getProxyHelper(_conn, ASSOC_SERVICE);
	_conn.addStartedListener(this);
    }

    /**
     * Obtain a handle to an AssociationProxy instance.
     * 
     * @param conn Connection to the server.
     * @param parentCategory The type of the parent Item.
     * @param parentSelector The selector of the parent Item.
     * @param childCategory The type of the related Item(s).
     * 
     * @return Association instance
     */
    public static Association getAssociationProxy(Connection conn,
						  String parentCategory,
						  String parentSelector,
						  String childCategory) {
	AssociationProxy proxy = null;
	
	// Get the vector of proxies with the selector of this association
	// The vector returned represents proxy classes for 
	// the association on different hosts.
	String assocName = computeSelector(parentCategory, parentSelector,
					   childCategory);

	// Need to synchronize access to _assocProxies, because
	// AssocProxies are typically released from within garbage
	// collection thread.
	synchronized (_assocProxies) {
	    Vector proxies = (Vector) _assocProxies.get(assocName);
	    boolean createIt = true;

	    if (proxies == null) {

		Log.debug(ASSOCPROXY, "No proxies for assocName " + assocName);
		proxies = new Vector(CAPACITY_INCR, CAPACITY_INCR);
		_assocProxies.put(assocName, proxies);

	    } else {

		Log.debug(ASSOCPROXY, "Proxies exist for assocName " + assocName);
		Enumeration enum = proxies.elements();
		while (enum.hasMoreElements()) {
		    // If so, incr ref count, return proxy
		    proxy = (AssociationProxy) enum.nextElement();
		    if (proxy._conn == conn) {
			createIt = false;
			Log.debug(ASSOCPROXY, "Proxies exist for this conn");
			break;
		    }
		}

	    }

	    // Else, create AssociationProxy and insert into Vector
	    if (createIt) {
		Log.debug(ASSOCPROXY, "No proxies exist for this assocName/conn");
		proxy = new AssociationProxy(conn, parentCategory,
					     parentSelector,
					     childCategory);
		proxies.addElement(proxy);
	    }	
	
	    Log.assert(proxy != null, 
		       "getAssociationProxy returning a null proxy!");
	    
	    AssociationRef ref = new AssociationRef(proxy);
	    proxy.get();
	    return ref;
	}
    }

    // Used by clients of Association to increment reference count
    private void get() {
	synchronized (_assocProxies) {
	    _refCount++;
	}
    }

    /**
     * Indicate that you are no longer holding on to the AssociationProxy
     * handle obtained via getAssociationProxy
     */
    public void release() {

	synchronized (_assocProxies) {
	    _refCount--;
	    Log.debug(ASSOCPROXY, "refCount: " + _refCount);
	    if (_refCount == 0) {
		_releasing = true;
		String assocName = getSelector();
		Vector proxies = (Vector) _assocProxies.get(assocName);
	
		Log.assert((proxies != null), "No proxies exist for assocName " +
			   assocName + " ?!!");
	
		boolean foundIt = proxies.contains(this);
		Log.assert((foundIt), "No proxies exist for my conn  ?!!");

		// unload
		unload();
		proxies.removeElement(this);
		_helper.release();

		if (proxies.size() == 0) {
		    _assocProxies.remove(assocName);
		}
	    }
	}
    }

    /**
     * Translate message from server into the corresponding method
     * invocation on base Category class.
     * 
     * @param packet Packet received.
     */
    public void receivePacket(Packet packet) {
	
	Log.assert(packet != null, "Invalid argument to receivePacket");
	
	if (packet.getSelector().equals(ITEMADDED)) {

	    Item item = new Item(packet.getBundle(ITEM));
	    if (Log.isLevelOn(Log.DEBUG)) {
                // Calling item.toString is really expensive - skip it
                // if it's not going to be used.
		Log.debug(ASSOCPROXY,
			  "I got an added item: " + item.toString());
	    }
	    if (_receivingListOnRestart) {
		_listOnRestart.put(item.getSelector(), item);
	    } else {
		addItem(item);
	    }

	} else if (packet.getSelector().equals(BEGINBLOCKCHANGES)) {

	    Log.debug(ASSOCPROXY, "I got a beginBlockChanges");
	    if (_receivingListOnRestart) {
		_skipEndBlockChanges++;
	    } else {
		beginBlockChanges();
	    }

	} else if (packet.getSelector().equals(ENDBLOCKCHANGES)) {

	    Log.debug(ASSOCPROXY, "I got an endBlockChanges");
	    if (_skipEndBlockChanges > 0) {
		_skipEndBlockChanges--;
	    } else {
		endBlockChanges();
	    }

	} else if (packet.getSelector().equals(ENDEXISTS)) {

	    Log.debug(ASSOCPROXY, "I got an endExists");
	    if (_receivingListOnRestart) {
		Log.debug(ASSOCPROXY, getSelector() + 
			   " Restart list ... will *not* pass thru endExists");
		replaceItemList(_listOnRestart);
		_listOnRestart.clear();
		_receivingListOnRestart = false;
	    } else {
		Log.debug(ASSOCPROXY, getSelector() + 
			   " Original list ... will pass thru endExists");
		endExists();
	    }

	} else if (packet.getSelector().equals(ITEMCHANGED)) {

	    Log.assert(!_receivingListOnRestart, 
		       "Got changeItem while receivingListOnRestart");
	    Item newItem = new
	        Item(packet.getBundle(NEWITEM));
	    Log.debug(ASSOCPROXY, "I got newItem: " +
		      newItem.toString());
	    changeItem(newItem);
	    
	} else if (packet.getSelector().equals(ITEMREMOVED)) {

	    Log.assert(!_receivingListOnRestart, 
		       "Got removeItem while receivingListOnRestart");
	    String selector = packet.getString(SELECTOR);
	    Log.debug(ASSOCPROXY, "I got a removed item: " +
		      selector);
	    removeItem(selector);

	} else if (packet.getSelector().equals(ERROR)) {

	    Log.debug(ASSOCPROXY, "I got error: ");
	    Log.debug(ASSOCPROXY, "errorString: " +
		      packet.getAttr(ERRORSTRING).stringValue());
	    _conn.notifyError(packet.getAttr(ERRORSTRING).stringValue());

	} else {
	    Log.debug(ASSOCPROXY, "Received unrecognized packet");
	}
    }

    /**
     * Called at initialization, and also after a connection has been
     * lost and then re-established.
     *
     * @param event The connection event.
     */
    public void connectionStarted(ConnectionEvent event) {
	Log.debug(ASSOCPROXY,  getSelector() + 
		  " Trying to restart connection");

	synchronized (_assocProxies) {
	    if (_releasing) {
		return;
	    }

	    load();

	    // Start the flow of notifications
	    if (isMonitoring()) {	
		while (isInBlockChangesNotification()) {
		    endBlockChanges();
		}

		if (!hasExistsEnded()) {
		    endExists();
		}

		_listOnRestart.clear();
		Log.debug(ASSOCPROXY,  getSelector() + 
			  " Setting receivingListOnRestart");
		_receivingListOnRestart = true;
		_skipEndBlockChanges = 0;

		startMonitor();
	    }
	}
    }
    
    /**
     * Define base class abstract method 
     */
    protected void startMonitor() {
	// This messages starts the flow of
	// notifications from server to client
	Packet packet = new Packet(_serviceName, ADDASSOCIATIONLISTENER);
	_conn.sendPacket(packet);
	Log.debug(ASSOCPROXY, getSelector() + " receivingListOnRestart is" +
		  ((_receivingListOnRestart) ? "true" : "false"));
    }

    /**
     * Called when AssociationProxy is constructed and when a 
     * ConnectionStarted event is received.
     *
     * Loads Association service plugin and the requested Association.
     * 
     * Note: the load() function is not symmetrical to the unload()
     * function.  This is because load() is called from
     * connectionRestarted(), which can be called multiple times as
     * the Connection comes and goes.  One-time initialization occurs
     * in the constructor, and connection initialization happens here
     * in load().  unload() is called only once, when the
     * CategoryProxy is released, and hence it releases resources
     * acquired in the constructor and during load().
     */
    private void load() {
	
	// Load the Association service plugin 
	_sysadmProxy.loadService(ASSOC_SERVICE, new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(ASSOCPROXY, 
			  ASSOC_SERVICE + " service loaded successfully");
		_helper.serviceLoaded();
	    }

	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "AssociationProxy.Error.loadAssocService"),   
		    new String[] { event.getReason() }));
	    }

	});	

	// We do not block until loading of Association service 
	// plugin returns to send the following load requests
	// Instead we proceed to send the following messages 
	// and handle failure of loading of Association service
	// when it is detected.

	// Ask Association service plugin to launch the plugin
	// Association.
	_helper.sendRequest(assocRequest(LOAD), new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(ASSOCPROXY, 
			  getSelector() + " association loaded successfully");
		
		if (!_initialized) {
		    // Set up AssociationProxy to receive
		    // notifications on _conn from the association specified
		    _conn.addPacketListener(_serviceName, 
					    AssociationProxy.this);
		    _initialized = true;
		}
	    }

	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "AssociationProxy.Error.loadSpecificAssoc"),   
		    new String[] { event.getReason() }));
	    }

	});
    }

    /**
     * Stop receiving packets and unload services that were 
     * previously loaded.
     */
    private void unload() {

	Log.debug(ASSOCPROXY, "Giving up connection.");

	// Unregister interest in notifications from
	// Association plugin
	_conn.removePacketListener(_serviceName);
	_conn.removeStartedListener(this);

	// Unload plugin Association
	_helper.sendRequest(assocRequest(UNLOAD), new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(ASSOCPROXY, 
			  getSelector() + 
			  " association unloaded successfully");
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "AssociationProxy.Error.unloadSpecificAssoc"),   
		    new String[] { event.getReason() }));
	    }
	});

	// Unload the Association service plugin 
	_sysadmProxy.unloadService(ASSOC_SERVICE, new ResultListener() {

	    public void succeeded(ResultEvent event) {
		Log.debug(ASSOCPROXY, 
			  ASSOC_SERVICE + " service unloaded successfully");
		_sysadmProxy.release();
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "AssociationProxy.Error.unloadAssocService"),   
		    new String[] { event.getReason() }));
	    	_sysadmProxy.release();
	    }
	});	
    }

    /**
     * Shared code between load and unload.
     * Create a packet with request info.
     * 
     * @param request Server request (LOAD or UNLOAD).
     * 
     * @return Packet to send to server for this request.
     */
    private Packet assocRequest(String request) {
	Packet packet = new Packet(ASSOC_SERVICE, request);
	packet.setAttr(new Attribute("parentCategoryName", 
				     getParentCategoryName()));
	packet.setAttr(new Attribute("parentSelector",
				     getParentSelector()));
	packet.setAttr(new Attribute("childCategoryName", 
				     getChildCategoryName()));
	return packet;		       
    }

    private void initResourceStack() {
	if (_rs == null) {
	    _rs = new ResourceStack();
	    _rs.pushBundle("com.sgi.sysadm.proxy.category.AssociationProxy" +
			   ResourceStack.BUNDLE_SUFFIX);
	}
    }

}
