//
// CategoryProxy.java
//	Proxy class for Category	
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.category;

import java.util.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.proxy.comm.*;
import com.sgi.sysadm.proxy.util.*;
import java.text.MessageFormat;

/**
 * CategoryProxy is the proxy class for Category.
 */
public class CategoryProxy extends Category 
                           implements PacketListener,
                                      ConnectionStartedListener,
                                      Releaseable {

    private Connection _conn = null;
    private int _refCount = 0;
    private boolean _initialized = false;
    private SysadmProxy _sysadmProxy = null;
    private static Hashtable _categoryProxies = new Hashtable();
    private ProxyHelper _helper = null;
    private static ResourceStack _rs = null;

    private  static final String CATEGORY_SERVICE = "category";
    private static final String CATEGORYPROXY = "categoryProxy";
    private static final int CAPACITY_INCR = 5;

    // restart logic
    private boolean _receivingListOnRestart = false;
    private long _skipEndBlockChanges = 0;
    private Hashtable _listOnRestart = new Hashtable();
    private String _serviceName;
    private boolean _releasing = false;

    // Handshake strings between CategoryProxy and Category service.
    private static final String LOAD = "load";
    private static final String UNLOAD = "unload";
    private static final String ADOPTNOTIFICATIONFILTER = 
        "adoptNotificationFilter";

    // Handshake strings between CategoryListenerProxy and CategoryProxy
    private static final String ITEM = "item";
    private static final String ITEMADDED = "itemAdded";
    private static final String ITEMCHANGED = "itemChanged";
    private static final String NEWITEM = "newItem";
    private static final String ITEMREMOVED = "itemRemoved";
    private static final String SELECTOR = "selector";
    private static final String BEGINBLOCKCHANGES = "beginBlockChanges";
    private static final String ENDBLOCKCHANGES = "endBlockChanges";
    private static final String ENDEXISTS = "endExists";
    private static final String ATTRCHANGED = "attrChanged";
    private static final String KEY = "key";
    private static final String TYPE = "type";
    private static final String VALUE = "value";
    private static final String ERROR = "error";
    private static final String ERRORSTRING = "errorString";

    /**
     * Protected constructor to ensure that clients get a handle
     * to a CategoryProxy instance via the getCategoryProxy method.
     *
     * @param conn Connection to the server.
     * @param catName Category Name.
     */
    protected CategoryProxy(Connection conn, String catName) {

	super(catName);

	Log.debug(CATEGORYPROXY, "CategoryProxy");
	Log.assert((conn != null && catName != null),
		   "Invalid arguments to CategoryProxy constructor");
	
	_conn = conn;

	Log.debug(CATEGORYPROXY, "Setting up connection.");
	_sysadmProxy = SysadmProxy.get(conn);	
	_serviceName = new String(CATEGORY_SERVICE + "/" + getSelector());
	_helper = ProxyHelper.getProxyHelper(_conn, CATEGORY_SERVICE);
	_conn.addStartedListener(this);
    }

    /**
     * Obtain a handle to a CategoryProxy instance.
     * 
     * @param conn Connection to the server.
     * @param catName Category Name.
     * 
     * @return Category instance.
     */
    public static Category getCategoryProxy(Connection conn, 
					    String catName) {

	Log.debug(CATEGORYPROXY, "getCategoryProxy");
	Log.assert((conn != null && catName != null),
		   "Invalid arguments to getCategoryProxy");

	CategoryProxy proxy = null;

	// Need to synchronize access to _categoryProxies, because
	// CategoryProxies are typically released from within garbage
	// collection thread.
	synchronized (_categoryProxies) {
	    // Get the vector of proxies with catName
	    // The vector returned represents proxy classes for 
	    // categories on different hosts.
	    Vector proxies = (Vector) _categoryProxies.get(catName);
	    boolean createIt = true;

	    if (proxies == null) {

		// No proxies for this catName
		Log.debug(CATEGORYPROXY,
			  "No proxies for this catName " + catName);

		proxies = new Vector (CAPACITY_INCR, CAPACITY_INCR);
		_categoryProxies.put(catName, proxies);
	    
	    } else {

		Log.debug(CATEGORYPROXY, "Proxies exist for this catName "
			  + catName);
		Enumeration enum = proxies.elements();
		while (enum.hasMoreElements()) {
		    // If so, incr ref count, return proxy
		    proxy = (CategoryProxy) enum.nextElement();
		    if (proxy._conn == conn) {
			createIt = false;
			Log.debug(CATEGORYPROXY, "Proxies exist for this conn");
			break;
		    }
		}
	    }

	    // Else, create CategoryProxy and insert into Vector
	    if (createIt) {
		Log.debug(CATEGORYPROXY, 
			  "No proxies exist for this catName/conn");
		proxy = new CategoryProxy(conn, catName);
		proxies.addElement(proxy);
	    }

	    Log.assert(proxy != null, "getCategoryProxy returning a null proxy!");
	    CategoryRef ref = new CategoryRef(proxy);
	    proxy.get();
	    return ref;
	}
    }                               
    
    /**
     * Used by clients of category to increment reference count
     */
    void get() {
	synchronized (_categoryProxies) {
	    _refCount++;
	}
    }
    
    /**
     * Indicate that you are no longer holding on to the CategoryProxy
     * handle obtained via getCategoryProxy
     */
    public void release() {
	
	synchronized (_categoryProxies) {
	    _refCount--;
	    Log.debug(CATEGORYPROXY, "refCount: " + _refCount);

	    if (_refCount == 0) {
		_releasing = true;
		String catName = getSelector();
		Vector proxies = (Vector) _categoryProxies.get(catName);
	
		Log.assert((proxies != null), "No proxies exist for catName " +
			   catName + " ?!!");
	    
		boolean foundIt = proxies.contains(this);
		Log.assert((foundIt), "No proxies exist for my conn  ?!!");

		// unload
		unload();
		proxies.removeElement(this);
		_helper.release();

		if (proxies.size() == 0) {
		    _categoryProxies.remove(catName);
		}
	    }
	}
    }

    /**
     * Translate message from server into the corresponding method
     * invocation on base Category class.
     * 
     * @param packet Packet received.
     */
    public void receivePacket(Packet packet) {
	
	Log.assert(packet != null, "Invalid argument to receivePacket");
	
	if (packet.getSelector().equals(ITEMADDED)) {

	    Item item = new Item(packet.getBundle(ITEM));
	    if (Log.isLevelOn(Log.DEBUG)) {
                // Calling item.toString is really expensive - skip it
                // if it's not going to be used.
		Log.debug(CATEGORYPROXY,
			  "I got an added item: " + item.toString());
	    }
	    if (_receivingListOnRestart) {
		_listOnRestart.put(item.getSelector(), item);
	    } else {
		addItem(item);
	    }

	} else if (packet.getSelector().equals(BEGINBLOCKCHANGES)) {

	    Log.debug(CATEGORYPROXY, "I got a beginBlockChanges");
	    if (_receivingListOnRestart) {
		_skipEndBlockChanges++;
	    } else {
		beginBlockChanges();
	    }

	} else if (packet.getSelector().equals(ENDBLOCKCHANGES)) {

	    Log.debug(CATEGORYPROXY, "I got an endBlockChanges");
	    if (_skipEndBlockChanges > 0) {
		_skipEndBlockChanges--;
	    } else {
		endBlockChanges();
	    }

	} else if (packet.getSelector().equals(ENDEXISTS)) {

	    Log.debug(CATEGORYPROXY, "I got an endExists");
	    if (_receivingListOnRestart) {
		replaceItemList(_listOnRestart);
		_listOnRestart.clear();
		_receivingListOnRestart = false;
	    } else {
		endExists();
	    }

	} else if (packet.getSelector().equals(ITEMCHANGED)) {

	    Log.assert(!_receivingListOnRestart, 
		       "Got changeItem while receivingListOnRestart");
	    Item newItem = new
	        Item(packet.getBundle(NEWITEM));
	    Log.debug(CATEGORYPROXY, "I got newItem: " +
		      newItem.toString());
	    changeItem(newItem);

	} else if (packet.getSelector().equals(ITEMREMOVED)) {

	    Log.assert(!_receivingListOnRestart, 
		       "Got removeItem while receivingListOnRestart");
	    String selector = packet.getAttr(SELECTOR).stringValue();
	    Log.debug(CATEGORYPROXY, "I got a removed item: " +
		      selector);	
	    removeItem(selector);

	} else if (packet.getSelector().equals(ATTRCHANGED)) {

	    Log.debug(CATEGORYPROXY, "I got attrChanged: ");
	    Log.debug(CATEGORYPROXY, "key: " + 
		      packet.getAttr(KEY).stringValue());
	    Log.debug(CATEGORYPROXY, "type: " + 
		      packet.getAttr(TYPE).stringValue());
	    Log.debug(CATEGORYPROXY, "value: " + 
		      packet.getAttr(VALUE).stringValue());

	    Attribute attr = new
	        Attribute(packet.getAttr(KEY).stringValue(),
			  packet.getAttr(TYPE).stringValue(),
			  packet.getAttr(VALUE).stringValue());

	    super.setAttr(attr);	

	} else if (packet.getSelector().equals(ERROR)) {

	    Log.debug(CATEGORYPROXY, "I got error: ");
	    Log.debug(CATEGORYPROXY, "errorString: " +
		      packet.getAttr(ERRORSTRING).stringValue());
	    _conn.notifyError(packet.getAttr(ERRORSTRING).stringValue());

	} else {
	    Log.debug(CATEGORYPROXY, "Received unrecognized packet");
	}
    }
    
    /**
     * Called at initialization, and also after a connection has been
     * lost and then re-established.
     *
     * @param event The connection event.
     */
    public void connectionStarted(ConnectionEvent event) {
	Log.debug(CATEGORYPROXY, "Trying to restart connection");

	synchronized (_categoryProxies) {
	    if (_releasing) {
		return;
	    }	

	    load();

	    // Start the flow of notifications
	    if (isMonitoring()) {	
		while (isInBlockChangesNotification()) {
		    endBlockChanges();
		}

		if (!hasExistsEnded()) {
		    endExists();
		}

		_listOnRestart.clear();
		_receivingListOnRestart = true;
		_skipEndBlockChanges = 0;

		startMonitor();
	    }
	}
    }

    /**
     * Define base class abstract method 
     */
    protected void startMonitor() {
	// This messages starts the flow of
	// notifications from server to client
	Packet packet = new Packet(_serviceName, ADOPTNOTIFICATIONFILTER);
	_conn.sendPacket(packet);

    }

    /**
     * Called when CategoryProxy is constructed and when a 
     * ConnectionStarted event is received.
     *
     * Loads category service plugin and the requested category.
     *
     * Note: the load() function is not symmetrical to the unload()
     * function.  This is because load() is called from
     * connectionRestarted(), which can be called multiple times as
     * the Connection comes and goes.  One-time initialization occurs
     * in the constructor, and connection initialization happens here
     * in load().  unload() is called only once, when the
     * CategoryProxy is released, and hence it releases resources
     * acquired in the constructor and during load().
     */ 
    private void load() {
	
	// Load the category service plugin 
	_sysadmProxy.loadService(CATEGORY_SERVICE, new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(CATEGORYPROXY, 
			  CATEGORY_SERVICE + " service loaded successfully");
		_helper.serviceLoaded();
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "CategoryProxy.Error.loadCategoryService"),   
		    new String[] { event.getReason() }));
	    }

	});	

	// We do not block until loading of category service 
	// plugin returns to send the following load requests
	// Instead we proceed to send the following messages 
	// and handle failure of loading of category service
	// when it is detected.

	// Ask category service plugin to launch the plugin category
	// specified by getSelector() which could be for example 
	// "User" category.
	_helper.sendRequest(categoryRequest(LOAD), new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(CATEGORYPROXY, 
			  getSelector() + "  category loaded successfully");

		if (!_initialized) {	
		    // Set up CategoryProxy to receive
		    // notifications on _conn from the category specified
		    _conn.addPacketListener(_serviceName, CategoryProxy.this);
		    _initialized = true;
		}
	    }

	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "CategoryProxy.Error.loadSpecificCategory"),   
		    new String[] { event.getReason() }));
	    }

	});
    }

    /**
     * Stop receiving packets and unload services that were 
     * previously loaded.
     */
    private void unload() {

	Log.debug(CATEGORYPROXY, "Giving up connection.");

	// Unregister interest in notifications from
	// category plugin
	_conn.removePacketListener(_serviceName);
	_conn.removeStartedListener(this);

	// Unload plugin category
	_helper.sendRequest(categoryRequest(UNLOAD), new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(CATEGORYPROXY, 
			  getSelector() + 
			  " category unloaded successfully");
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "CategoryProxy.Error.unloadSpecificCategory"),   
		    new String[] { event.getReason() }));
	    }
	});

	// Unload the category service plugin 
	_sysadmProxy.unloadService(CATEGORY_SERVICE, new ResultListener() {

	    public void succeeded(ResultEvent event) {
		Log.debug(CATEGORYPROXY, 
			  CATEGORY_SERVICE + " service unloaded successfully");
		_sysadmProxy.release();
	    }
	    public void failed(ResultEvent event) {
			initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "CategoryProxy.Error.unloadCategoryService"),   
		    new String[] { event.getReason() }));
		_sysadmProxy.release();
	    }
	});	
    }

    /**
     * Shared code between load and unload.
     * Create a packet with request info.
     * 
     * @param request Server request (LOAD or UNLOAD).
     * 
     * @return Packet to send to server for this request.
     */
    private Packet categoryRequest(String request) {
	Packet packet = new Packet(CATEGORY_SERVICE, request);
	packet.setAttr(new Attribute("name", getSelector()));
	return packet;	       
    }

    private void initResourceStack() {
	if (_rs == null) {
	    _rs = new ResourceStack();
	    _rs.pushBundle("com.sgi.sysadm.proxy.category.CategoryProxy" +
			   ResourceStack.BUNDLE_SUFFIX);
	}
    }
}


