//
// ProxyHelper.java
//	Helper class for CategoryProxy and AssociationProxy
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.category;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.util.*;
import java.util.*;

/**
 * Helper class for CategoryProxy and AssociationProxy
 */
class ProxyHelper implements PacketListener {

    private Connection _conn;
    private String _serviceName;
    static private long _cookie = 0;
    private Hashtable _requests = new Hashtable();
    private int _refCount = 0;
    private boolean startedListening = false;

    private static Hashtable _helpers = new Hashtable();
    public static final String PROXYHELPER = "ProxyHelper";
    public static final String RESULT = "result";
    public static final String REASON = "reason";
    public static final String COOKIE = "cookie";

    private static final int CAPACITY_INCR = 5;

    /**
     * Protected constructor to ensure that clients get a handle
     * to a ProxyHelper instance via the getProxyHelper method.
     * 
     * @param conn Connection to the server.
     * @param serviceName Name of the service for which a single
     *                    instance of ProxyHelper is required.
     */
    protected ProxyHelper(Connection conn, String serviceName) {
	_conn = conn;
	_serviceName = serviceName;
    }

    /**
     * Obtain a handle to a ProxyHelper instance.
     * 
     * @param conn Connection to the server.
     * @param serviceName Name of the service for which a single
     *                    instance of ProxyHelper is required.
     * 
     */
    public static ProxyHelper getProxyHelper(Connection conn, 
					     String serviceName) {
	ProxyHelper helper = null;

	Vector helpers = (Vector) _helpers.get(serviceName);
	boolean createIt = true;

	if (helpers == null) {
	     Log.debug(PROXYHELPER, "No helpers for this serviceName");
	     helpers = new Vector(CAPACITY_INCR, CAPACITY_INCR);
	     _helpers.put(serviceName, helpers);
	} else {
	    Log.debug(PROXYHELPER, "Helpers exist for this serviceName");
	    Enumeration enum = helpers.elements();
	    while (enum.hasMoreElements()) {
		// If so, incr ref count, return proxy
		helper = (ProxyHelper) enum.nextElement();
		if (helper._conn == conn) {
		    createIt = false;
		    Log.debug(PROXYHELPER, "Helpers exist for this conn");
		    helper.get();
		    break;
		}
	    }
	}

	// Else, create ProxyHelper and insert into Vector
	if (createIt) {
	    Log.debug(PROXYHELPER, 
		      "No helpers exist for this serviceName/conn");
	    helper = new ProxyHelper(conn, serviceName);
	    helper.get();
	    helpers.addElement(helper);
	}	
	Log.assert(helper != null, 
		   "getProxyHelper returning a null helper!");
	return helper;
    }

    /**
     * Used by clients of ProxyHelper to increment reference count
     */
    public synchronized void get() {
	_refCount++;
    }

    /**
     * Indicate that you are no longer holding on to the ProxyHelper
     * handle obtained via getProxyHelper
     */
    public synchronized void release() {
	_refCount--;
	Log.debug(PROXYHELPER, "refCount: " + _refCount);

	if (_refCount == 0) {	
	    Vector helpers = (Vector) _helpers.get(_serviceName);
	
	    Log.assert((helpers != null), "No helpers exist for serviceName " +
		       _serviceName + " ?!!");

	    boolean foundIt = helpers.contains(this);
	    Log.assert((foundIt), "No helpers exist for my conn  ?!!");
	    
	    _conn.removePacketListener(_serviceName);
	    helpers.removeElement(this);

	    if (helpers.size() == 0) {
		_helpers.remove(_serviceName);
	    }
	}	
    }

    /**
     * Send a request to a server.
     * 
     * @param request Packet to be sent
     * @param listener Listener to be notified of reply.
     */
    public void sendRequest(Packet request, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	request.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendPacket(request);
    }

    /**
     * Translate message from server into the corresponding method
     * invocation on ResultListener.
     * 
     * @param packet Packet received.
     */
    public void receivePacket(Packet packet) {
	boolean result = packet.getAttr(RESULT).booleanValue();
	Long cookie = new Long(packet.getAttr(COOKIE).longValue());
	ResultListener listener
	    = (ResultListener)_requests.remove(cookie);
	if (listener == null) {
	    Log.info(PROXYHELPER, "No listener corresponding to cookie");
	    return;
	}

	ResultEvent event = new ResultEvent(this);
	 
	if (result) {
	    listener.succeeded(event);
	} else {
	    Attribute attr = packet.getAttr(REASON);
	    if (attr != null) {
		event.setReason(attr.stringValue());
	    }
	    listener.failed(event);
	} 
    }

    /**
     * Called when Service is successfully loaded.
     */
    public void serviceLoaded() {
	if (!startedListening) {
	    _conn.addPacketListener(_serviceName, this);
	    startedListening = true;
	}
    }
    
}
