//
// SyadmProxy.java
//
//	Client interface to base sysadmd services.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.comm;

import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.proxy.util.*;

/**
 * SysadmProxy provides an interface for interacting with the services
 * provided by sysadmd.
 */
public class SysadmProxy implements PacketListener {

    private Connection _conn;
    private static final long PING_COOKIE = 0;
    private long _cookie = PING_COOKIE + 1;
    private Hashtable _requests = new Hashtable();
    private static Hashtable _proxies = new Hashtable();
    private static Hashtable _pingThreads = new Hashtable();
    private int _refCount = 0;

    /**
     * Packet type that sysadmd responds to.
     */
    public static final String SYSADM_SERVICE = "sysadmd";

    /**
     * Selector for sysadmd's "load" operation.
     */
    public static final String LOAD = "load";

    /**
     * Selector for sysadmd's "unload" operation.
     */
    public static final String UNLOAD = "unload";

    /**
     * Selector for sysadmd's "loadAuth" operation.
     */
    public static final String LOAD_AUTH = "loadAuth";

    /**
     * Selector for sysadmd's "setRemoteHost" operation.
     */
    public static final String SET_REMOTE_HOST = "setRemoteHost";

    /**
     * Selector for sysadmd's "getRemoteHost" operation.
     */
    public static final String GET_REMOTE_HOST = "getRemoteHost";

    /**
     * Selector for sysadmd's "version" operation
     */
    public static final String VERSION = "version";

    /**
     * Selector for sysadmd's "productVersion" operation
     */
    public static final String PRODUCT_VERSION = "productVersion";

    /**
     * cookie attribute name.  This is both an argument and return
     * parameter of all sysadmd operations, and is used to associate
     * asynchronous replies with their corresponding requests.
     */
    public static final String COOKIE = "cookie";

    /**
     * result attribute name.  This is a return parameter of all
     * sysadmd operations.  "result" is a boolean attribute which is
     * "true" if the operation succeeded and "false" if the operation
     * failed.
     */
    public static final String RESULT = "result";

    /**
     * reason attribute name.  This is a return parameter of some
     * sysadmd operations when "result" is "false".  This attribute
     * should not be assumed to be always present, even when "result"
     * is "false".  If present, "reason" is a String attribute
     * containing a key which should be looked up in the sysadmd
     * resource bundle to retrieve a message describing to the user
     * why the operation failed.
     */
    public static final String REASON = "reason";

    /**
     * service attribute name.  This is an argument to the "load"
     * operation which specifies the name of the service to load.
     */
    public static final String SERVICE = "service";

    /**
     * authScheme attribute name.  This is an argument to the
     * "loadAuth" operation and specifies the name of the
     * authentication scheme to load.
     */
    public static final String AUTH_SCHEME = "authScheme";

    /**
     * hostnam attribute name.  This is an argument to the
     * "setHostName" operation which specifies the name
     * of the remote host for sysadmd.
     */
    public static final String REMOTE_HOST = "remoteHostname";
    
    /**
     * Product attribute name.  This is an argument to the
     * "productVersion" operation which specifies the name of
     * the product that the client is interested in.
     */
    public static final String PRODUCT = "product";

    /**
     * The version of a product that this client is using. 
     * This is an argument to the
     * "productVersion" operation which specifies the version of the
     * product that the client is interested in. 
     */
    public static final String CLIENT_PRODUCT_VERSION =
        "clientProductVersion";


    /**
     * The version of a product that this server is using. 
     * This is a return value from the
     * "productVersion" operation which specifies the version of the
     * product that is installed on the server.
     */
    public static final String SERVER_PRODUCT_VERSION =
        "serverProductVersion";
     
    /**
     * A system property that controls the interval in seconds at
     * which the client sends a ping message to the server.  The ping
     * message is used to detect the situation where the network
     * connection to the server is down or the server has been power reset.
     * If this property is not specified, the default used is 60
     * seconds. If the value is 0, then no ping messages are sent.
     */
    public static final String PING_INTERVAL = "SysadmProxy.pingInterval";
    
    /**
     * Selector for sysadmd's "ping" operation.
     */
    private static final String PING = "ping";
    private static final String SYSADMPROXY = "sysadmProxy";
    private static long _pingInterval = 60000;

    static {
	try {
	    String value;
	    if ((value = SysUtil.getSystemProperty(PING_INTERVAL)) != null) {
		_pingInterval = (new Long(value)).longValue() * 1000;
	    }
	} catch (SecurityException ex) {
	}
    }

    /**
     * Get an instance of SysadmProxy for "conn".
     * 
     * @param conn Connection to get a SysadmProxy for.
     * 
     * @return SysadmProxy instance for "conn".
     */
    public static SysadmProxy get(Connection conn) {
	SysadmProxy proxy = (SysadmProxy)_proxies.get(conn);
	if (proxy == null) {
	    proxy = new SysadmProxy(conn);
	    conn.addPacketListener(SYSADM_SERVICE, proxy);
	    _proxies.put(conn, proxy);
	}
	proxy._refCount++;
	return proxy;
    }

    /**
     * Release an instance of SysadmProxy.  After calling release(),
     * this instance of SysadmProxy may no longer be valid.
     */
    public void release() {
	_refCount--;
	if (_refCount == 0) {
	    if (_pingInterval != 0) {
		Thread pingThread = (Thread)_pingThreads.remove(_conn);
		Log.assert(pingThread != null, 
			   "No pingThread for this connection");
		pingThread.stop();
	    }
	    _proxies.remove(_conn);
	    _conn.removePacketListener(SYSADM_SERVICE);
	}
    }

    /**
     * Load a service into sysadmd.  This must be called prior to
     * communicating with a service.
     * 
     * @param serviceName Name of the service to load.
     * @param listener  Gets notified of the success or failure of the
     *                  load.
     */
    public void loadService(String serviceName, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, LOAD);
	packet.setAttr(new Attribute(SERVICE, serviceName));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendPacket(packet);
    }

    /**
     * Unload a service from sysadmd.  This should be called when a
     * service is no longer needed.
     * 
     * @param serviceName Name of the service to unload.
     * @param listener Gets notified of the success or failure of the
     *                 unload.
     */
    public void unloadService(String serviceName, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, UNLOAD);
	packet.setAttr(new Attribute(SERVICE, serviceName));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendPacket(packet);
    }

    /**
     * Load an authentication scheme into sysadmd.  This must be
     * called by Authenticator subclasses to load the corresponding
     * authentication scheme into the server.
     * 
     * @param authScheme Name of authentication scheme to load.
     * @param listener Gets notified of the success or failure of the
     *                 loadAuth.
     */
    public void loadAuth(String authScheme, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, LOAD_AUTH);
	packet.setAttr(new Attribute(AUTH_SCHEME, authScheme));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendAuthPacket(packet);
    }

    /**
     * Set the name of the remote client for sysadmd.  This must be
     * called when using a connection protocol where the peer name of a
     * socket will not return the client hostname.
     * 
     * @param hostName Name of remote client.
     * @param listener Gets notified of the success or failure of the
     *                 setHostName.
     */
    public void setHostName(String hostName, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, SET_REMOTE_HOST);
	packet.setAttr(new Attribute(REMOTE_HOST, hostName));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendAuthPacket(packet);
    }

    /**
     * Get the name of the remote server for the client.  This must be
     * called when using a connection protocol where the peer name of a
     * socket will not return the client hostname.
     * 
     * @param listener Gets notified of the success or failure of the
     *                 setHostName.
     */
    public void getHostName(ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, GET_REMOTE_HOST);
	packet.setAttr(new Attribute(COOKIE, _cookie));
	_conn.sendAuthPacket(packet);
    }
    
    /**
     * Called to verify that the server sysadmd can handle the version
     * of the protocol that this client is requesting.  
     * <p>
     * In general, the client will pass the minimum version of the
     * protocol that is necessary.  The server will typically agree to
     * talk to clients that request a version lower than their current
     * version (for backward compatibility).  This is not a
     * reqirement, though, and if some old functionality is not
     * supported by the server, it may refuse to talk.
     * 
     * @param version The version of the Rhino protocol the client is
     * requesting.  This will be sent to the server to make sure that
     * the server supports this version.
     * @param listener The listener that will be notified of the
     * result of the operation.  <tt>succeeded</tt> will be called if the
     * server accepts the version passed as tt>version</tt>, otherwise
     * <tt>failed</tt> will be called.
     * The versions of the protocol that the server accepts will be
     * returned in the <tt>result</tt> field of the ResultEvent passed
     * to the listener as a Vector of String objects.
     */
    public void checkVersion(String version, ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, VERSION);
	packet.setAttr(new Attribute(VERSION, version));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	// Use sendAuthPacket so that we can send a packet before the
	// connection is officially "up"
	_conn.sendAuthPacket(packet);
    }

    /**
     * Called to verify that the server sysadmd can handle the version
     * of the product (e.g. Failsafe) that the client needs.
     * <p>
     * In general, the client will only communicate with a server
     * whose product version is equal to its own.
     * 
     * @param version The version of the product the client is
     * requesting.  This will be sent to the server to make sure that
     * the server supports this version.
     * @param listener The listener that will be notified of the
     * result of the operation.  <tt>succeeded</tt> will be called if the
     * server accepts the version passed as tt>version</tt>, otherwise
     * <tt>failed</tt> will be called.
     * The version of the protocol that the server accepts will be
     * returned in the <tt>result</tt> field of the ResultEvent passed
     * to the listener as String object.  If the product is not
     * installed on the server at all, an empty string will be sent.
     */
    public void checkProductVersion(String product, String version, 
				    ResultListener listener) {
	_requests.put(new Long(++_cookie), listener);
	Packet packet = new Packet(SYSADM_SERVICE, PRODUCT_VERSION);
	packet.setAttr(new Attribute(PRODUCT, product));
	packet.setAttr(new Attribute(CLIENT_PRODUCT_VERSION, version));
	packet.setAttr(new Attribute(COOKIE, _cookie));
	// Use sendAuthPacket so that we can send a packet before the
	// connection is officially "up"
	_conn.sendAuthPacket(packet);
    }

    /**
     * Called by Connection when a packet arrives from the sysadmd
     * service.  Packets should all be results of various requests.
     * We match the request to its corresponding listener by looking
     * at the cookie.
     * 
     * @param packet Result packet.
     */
    public void receivePacket(Packet packet) {
	boolean result = packet.getAttr(RESULT).booleanValue();
	Long cookie = new Long(packet.getAttr(COOKIE).longValue());
	ResultListener listener
	    = (ResultListener)_requests.remove(cookie);
	if (listener == null) {
	    return;
	}

	ResultEvent event = new ResultEvent(this);

	if (result && packet.getSelector().equals(GET_REMOTE_HOST)) {
	    setRemoteHost(packet);
	}

	if (packet.getSelector().equals(VERSION)) {
	    Vector versions = new Vector();
	    Attribute attr;
	    for (int ii = 0; 
		 (attr = packet.getAttr(VERSION + ii)) != null; 
		 ii++) {
		versions.addElement(attr.getValue());
	    }
	    
	    event.setResult(versions);
	}
	
	if (packet.getSelector().equals(PRODUCT_VERSION)) {
	    Attribute serverProductVersion = 
		packet.getAttr(SERVER_PRODUCT_VERSION);
	    // This is only possible if the server is running a pre-MR
	    // version of the rhino 1.2 product.
	    Log.assert(serverProductVersion != null,
		       "Server didn't understand our productVersion request.");
	    event.setResult(serverProductVersion.stringValue());
	}

	if (result) {
	    listener.succeeded(event);
	} else {
	    Attribute attr = packet.getAttr(REASON);
	    if (attr != null) {
		event.setReason(attr.stringValue());
	    }
	    listener.failed(event);
	}
    }

    /**
     * Sysadm supported method to set the name of the remote host
     *
     * @param packet Packet from server containing remote host name.
     */
    private void setRemoteHost(Packet packet) {
	Attribute attr = packet.getAttr(REMOTE_HOST);
	_conn.setRemoteHostName(attr.stringValue());
    }
	
    /**
     * SysadmProxy constructor.  Should only be called by
     * SysadmProxy.get().
     * 
     * @param conn Connection to use to communicate with sysadmd.
     */
    private SysadmProxy(Connection conn) {
	_conn = conn;
	if (_pingInterval != 0) {
	    startPingThread();
	}
    }

    /**
     * Start the thread that pings the server.  This is used
     * to detect the situation where the server network connection
     * is down or the server has been power reset.  In these cases,
     * the receive thread does not get an IOException and the GUI
     * does not detect that the connection is lost until the
     * user performs an operation that causes network traffic.
     */
    private void startPingThread() {
	Thread pingThread = new Thread(new Runnable() {
	     public void run() {
		 Packet packet = new Packet(SYSADM_SERVICE, PING);
		 packet.setAttr(new Attribute(COOKIE, PING_COOKIE));
		 Log.debug(SYSADMPROXY, packet.toString());
		 while (true) {
		     try {
			 Thread.sleep(_pingInterval);
			 _conn.sendPacket(packet);
		     } catch (InterruptedException e) {
			 Log.debug(SYSADMPROXY,
				   "InterruptedException in startPingThread: "
				   + e.getMessage());
		     }
		}
	     }
	});
	_pingThreads.put(_conn, pingThread);
	pingThread.start();
    }
}
