//
// UnixClientAuth.java
//
//	Client of unix authentication scheme.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.comm;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;

/**
 * Client side of unix authentication scheme.  Send user name and
 * password to server for verification.
 */
public class UnixClientAuth extends Authenticator implements PacketListener {

    private String _name;
    private String _password;

    // String attribute name for password argument.
    static private final String PASSWORD = "password";

    // Name of the unix authentication scheme.
    static private final String UNIX = "unix";

    // Name of the service for the unix authentication scheme.
    static private final String AUTH_UNIX
	= SysadmProxy.AUTH_SCHEME + "/" + UNIX;

    /**
     * Called by Unix Authentication user interface classes to tell us
     * the name and password that the user typed.
     * 
     * @param name login name that the user typed.
     * @param password password that the user typed.
     */
    public void setParams(String name, String password) {
	_name = name;
	_password = password;
    }

    /**
     * Called to authenticate the user with the server.  We send the
     * name and password that the user typed to the server and wait
     * for results.
     */
    public void authenticate() {
	final SysadmProxy proxy = SysadmProxy.get(getConnection());

	proxy.loadAuth(UNIX, new ResultListener() {

 	    // Called when we successfully load the "unix"
 	    // authentication scheme.  Send the name and password to
 	    // the server.
 	    public void succeeded(ResultEvent event) {
		authLoadSucceeded(proxy);
 	    }

 	    // Called if the server couldn't load the "unix"
 	    // authentication scheme.  Tell clients to give up.
 	    public void failed(ResultEvent event) {
		authLoadFailed(proxy);
 	    }
 	});
    }

    /**
     * Called by Connection when a packet arrives from the server-side
     * "unix" authentication scheme.  This packet will contain the
     * result of a login attempt.
     * <p>
     * Note that this method never calls notifyFailed().  Failure to
     * login happens at the Connection level, because the server will
     * eventually exit after a certain number of failed login attempts.
     * 
     * @param packet Packet containing results of login attempt.
     */
    public void receivePacket(Packet packet) {
	getConnection().removePacketListener(AUTH_UNIX);
	if (packet.getAttr(SysadmProxy.RESULT).booleanValue()) {
	    // We successfully logged in!
	    notifySucceeded(new AuthEvent(this));
	} else {
	    // username/password pair was incorrect.  Tell the user
	    // interface classes to prompt the user to re-enter
	    // parameters.
	    notifyRetry(new AuthEvent(this));
	}
    }

    private void authLoadSucceeded(SysadmProxy proxy) {
	getConnection().removePacketListener(AUTH_UNIX);
	getConnection().addPacketListener(AUTH_UNIX, UnixClientAuth.this);
	Packet packet = new Packet(AUTH_UNIX, _name);
	packet.setAttr(new Attribute(PASSWORD, _password));
	packet.setAttrVisible(PASSWORD, false);
	getConnection().sendAuthPacket(packet);
	proxy.release();
    }

    private void authLoadFailed(SysadmProxy proxy) {
	notifyFailed();
	proxy.release();
    }
}
