//
// ProductInfoProxy.java
//
//	Proxy class for ProductInfo
//
//
//  Copyright (c) 1999, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.proxy.ui;

import java.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.proxy.comm.*;
import com.sgi.sysadm.proxy.util.*;
import java.text.MessageFormat;

/*
 * Proxy class for ProductInfo
 */
public class ProductInfoProxy extends ProductInfo 
                                      implements PacketListener,
                                          ConnectionStartedListener,
                                          Releaseable {

    private Connection _conn = null;
    private HostContext _hostContext = null;
    private static Hashtable _productInfos = new Hashtable();

    // Keeps track of the products whose attributes have been
    // requested, but not yet received from the server.  When product
    // attributes are received, the product is removed from this list.
    private static Hashtable _productAttrsRequested = new Hashtable();
    private boolean _initialized = false;
    private int _refCount = 0;
    private SysadmProxy _sysadmProxy = null;
    private boolean _releasing;
    long _nextCookie = 0;	
    private static ResourceStack _rs = null;

    private static final String CLASS_NAME = "ProductInfoProxy";
    private static final String GET_PRODUCT_ATTRIBUTES = "getProductAttrs";
    private static final String COOKIE = "cookie";
    private static final String RESULT = "result";
    private static final String PRODUCT = "product";
    private static final String REASON = "reason";
    private static final String ATTRS = "attrs";
    private static final String PRODUCT_INFO = ProductInfo.PRODUCT_INFO;
    Hashtable _listeners = new Hashtable();

    //
    // ProductInfoRef handles reference counting of ProductInfoProxy
    // instances.
    //
    private class ProductInfoRef extends ProductInfo {
	public void setHostContext(HostContext hostContext) {
	    ProductInfoProxy.this.setHostContext(hostContext);
	}
	public void getProductAttrs(String product, ResultListener listener) {
	    ProductInfoProxy.this.getProductAttrs(product, listener);
	}
	protected void computeProductAttrs(String product) {
	}
	
	
	protected void finalize() throws Throwable {
	    Releaser.release(ProductInfoProxy.this);
	    super.finalize();
	}
    }

    /**
     * Protected constructor to ensure that clients get a handle
     * to a ProductInfo instance via the getProductInfo method
     *
     * @param conn Connection to the server
     */
    protected ProductInfoProxy(Connection conn) {
	Log.assert(conn != null,
		   "Cannot pass null Connection to ProductInfoProxy"
		   + "constructor");
	_conn = conn;	

	Log.debug(CLASS_NAME, "Setting up connection.");
	_sysadmProxy = SysadmProxy.get(conn);

	Log.debug(CLASS_NAME, "Adding restart listener.");
	_conn.addStartedListener(this);
    }

    /**
     * Obtain a handle to a ProductInfo instance.
     * 
     * @param conn Connection to the server.
     * 
     * @return ProductInfo instance.
     */
    public static ProductInfo getProductInfoProxy(Connection conn) {
	// Need to synchronize access to _productInfos, because
	// ProductInfoProxies are typically released from within
	// garbage collection thread.
	synchronized (_productInfos) {
	    Log.assert(conn != null,
		       "Cannot pass null Connection to getProductInfoProxy");

	    ProductInfoProxy proxy = null;

	    // Check if there is an existing ProductInfo for this
	    // connection
	    proxy = (ProductInfoProxy) _productInfos.get(conn);
	
	    if (proxy != null) {
		Log.debug(CLASS_NAME, 
			  "ProductInfo already exists for this connection");
	    } else {
		Log.debug(CLASS_NAME, 
			  "Creating ProductInfo for this connection");
		proxy = new ProductInfoProxy(conn);
		_productInfos.put(conn, proxy);
	    }

	    proxy.get();
	    ProductInfoRef ref = proxy.new ProductInfoRef();
	    return ref;
	}
    }

    /**
     * Set the HostContext needed for creating TaskLoader instances.
     * 
     * @param hostContext HostContext for creating TaskLoader
     *                    instances.
     */
    public void setHostContext(HostContext hostContext) {
	_hostContext = hostContext;
    }

    /**
     * Used by clients of ProductInfo to increment reference count.
     */
    private void get() {
	synchronized (_productInfos) {
	    _refCount++;
	}
    }

    /**
     * Indicate that you are no longer holding on to the ProductInfoProxy
     * handle obtained via getProductInfoProxy.
     */
    public void release() {
	synchronized (_productInfos) {
	    Log.trace(CLASS_NAME, "release: _refCount: " + _refCount);
	    Log.assert(_productInfos.containsKey(_conn),
		       "No product info exists for my connection?!!");
	    _refCount--;
	    if (_refCount == 0) {
		_releasing = true;
		unload();
		_productInfos.remove(_conn);
	    }
	}
    }

    /**
     * Gets the productInfo about a particular product by communicating
     * with ProductInfo on server.
     * 
     * @param product The product (e.g. com.sgi.fsmgr) to get info
     *                about
     */
    protected void computeProductAttrs(String product) {
	// Communicate with ProductInfo on server
	sendProductAttrsRequestPacket(product);	
    }

    /**
     * Translate message from server into the corresponding method
     * invocation on base ProductInfo class.
     * 
     * @param packet Packet received.
     */
    public void receivePacket(Packet packet) {
	if (packet.getSelector().equals(RESULT)) {
	    Log.assert(_productAttrsRequested.containsKey(
		packet.getString(PRODUCT)),
		"Got a result for an unrecognized operation");
	    Log.debug(CLASS_NAME, "Going to call initializeProductInfo");
	    initializeProductAttrs(packet);
	} else {
	    Log.debug(CLASS_NAME, "Got unrecognized packet");
	}
    }

    
    /**
     * Called after a connection has been lost and then
     * re-established.
     *
     * @param event The event representing the error.
     */
    public void connectionStarted(ConnectionEvent event) {
	Log.debug(CLASS_NAME, "Trying to restart connection");
	
	synchronized (_productInfos) {
	    if (_releasing) {
		return;
	    }

	    load();
	}
    }

    /**
     * Initializes _productInfo based on the response from the server
     * 
     * @param packet Packet received.
     */
    private synchronized void initializeProductAttrs(Packet packet) {
	String product = packet.getString(PRODUCT);
	Log.debug(CLASS_NAME, "initializeProductAttrs for: " + product);
	boolean result = packet.getBoolean(RESULT);
	String attrs;
	if (!result) {
	    String reason = packet.getString(REASON);
	    if (reason.equals("NoAttributesDirectory")) {
		Log.error(CLASS_NAME, 
			  "No attributes directory found on server.   " +
			  "Assuming no attributes.");
	    } else {
		Log.error(CLASS_NAME, "Unknown error from server: " + reason + 
			  ".  Assuming no attributes.");
	    }
	    attrs = "";
	} else {
	    attrs = packet.getString(ATTRS);
	}
	setProductAttrs(product, attrs);
	_productAttrsRequested.remove(product);
    }
    
    /**
     * Called when ProductInfoProxy is constructed and when a 
     * ConnectionStarted event is received.
     *
     * Loads ProductInfo service plugin.
     */
    private void load() {
	// Load the ProductInfo plugin 
	_sysadmProxy.loadService(PRODUCT_INFO, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		Log.debug(CLASS_NAME, 
			  PRODUCT_INFO + " service loaded successfully");
		
		// Check if this is the first load
		if (!_initialized) {
		    // Set up ProductInfoProxy to receive
		    // notifications on _conn from ProductInfo
		    _conn.addPacketListener(PRODUCT_INFO, 
					    ProductInfoProxy.this);
		    _initialized = true;
		} else {
		    // resend outstanding requests
		    Enumeration enum = _productAttrsRequested.elements();
		    while (enum.hasMoreElements()) {
			sendRequestForProductAttrs((String)enum.nextElement());
		    }
		}
	    }

	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "ProductInfoProxy.Error.load"),   
		    new String[] { event.getReason() }));
	    }
	});
    }

    /**
     * Stop receiving packets and unload services that were 
     * previously loaded.
     */
    private void unload() {
	
	Log.debug(CLASS_NAME, "Giving up connection.");

	// Unregister interest in notifications from ProductInfo
	_conn.removePacketListener(PRODUCT_INFO);
	_conn.removeStartedListener(this);

	// Unload ProductInfo plugin 
	_sysadmProxy.unloadService(PRODUCT_INFO, new ResultListener() {

	    public void succeeded(ResultEvent event) {
		Log.debug(CLASS_NAME, 
			  PRODUCT_INFO + " service unloaded successfully");
		_sysadmProxy.release();
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "ProductInfoProxy.Error.unload"),   
		    new String[] { event.getReason() }));
		_sysadmProxy.release();
	    }
	});	
    }
    
    private synchronized void sendProductAttrsRequestPacket(String product) {
	// Check if request already sent for the product
	if (!_productAttrsRequested.containsKey(product)) {	   
	    _productAttrsRequested.put(product, product);
	    sendRequestForProductAttrs(product);
	} else {
	    Log.debug(CLASS_NAME, 
		      "Already sent message to server for product: " 
		      + product);
	}
    }

    private void initResourceStack() {
	if (_rs == null) {
	    _rs = new ResourceStack();
	    _rs.pushBundle("com.sgi.sysadm.proxy.ui.ProductInfoProxy" +
			   ResourceStack.BUNDLE_SUFFIX);
	}
    }

    private void sendRequestForProductAttrs(String product) {
	Packet packet = new Packet(PRODUCT_INFO, GET_PRODUCT_ATTRIBUTES);
	packet.setString(PRODUCT, product);
	packet.setLong(COOKIE, _nextCookie++);
	_conn.sendPacket(packet);
	Log.debug(CLASS_NAME, 
		  "Sending message to server for product: " + product);
    }
}



