//
// RemoteHostContext.java
//
//	HostContext subclass that represents a remote host.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.ui;

import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.proxy.category.*;
import com.sgi.sysadm.proxy.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.*;

/**
 * HostContext provides a mechanism for clients to get access to
 * services on a remote host for system administration.
 */
public class RemoteHostContext extends HostContext {

    Connection _conn;
    PrivBroker _privBroker;

    /**
     * Construct a RemoteHostContext.
     * 
     * @param conn Connection to the remote host.
     */
    public RemoteHostContext(Connection conn) {
	_conn = conn;
    }

    /**
     * Get a Category for a host.
     * 
     * @param name Name of Category to get.
     * 
     * @return Category
     */
    public Category getCategory(String name) {
	return CategoryProxy.getCategoryProxy(_conn, name);
    }

    /**
     * Get an Association for a host.
     * 
     * @param parentCategory The type of the parent Item.
     * @param parentSelector The selector of the parent Item.
     * @param childCategory The type of the related Item(s).
     * 
     * @return Association
     */
    public Association getAssociation(String parentCategory, 
				      String parentSelector,
				      String childCategory) {
	return AssociationProxy.getAssociationProxy(_conn,
						    parentCategory,
						    parentSelector,
						    childCategory);
    }

    /**
     * Get the PrivBroker instance for a host.
     * 
     * @return PrivBroker instance.
     */
    public PrivBroker getPrivBroker() {
	/*
	 * PrivBrokerProxy caches the root password.  It would be
	 * really annoying to have to re-type the password at
	 * seemingly arbitrary times (when all PrivBrokerRefs get
	 * garbage collected, and then another one is created), so we
	 * cache it.
	 */
	if (_privBroker == null) {
	    _privBroker = PrivBrokerProxy.getPrivBroker(_conn);
	}
	return _privBroker;
    }

    /**
     * Get the TaskRegistry instance for a host.
     * 
     * @return TaskRegistry instance.
     */
    public TaskRegistry getTaskRegistry() {
	TaskRegistry reg = TaskRegistryProxy.getTaskRegistryProxy(_conn);
	reg.setHostContext(this);
	return reg;
    }

    /**
     * Get the ProductInfo instance for a host.
     * 
     * @return ProductInfo instance.
     */
    public ProductInfo getProductInfo() {
	ProductInfo pi = ProductInfoProxy.getProductInfoProxy(_conn);
	pi.setHostContext(this);
	return pi;
    }

    /**
     * Get the name of the host this HostContext represents.
     * 
     * @return name of the host this HostContext represents.
     */
    public String getHostName() {
	return _conn.getRemoteHostName();
    }

    /**
     * Access to Connection to the server.
     * 
     * @return Connection to the server.
     */
    public Connection getConnection() {
	return _conn;
    }

    /**
     * Override parent class exit() in order to close our Connection.
     * 
     * @param exitStatus exit status.
     */
    public void exit(final int exitStatus) {
	// We have to do this in another thread for it to be able to
	// work from the connection thread.  Calling _conn.close()
	// stops the connection thread, so if we're running in the
	// connection thread we'd never run super.exit().
	new Thread() {
	    public void run() {
		_conn.close();
		superExit(exitStatus);
	    }
	}.start();
    }

    /**
     * Workaround the fact that there's no way to call
     * HostContext.exit() from within a nested class.
     * 
     * @param status 
     */
    private void superExit(int status) {
	super.exit(status);
    }
}
