//
// TaskRegistryProxy.java
//
//	Proxy class for TaskRegistry
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.proxy.ui;

import java.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.proxy.comm.*;
import com.sgi.sysadm.proxy.util.*;
import java.text.MessageFormat;

/*
 * Proxy class for TaskRegistry
 */
public class TaskRegistryProxy extends TaskRegistry 
                               implements PacketListener,
                                          ConnectionStartedListener,
                                          Releaseable {

    private Connection _conn = null;
    private HostContext _hostContext = null;
    private static Hashtable _taskRegistries = new Hashtable();
    private static Hashtable _categoriesRequested = new Hashtable();
    private boolean _initialized = false;
    private int _refCount = 0;
    private SysadmProxy _sysadmProxy = null;
    private boolean _releasing;
    long _nextCookie = 0;	
    private static ResourceStack _rs = null;

    private static final String TASKREGISTRYPROXY =
        "taskRegistryProxy";
    private static final String GETTASKLIST = "getTaskList";
    private static final String COOKIE = "cookie";
    private static final String CATEGORY = "category";
    private static final String RESULT = "result";
    private static final String NUMTASKS = "numTasks";
    private static final String PARAM = "param";
    private static final String TASK_REGISTRY = TaskRegistry.TASK_REGISTRY;

    Hashtable _listeners = new Hashtable();

    //
    // TaskRegistryRef handles reference counting of TaskRegistryProxy
    // instances.
    //
    private class TaskRegistryRef extends TaskRegistry {
	public void setHostContext(HostContext hostContext) {
	    TaskRegistryProxy.this.setHostContext(hostContext);
	}
	public void getTaskList(String catName, ResultListener listener) {
	    TaskRegistryProxy.this.getTaskList(catName, listener);
	}
	public void getTaskList(Item item, ResultListener listener) {
	    TaskRegistryProxy.this.getTaskList(item, listener);
	}
	protected void computeTaskList(String catName) {
	}
	protected void finalize() throws Throwable {
	    Releaser.release(TaskRegistryProxy.this);
	    super.finalize();
	}
    }

    /**
     * Protected constructor to ensure that clients get a handle
     * to a TaskRegistry instance via the getTaskRegistry method
     *
     * @param conn Connection to the server
     */
    protected TaskRegistryProxy(Connection conn) {
	Log.assert(conn != null,
		   "Cannot pass null Connection to TaskRegistryProxy"
		   + "constructor");
	_conn = conn;	

	Log.debug(TASKREGISTRYPROXY, "Setting up connection.");
	_sysadmProxy = SysadmProxy.get(conn);

	Log.debug(TASKREGISTRYPROXY, "Adding restart listener.");
	_conn.addStartedListener(this);
    }

    /**
     * Obtain a handle to a TaskRegistry instance.
     * 
     * @param conn Connection to the server.
     * 
     * @return TaskRegistry instance.
     */
    public static TaskRegistry getTaskRegistryProxy(Connection conn) {
	// Need to synchronize access to _taskRegistries, because
	// TaskRegistryProxies are typically released from within
	// garbage collection thread.
	synchronized (_taskRegistries) {
	    Log.assert(conn != null,
		       "Cannot pass null Connection to getTaskRegistryProxy");

	    TaskRegistryProxy proxy = null;

	    // Check if there is an existing TaskRegistry for this
	    // connection
	    proxy = (TaskRegistryProxy) _taskRegistries.get(conn);
	
	    if (proxy != null) {
		Log.debug(TASKREGISTRYPROXY, 
			  "TaskRegistry already exists for this connection");
	    } else {
		Log.debug(TASKREGISTRYPROXY, 
			  "Creating TaskRegistry for this connection");
		proxy = new TaskRegistryProxy(conn);
		_taskRegistries.put(conn, proxy);
	    }

	    proxy.get();
	    TaskRegistryRef ref = proxy.new TaskRegistryRef();
	    return ref;
	}
    }

    /**
     * Set the HostContext needed for creating TaskLoader instances.
     * 
     * @param hostContext HostContext for creating TaskLoader
     *                    instances.
     */
    public void setHostContext(HostContext hostContext) {
	_hostContext = hostContext;
    }

    /**
     * Used by clients of TaskRegistry to increment reference count.
     */
    private void get() {
	synchronized (_taskRegistries) {
	    _refCount++;
	}
    }

    /**
     * Indicate that you are no longer holding on to the TaskRegistryProxy
     * handle obtained via getTaskRegistryProxy.
     */
    public void release() {
	synchronized (_taskRegistries) {
	    Log.trace(TASKREGISTRYPROXY, "release: _refCount: " + _refCount);
	    Log.assert(_taskRegistries.containsKey(_conn),
		       "No task registry exists for my connection?!!");
	    _refCount--;
	    if (_refCount == 0) {
		_releasing = true;
		unload();
		_taskRegistries.remove(_conn);
	    }
	}
    }

    /**
     * Determines the tasks relevant to a Category by communicating
     * with TaskRegistry on server.
     * 
     * @param catName Category Name.
     */
    protected void computeTaskList(String catName) {
	// Communicate with TaskRegistry on server
	sendTaskListRequestPacket(catName);	
    }

    /**
     * Translate message from server into the corresponding method
     * invocation on base TaskRegistry class.
     * 
     * @param packet Packet received.
     */
    public void receivePacket(Packet packet) {
	if (packet.getSelector().equals(RESULT)) {
	    Log.assert(
		_categoriesRequested.containsKey(packet.getString(CATEGORY)),
		"Got a result for an unrecognized operation");
	    Log.debug(TASKREGISTRYPROXY, "Going to call initializeTaskList");
	    initializeTaskList(packet);
	} else {
	    Log.debug(TASKREGISTRYPROXY, "Got unrecognized packet");
	}
    }

    
    /**
     * Called after a connection has been lost and then
     * re-established.
     *
     * @param event The event representing the error.
     */
    public void connectionStarted(ConnectionEvent event) {
	Log.debug(TASKREGISTRYPROXY, "Trying to restart connection");
	
	synchronized (_taskRegistries) {
	    if (_releasing) {
		return;
	    }

	    load();
	}
    }

    /**
     * Initializes _taskList based on the response from the server
     * 
     * @param packet Packet received.
     */
    private synchronized void initializeTaskList(Packet packet) {
	Log.debug(TASKREGISTRYPROXY, "In initializeTaskList");

	String catName = packet.getString(CATEGORY);

	Log.debug(TASKREGISTRYPROXY, "initializeTaskList for " + catName);
	// safe to do so, 'cos at the server end it is an int
	int taskListSize = (int) packet.getLong(NUMTASKS);
	Vector taskList = new Vector(taskListSize);
        
        // Create a loader for the Task.
	TaskLoader taskLoader = null;

	for (int ii = 0; ii < taskListSize ; ii++) {
	    if (Log.isLevelOn(Log.DEBUG)) {
		// String concats are expensive - skip them
		// if they're not going to be used.
		Log.debug(TASKREGISTRYPROXY, PARAM + ii);
		Log.debug(TASKREGISTRYPROXY,
			  packet.getString(PARAM + ii));
	    }
	    try {
		    
		taskLoader = 
		    new TaskLoader(_hostContext, 
				   packet.getString(PARAM+String.valueOf(ii)));

	    } catch (MissingResourceException exception) {

		Log.warning(TASKREGISTRYPROXY, exception.getMessage());
		continue;

	    }
	    taskList.addElement(taskLoader);
	}

	_categoriesRequested.remove(catName);

	TaskLoader[] taskArray = new TaskLoader[taskList.size()];
	taskList.copyInto(taskArray);

	setTaskList(catName, taskArray);
    }
    
    /**
     * Called when TaskRegistryProxy is constructed and when a 
     * ConnectionStarted event is received.
     *
     * Loads taskRegistry service plugin.
     */
    private void load() {
	
	// Load the taskRegistry plugin 
	_sysadmProxy.loadService(TASK_REGISTRY, new ResultListener() {
	    
	    public void succeeded(ResultEvent event) {
		Log.debug(TASKREGISTRYPROXY, 
			  TASK_REGISTRY + " service loaded successfully");
		
		// Check if this is the first load
		if (!_initialized) {
		    // Set up TaskRegistryProxy to receive
		    // notifications on _conn from TaskRegistry
		    _conn.addPacketListener(TASK_REGISTRY, 
					    TaskRegistryProxy.this);
		    _initialized = true;
		} else {
		    // resend outstanding requests
		    Enumeration enum = _categoriesRequested.elements();
		    String catName;
		    while (enum.hasMoreElements()) {
			catName = (String) enum.nextElement();
			sendRequest(catName);
		    }
		}
	    }

	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "TaskRegistryProxy.Error.load"),   
		    new String[] { event.getReason() }));
	    }
	});
    }

    /**
     * Stop receiving packets and unload services that were 
     * previously loaded.
     */
    private void unload() {
	
	Log.debug(TASKREGISTRYPROXY, "Giving up connection.");

	// Unregister interest in notifications from TaskRegistry
	_conn.removePacketListener(TASK_REGISTRY);
	_conn.removeStartedListener(this);

	// Unload taskRegistry plugin 
	_sysadmProxy.unloadService(TASK_REGISTRY, new ResultListener() {

	    public void succeeded(ResultEvent event) {
		Log.debug(TASKREGISTRYPROXY, 
			  TASK_REGISTRY + " service unloaded successfully");
		_sysadmProxy.release();
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "TaskRegistryProxy.Error.unload"),   
		    new String[] { event.getReason() }));
		_sysadmProxy.release();
	    }
	});	
    }
    
    private synchronized void sendTaskListRequestPacket(String catName) {
	// Check if request already sent for catName
	if (!_categoriesRequested.containsKey(catName)) {	   
	    _categoriesRequested.put(catName, catName);
	    sendRequest(catName);
	} else {
	    Log.debug(TASKREGISTRYPROXY, 
		      "Already sent message to server for category " 
		      + catName);
	}
    }

    private void initResourceStack() {
	if (_rs == null) {
	    _rs = new ResourceStack();
	    _rs.pushBundle("com.sgi.sysadm.proxy.ui.TaskRegistryProxy" +
			   ResourceStack.BUNDLE_SUFFIX);
	}
    }

    private void sendRequest(String catName) {
	Packet packet = new Packet(TASK_REGISTRY, GETTASKLIST);
	packet.setString(CATEGORY, catName);
	packet.setLong(COOKIE, _nextCookie++);
	_conn.sendPacket(packet);
	Log.debug(TASKREGISTRYPROXY, 
		  "Sending message to server for category " + catName);
    }
}
