//
// PrivBrokerProxy.java
//
//	API for dealing with privileges.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.util;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;
import com.sgi.sysadm.proxy.comm.*;
import java.util.*;
import java.text.*;

/**
 * PrivBrokerProxy provides an API for checking, adding, and running
 * privileges on the server.
 */
public class PrivBrokerProxy implements PrivBroker, PacketListener,
                                        ConnectionStartedListener,
                                        Releaseable {

    static Hashtable _proxies = new Hashtable();
    static int _refCount = 0;
    private boolean _initialized = false;
    private boolean _releasing = false;

    // Service name
    static final String PRIVILEGE = "privilege";

    // Requests we send to the server.
    static final String RUN = "run";
    static final String CHECK = "check";
    static final String ADD = "add";
    static final String INPUT = "input";
    static final String DESTROY = "destroy";
    static final String CHECK_PASSWORD = "checkPassword";

    // Asynchronous events we get from the server.
    static final String OUTPUT = "output";
    static final String ERROR = "error";
    static final String FINISHED = "finished";

    // Attribute keys.
    static final String JOB_ID = "jobId";
    static final String ARGC = "argc";
    static final String ARG = "arg";
    static final String ARGS = "args";
    static final String COOKIE = "cookie";
    static final String NUM_PRIVS = "numPrivs";
    static final String PRIV = "priv";
    static final String EXIT_VALUE = "exitValue";
    static final String PASSWORD = "password";
    static final String RESULT = "result";
    static final String USER = "user";
    static final String REASON = "reason";

    // Reasons why checkpriv can fail.
    static final int NOT_PRIVILEGED = -1;
    static final int NO_SUCH_PRIV = -2;
    
    // Reasons why runpriv can fail.
    static final int SYSTEM_ERROR = 11;
 
    Connection _conn = null;
    long _nextJobId = 0;
    long _nextCookie = 0;
    Hashtable _procs = new Hashtable();
    Hashtable _privsStarted = new Hashtable();
    Hashtable _listeners = new Hashtable();
    Hashtable _privsChecked = new Hashtable();
    Hashtable _privListTable = new Hashtable();
    String _password = null;
    ResourceStack _rs = null;

    // Cache privRequests so that we can resend ADD, CHECK and 
    // CHECK_PASSWORD requests in case of connection failure.
    private class Request {
	public Request(String request, String privs[], String user,
		       ResultListener listener) {
	    _request = request;
	    _privs = privs;
	    _listener = listener;
	    _user = user;
	}

	private String _request;
	private String _privs[];
	private ResultListener _listener;
	private String _user;
    }

    
    //
    // PrivBrokerRef takes care of reference counting PrivBrokerProxy
    // instances.
    //
    private class PrivBrokerRef implements PrivBroker {
	public void setPassword(String password) {
	    PrivBrokerProxy.this.setPassword(password);
	}
	public void checkPassword(ResultListener listener) {
	    PrivBrokerProxy.this.checkPassword(listener);
	}
	public Process runPriv(String privilege, String command) {
	    return PrivBrokerProxy.this.runPriv(privilege, command);
	}
	public Process runPriv(String privilege, String args[]) {
	    return PrivBrokerProxy.this.runPriv(privilege, args);
	}
	public Process runPriv(String privilege, AttrBundle args) {
	    return PrivBrokerProxy.this.runPriv(privilege, args);
	}
	public void checkPrivs(String privs[], ResultListener listener) {
	    PrivBrokerProxy.this.checkPrivs(privs, listener);
	}
	public void addPrivs(String privs[], String user,
			    ResultListener listener) {
	    PrivBrokerProxy.this.addPrivs(privs, user, listener);
	}
	protected void finalize() throws Throwable {
	    Releaser.release(PrivBrokerProxy.this);
	    super.finalize();
	}
    }

    /**
     * Construct a PrivBrokerProxy.  This is private because you're
     * supposed to use the static get() method.
     * 
     * @param conn Connection to server.
     */
    private PrivBrokerProxy(Connection conn) {
	_conn = conn;
	conn.addStartedListener(this);
    }
    
    /**
     * Called after a connection has been lost and then
     * re-established.
     *
     * @param event The event representing the error.
     */
    public void connectionStarted(ConnectionEvent event) {
	synchronized (_proxies) {
	    if (_releasing) {
		return;
	    }
	    loadService();
	}
    }

    /**
     * Get an instance of PrivBroker for the server we're
     * connected to via <TT>conn</TT>
     * 
     * @param conn Connection to server.
     * 
     * @return PrivBroker instance.
     */
    static public PrivBroker getPrivBroker(Connection conn) {
	// Need to synchronize access to _proxies, because
	// PrivBrokerProxies are typically released from within
	// garbage collection thread.
	synchronized (_proxies) {
	    PrivBrokerProxy proxy = (PrivBrokerProxy)_proxies.get(conn);
	    if (proxy == null) {
		proxy = new PrivBrokerProxy(conn);
		_proxies.put(conn, proxy);
	    }
	    proxy.get();
	    PrivBrokerRef ref = proxy.new PrivBrokerRef();
	    return ref;
	}
    }

    /**
     * Used by clients of PrivBroker to increment reference cound.
     */
    private void get() {
	synchronized(_proxies) {
	    _refCount++;
	}
    }

    /**
     * Release an instance of PrivBroker.  PrivBroker reference is no
     * longer valid.
     */
    public void release() {
	synchronized (_proxies) {
	    if (--_refCount == 0) {
		_releasing = true;
		Log.trace(PRIVILEGE, "Removing packet listener");
		_conn.removePacketListener(PRIVILEGE);
		_conn.removeStartedListener(this);

		_proxies.remove(_conn);

		final SysadmProxy sp = SysadmProxy.get(_conn);
		sp.unloadService(PRIVILEGE, new ResultListener() {
		    public void succeeded(ResultEvent event) {
			sp.release();
		    }
		    public void failed(ResultEvent event) {
			initResourceStack();
			_conn.notifyError(MessageFormat.format(_rs.getString(
			    "PrivBrokerProxy.Error.unload"),   
                            new String[] { event.getReason() }));
			sp.release();
		    }
		});
	    }
	}
    }

    /**
     * Set the password to be passed to the server when trying to run
     * commands with runpriv.
     * 
     * @param password password to send to the server.
     */
    public void setPassword(String password) {
	_password = password;
    }

    /**
     * Check that the current password (set via setPassord()) is
     * valid.
     *
     * @param listener Gets notified if password is correct.
     */
    public void checkPassword(ResultListener listener) {
	privRequest(CHECK_PASSWORD, null, null, listener);
    }

    /**
     * Run a privileged command on the server.
     * 
     * @param privilege The privilege to run.
     * @param command Command line arguments.
     * 
     * @return Process object for the privileged command on the
     *         server.
     */
    public Process runPriv(String privilege, String command) {
	StringTokenizer st = new StringTokenizer(command);
	String args[] = new String[st.countTokens()];
	int i = 0;
	while (st.hasMoreElements()) {
	    args[i++] = st.nextToken();
	}
	return runPriv(privilege, args);
    }

    /**
     * Run a privileged command on the server.
     * 
     * @param privilege The privilege to run.
     * @param args Arguments to the privileged command.
     * 
     * @return Process object for the privileged command on the
     *         server.
     */
    public Process runPriv(String privilege, String args[]) {
	Packet packet = createRunPrivPacket(privilege);
	packet.setLong(ARGC, args.length);
	for (int i = 0; i < args.length; i++) {
	    packet.setString(ARG + String.valueOf(i), args[i]);
	}
	return runPriv(packet);
    }

    /**
     * Run a privileged command on the server.
     * 
     * @param privilege The privilege to run.
     * @param args Arguments to the privileged command.  These will
     *             end up on the command line in "key=value" form.
     * 
     * @return Process object for the privileged command on the
     *         server.
     */
    public Process runPriv(String privilege, AttrBundle args) {
	Packet packet = createRunPrivPacket(privilege);
	packet.setBundle(ARGS, args);
	return runPriv(packet);
    }


    /**
     * Check with the server to see if we're authorized to run a set
     * of privileges.
     * 
     * @param privs Set of privileges to check.
     * @param listener Gets the result of the check.
     */
    public void checkPrivs(String privs[],
			   ResultListener listener) {
	_privListTable.put(new Long(_nextCookie), privs);
	privRequest(CHECK, privs, null, listener);
    }

    /**
     * Add privileges for ourselves to the server.
     * 
     * @param privs Set of privileges to add.
     * @param user user name to add privileges for.
     * @param listener Gets the result of the add.
     */
    public void addPrivs(String privs[], String user,
			 ResultListener listener) {
	privRequest(ADD, privs, user, listener);
    }

    /**
     * Run the privileged command corresponding to "packet" on the
     * server.
     * 
     * @param packet Specifies privileged command and args.
     * 
     * @return Process object for the privileged command on the
     *         server.
     */
    private Process runPriv(Packet packet) {
	Log.assert(_privsChecked.get(packet.getString(PRIVILEGE)) != null,
		   "You must use checkPrivs() to check each privilege you run."
		   + "\nTo do this, you can define the Task.privList resource set"
		   + "\nin your task's properties file.");
	ProcessProxy proc = new ProcessProxy(_conn, _nextJobId);
	_procs.put(new Long(_nextJobId++), proc);
	_privsStarted.put(proc, packet.getString(PRIVILEGE));
	_conn.sendPacket(packet);
	return proc;
    }

    /**
     * Create a packet to be passed to the privilege service on the
     * server for running a privilege.  Does not include any of the
     * arguments.
     * 
     * @param privilege Privilege to run.
     * 
     * @return Packet for starting a privileged command on the
     *         server.
     */
    private Packet createRunPrivPacket(String privilege) {
	Packet packet = new Packet(PRIVILEGE, RUN);
	packet.setString(PRIVILEGE, privilege);
	packet.setLong(JOB_ID, _nextJobId);
	if (_password != null) {
	    packet.setString(PASSWORD, _password);
	    packet.setAttrVisible(PASSWORD, false);
	}
	return packet;
    }
	
    /**
     * Shared code between checkPassword, checkPrivs and addPrivs.  
     * Create a packet for the request.
     * 
     * @param request Server request (CHECK_PASSWORD, CHECK or ADD).
     * @param privs Set of privileges for this request.
     * @param user user name, for ADD request.
     * @param listener Gets notified of the result of this request.
     */
    private synchronized void privRequest(String request,
					    String privs[],
					    String user,
					    ResultListener listener) {
	Packet packet = createPrivRequestPacket(request, privs, user);
	packet.setLong(COOKIE, _nextCookie);
	_listeners.put(new Long(_nextCookie++), 
		       new Request(request, privs, user, listener));
	_conn.sendPacket(packet);
    }

    /**
     * Shared code between privRequest and loadService.  
     * Create a packet for the request.
     * 
     * @param request Server request (CHECK_PASSWORD, CHECK or ADD).
     * @param privs Set of privileges for this request.
     * @param user user name, for ADD request.
     * 
     * @return Packet to send to server for this request.
     */
    private Packet createPrivRequestPacket(String request,
					   String privs[],
					   String user) {
	Packet packet = new Packet(PRIVILEGE, request);
	if (_password != null) {
	    packet.setString(PASSWORD, _password);
	    packet.setAttrVisible(PASSWORD, false);
	}
	if (privs != null) {
	    packet.setLong(NUM_PRIVS, privs.length);
	    for (int i = 0; i < privs.length; i++) {
		packet.setString(PRIV + String.valueOf(i), privs[i]);
	    }
	}
	if (user != null) {
	    packet.setString(USER, user);
	}
	return packet;
    }

    /**
     * Get the ProcessProxy corresponding to the jobid from
     * <TT>packet</TT>.
     * 
     * @param packet Packet specifying jobId.
     * 
     * @return ProcessProxy corresponding to jobId of <TT>packet</TT>.
     */
    private ProcessProxy getProc(Packet packet) {
	ProcessProxy proc
	    = (ProcessProxy)_procs.get(new Long(packet.getLong(JOB_ID)));
	Log.assert(proc != null,
		   "Got a message for a jobid that doesn't exist");
	return proc;
    }

    /**
     * Called by Connection when a packet arrives for us.
     * 
     * @param packet Packet from Connection.
     */
    public void receivePacket(Packet packet) {
	if (packet.getSelector().equals(OUTPUT)) {
	    getProc(packet).addOutput(packet.getData());
	} else if (packet.getSelector().equals(ERROR)) {
	    getProc(packet).addError(packet.getData());
	} else if (packet.getSelector().equals(FINISHED)) {
	    ProcessProxy proc = getProc(packet);
	    proc.finished(packet.getLong(EXIT_VALUE));
	    _procs.remove(new Long(packet.getLong(JOB_ID)));
	    _privsStarted.remove(proc);
	} else if (packet.getSelector().equals(CHECK_PASSWORD)) {
	    long cookie = packet.getLong(COOKIE);
	    ResultListener listener
		= ((Request)_listeners.remove(new Long(cookie)))._listener;
	    Log.assert(listener != null,
		       "Got a result for an unrecognized operation");
	    ResultEvent event = new ResultEvent(this);
	    if (packet.getBoolean(RESULT)) {
		listener.succeeded(event);
	    } else {
		listener.failed(event);
	    }
	} else if (packet.getSelector().equals(RESULT)) {
	    long cookie = packet.getLong(COOKIE);
	    ResultListener listener
		= ((Request)_listeners.remove(new Long(cookie)))._listener;
	    Log.assert(listener != null,
		       "Got a result for an unrecognized operation");

	    ResultEvent event = new ResultEvent(this);
	    if (packet.getBoolean(RESULT)) {
		// Remember which privileges we have so we can assert
		// that they've been checked in runPriv().
		String privs[] = (String[])
		    _privListTable.remove(new Long(cookie));
		if (privs != null) {
		    for (int ii = 0; ii < privs.length; ii++) {
			_privsChecked.put(privs[ii], privs[ii]);
		    }
		    listener.succeeded(event);
		}
	    } else {
		initResourceStack();
		Attribute attr = packet.getAttr(REASON);
		if (attr != null && attr.longValue() == NO_SUCH_PRIV) {
		    event.setResult(REASON_NO_SUCH_PRIV);
		    event.setReason(MessageFormat.format(_rs.getString(
			"PrivBrokerProxy.Error.noSuchPriv"),
			 new String[] { packet.getString(PRIV) }));
		} else {
		    event.setResult(REASON_NOT_PRIVILEGED);
		    event.setReason(_rs.getString(
			"PrivBrokerProxy.Error.noPrivileges"));
		}
		listener.failed(event);
	    }
	}
    }

    /**
     * Load the "privilege" service on the server.
     */
    private void loadService() {
	final SysadmProxy sp = SysadmProxy.get(_conn);
	sp.loadService(PRIVILEGE, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		if (!_initialized) {
		    Log.trace(PRIVILEGE, "Adding packet listener");
		    _conn.addPacketListener(PRIVILEGE, PrivBrokerProxy.this);
		    _initialized = true;
		} else {
		    // Notify error for outstanding privcmds.
		    if (_procs.size() > 0) {
			initResourceStack();
			
			Enumeration enum = _procs.elements();
			ProcessProxy proc;
			String errorString;
			while (enum.hasMoreElements()) {
			    proc = (ProcessProxy) enum.nextElement();
			    errorString = 
				MessageFormat.format(
				    _rs.getString(
				     "PrivBrokerProxy.Error.cmdConnLost"),
				     new String[] { 
					 (String) _privsStarted.get(proc)});
			    proc.addError(errorString.getBytes());
			    proc.finished(SYSTEM_ERROR);
			}
			_procs.clear();
			_privsStarted.clear();
		    }
		    // Resend any outstanding privRequests.
		    if (_listeners.size() > 0) {
			Enumeration enum = _listeners.keys();
			Long cookie;
			Request request;
			while (enum.hasMoreElements()) {
			    cookie = (Long) enum.nextElement();
			    request = (Request) _listeners.get(cookie);
			    Packet packet = 
			        createPrivRequestPacket(request._request,
							request._privs,
							request._user);
			    packet.setLong(COOKIE, cookie.longValue());
			    _conn.sendPacket(packet);
			}
		    }
		}
		sp.release();
	    }
	    public void failed(ResultEvent event) {
		initResourceStack();
		_conn.notifyError(MessageFormat.format(_rs.getString(
		    "PrivBrokerProxy.Error.load"), 
		    new String[] { event.getReason() }));
		sp.release();
	    }
	});
    }

    private void initResourceStack() {
	if (_rs == null) {
	    _rs = new ResourceStack();
	    _rs.pushBundle("com.sgi.sysadm.proxy.util.PrivBrokerProxy" +
			   ResourceStack.BUNDLE_SUFFIX);
	}
    }
}
