//
// ProcessProxy.java
//
//	Proxy for a Process running on the server.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.util;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;
import java.io.*;

/**
 * ProcessProxy is a proxy for a Process running on the server.
 * Current implementation is specific to privileged commands.
 */
class ProcessProxy extends Process {

    boolean _exited = false;
    int _exitValue = 0;
    Connection _conn;
    long _jobId;
    RemoteInputStream _input = new RemoteInputStream();
    RemoteInputStream _error = new RemoteInputStream();
    RemoteOutputStream _output;

    /**
     * Construct a ProcessProxy.
     * 
     * @param conn Connection with server.
     * @param jobId distinguish this job from others on both client
     *              and server.
     */
    ProcessProxy(Connection conn, long jobId) {
	_conn = conn;
	_jobId = jobId;
	Packet packet = new Packet(PrivBrokerProxy.PRIVILEGE,
				   PrivBrokerProxy.INPUT);
	packet.setLong(PrivBrokerProxy.JOB_ID, jobId);
	_output = new RemoteOutputStream(conn, packet);
    }

    /**
     * Get the output stream for writing to process's standard input.
     * 
     * @return output stream for writing to process's standard input.
     */
    public OutputStream getOutputStream() {
	return _output;
    }

    /**
     * Get the input stream for reading from process's standard output.
     * 
     * @return input stream for reading from process's standard output.
     */
    public InputStream getInputStream() {
	return _input;
    }

    /**
     * Get the input stream for reading from the process's standard
     * error. 
     * 
     * @return input stream for reading from the process's standard
     *         error. 
     */
    public InputStream getErrorStream() {
	return _error;
    }

    /**
     * Wait for this Process to exit.
     * 
     * @exception InterruptedException if this thread is interrupted.
     * 
     * @return exit value of this process.
     */
    public synchronized int waitFor() throws InterruptedException {
	if (_exited) {
	    return _exitValue;
	}
	wait();
	Log.assert(_exited, "wait returned but the Process didn't exit");
	return _exitValue;
    }

    /**
     * Get the exit value of this process.
     * 
     * @exception IllegalThreadStateException if this process has not
     *            exited yet.
     * 
     * @return exit value of this process.
     */
    public int exitValue() throws IllegalThreadStateException {
	if (!_exited) {
	    throw new IllegalThreadStateException();
	}
	return _exitValue;
    }
    
    /**
     * Kill this process.
     */
    public void destroy() {
	Packet packet = new Packet(PrivBrokerProxy.PRIVILEGE,
				   PrivBrokerProxy.DESTROY);
	packet.setLong(PrivBrokerProxy.JOB_ID, _jobId);
	_conn.sendPacket(packet);
    }

    /**
     * Add some data to this process's standard output.
     * PrivBrokerProxy calls this when it receives output data from
     * the server.
     * 
     * @param data bytes to add to standard output.
     */
    void addOutput(byte[] data) {
	_input.addBytes(data);
    }

    /**
     * Add some data to this process's standard error.
     * PrivBrokerProxy calls this when it receives error data from
     * the server.
     * 
     * @param data bytes to add to standard error.
     */
    void addError(byte[] data) {
	_error.addBytes(data);
    }

    /**
     * Called by PrivBrokerProxy when the corresponding process on the
     * server has completed.
     * 
     * @param exitValue exit value of the process on the server.
     */
    synchronized void finished(long exitValue) {
	_exited = true;
	_exitValue = (int)exitValue;
	_input.endOfFile();
	_error.endOfFile();
	notifyAll();
    }
}
