//
// RemoteInputStream.java
//
//	Input stream from a remote object.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.util;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;
import java.io.*;

/**
 * RemoteInputStream is an input stream that gets its data from a
 * remote source.  Some other entity (like PrivBrokerProxy) receives
 * data from the server, which gets added to the stream via its
 * addBytes() method.
 */
class RemoteInputStream extends InputStream {
    private byte[] _data = null;
    private int _mark = 0;
    boolean _endOfFile = false;

    /**
     * Construct a RemoteInputStream.
     */
    RemoteInputStream() {
    }

    /**
     * Read a byte of data.
     * 
     * @exception IOException if an I/O error occurs.
     * 
     * @return the byte read.
     */
    public synchronized int read() throws IOException {
	if (!waitForData()) {
	    return -1;
	}
	int c = _data[_mark++];
	if (_mark == _data.length) {
	    _data = null;
	    _mark = 0;
	}

	return c;
    }

    /**
     * read up to <tt>len</tt> bytes into <tt>buf</tt>.
     * 
     * @param buf buffer to read data into.
     * @param off offset to start copying bytes.
     * @param len Maximum number of bytes to copy.
     * 
     * @exception IOException if an I/O exception occurs.
     * 
     * @return number of bytes read.
     */
    public synchronized int read(byte buf[], int off, int len)
 	throws IOException {
 	if (!waitForData()) {
 	    return -1;
 	}
 	int numRead = 0;
 	while (len-- > 0 && _data != null) {
 	    numRead++;
 	    buf[off++] = _data[_mark++];
 	    if (_mark == _data.length) {
 		_data = null;
 		_mark = 0;
 	    }
 	}
 	return numRead;
    }

    /**
     * Determine how many bytes can be read without blocking.
     * 
     * @exception IOException if an I/O error occurs.
     * 
     * @return The number of bytes which can be read without
     *         blocking.
     */
    public synchronized int available() throws IOException {
	return _data == null ? 0 : _data.length - _mark;
    }

    /**
     * Add some bytes to this stream so that readers can subsequently
     * read it.
     * 
     * @param data bytes to add to this stream.
     */
    synchronized void addBytes(byte[] data) {
	int oldSize = _data != null ? _data.length - _mark : 0;
	int newSize = oldSize + data.length;
	byte[] newData = new byte[newSize];
	if (oldSize > 0) {
	    System.arraycopy(_data, _mark, newData, 0, _data.length - _mark);
	}
	System.arraycopy(data, 0, newData, oldSize, data.length);
	_data = newData;
	_mark = 0;
	notifyAll();
    }

    /**
     * No more bytes will be added to this stream.
     */
    synchronized void endOfFile() {
	_endOfFile = true;
	notifyAll();
    }

    /**
     * Wait for some data to be available.
     * 
     * @exception IOException if our wait is interrupted
     * 
     * @return true if data is available, false otherwise.
     */
    private boolean waitForData() throws IOException {
	// Block until some data arrives.
	while (_data == null) {
	    if (_endOfFile) {
		return false;
	    }
	    try {
		wait();
	    } catch (InterruptedException ex) {
		throw new IOException("interrupted read");
	    }
	}
	Log.assert(_mark < _data.length,
		   "we ran past the end of our buffer");
	return true;
    }
}
